/** @file libtrack/AxisAlignedBoundingBox.cpp
 *  @brief Implement the Track::AxisAlignedBoundingBox class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "AxisAlignedBoundingBox.h"

#include <limits>

namespace Track{

const btScalar max_scalar = std::numeric_limits<btScalar>::max();

AxisAlignedBoundingBox::AxisAlignedBoundingBox()
    :   min(max_scalar, max_scalar, max_scalar)
    ,   max(-max_scalar, -max_scalar, -max_scalar)
{
    
}


AxisAlignedBoundingBox::~AxisAlignedBoundingBox()
{
    
}

void AxisAlignedBoundingBox::operator|=(const btVector3 & point)
{
    min.setMin(point);
    max.setMax(point);
}

void AxisAlignedBoundingBox::operator|=(const AxisAlignedBoundingBox & other)
{
    min.setMin(other.min);
    max.setMax(other.max);
}

void AxisAlignedBoundingBox::add_border(btScalar size)
{
    btVector3 pad(size, size, size);
    min -= pad;
    max += pad;
}

bool AxisAlignedBoundingBox::valid() const
{
    return (    min.x() <= max.x()
             && min.y() <= max.y()
             && min.z() <= max.z()
            );
}

btScalar AxisAlignedBoundingBox::get_max_x() const
{
    return max.x();
}

btScalar AxisAlignedBoundingBox::get_max_y() const
{
    return max.y();
}

btScalar AxisAlignedBoundingBox::get_max_z() const
{
    return max.z();
}

btScalar AxisAlignedBoundingBox::get_min_x() const
{
    return min.x();
}

btScalar AxisAlignedBoundingBox::get_min_y() const
{
    return min.y();
}

btScalar AxisAlignedBoundingBox::get_min_z() const
{
    return min.z();
}

btVector3 AxisAlignedBoundingBox::get_min() const
{
    return min;
}

btVector3 AxisAlignedBoundingBox::get_max() const
{
    return max;
}

btVector3 AxisAlignedBoundingBox::get_extent() const
{
    return max - min;
}

btScalar AxisAlignedBoundingBox::get_volume() const
{
    btVector3 extent = get_extent();
    return extent.x() * extent.y() * extent.z();
}

AxisAlignedBoundingBox AxisAlignedBoundingBox::transform(btTransform transform) const
{
    // find the AABB of the 8 transoformed corners of the original AABB.
    AxisAlignedBoundingBox result;
    for (unsigned int i = 0; i < 2; i++)
    {
        for (unsigned int j = 0; j < 2; j++)
        {
            for (unsigned int k = 0; k < 2; k++)
            {
                result |= transform(btVector3(i ? min.x() : max.x(),
                                              j ? min.y() : max.y(),
                                              k ? min.z() : max.z()));
            }
        }
    }
    return result;
}

}
