/** @file racer/ai_train.cpp
 * Train the neural network for the computer controlled (AI) players.
 * @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but without any warranty; without even the implied warranty of
    merchantability or fitness for a particular purpose.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "floatfann.h"
#include "racer/Engine/GameObjects/Car.h"

/** Main function for racer_ai_train program.
 * Trains AI based on training data. The training data can be produced by
 * recording how a human player (or a replay of one) reacts to different
 * combinations of sensor values. You must edit Engine/GameObjects/Car.cpp to
 * produce the training data.
 * A neural network is produced trained to fit this data. Training is
 * necessary when the AI sensors or car behaviour changes.
 * 
 * After running racer_ai_Train, you can further tweak the AI produced with a
 * genetic algortihm, by running "racer t track" where track is some track
 * filename.
 */
int main()
{
    const unsigned int num_input = Engine::GameObjects::Car::NUM_SENSORS;
    const unsigned int num_output = 3;
    const unsigned int num_layers = 4;
    struct fann *ann = fann_create_standard(num_layers,
            num_input, 8, 6, num_output);
    
    fann_set_activation_function_output(ann, FANN_SIGMOID_SYMMETRIC_STEPWISE);
    fann_set_activation_function_hidden(ann, FANN_SIGMOID_SYMMETRIC_STEPWISE);
    
    // This level of accuracy is not really going to happen, but we can hope.
    const float desired_error = 0.0001;
    const unsigned int max_iterations = 1000;
    const unsigned int iterations_between_reports = 100;
    fann_train_on_file(ann, "racer/ANN_training_data", max_iterations,
            iterations_between_reports, desired_error);
    
    fann_save(ann, "racer/data/ai.net");
    fann_print_connections(ann);
    fann_destroy(ann);
    return 0;
}

