/** @file Debug.h
 *  @brief define some useful macros for debugging. 
 * 
 *  Use @code DEBUG_MESSAGE(msg); @endcode to write comments to standard out in
 *  debug build.
 *  This is a no-op in a release build. The comments will have the filename
 *  and line number printed with them. It uses std::cout, so you can separate
 *  any stuff that will work with streams with @code << @endcode.
 * 
 *  Use @code PRINT_STUB_MESSAGE; @endcode to write a message saying "Not
 *  implemented yet" with the file and line number. A doxygen todo command would
 *  be good practice nearby too.
 * 
 *  Use @code assert(condition); @endcode to abort due to precondition
 *  violations and more. 
 *  These are no-ops in release build, so the test will only be executed in
 *  Debug build. Using them frequently and testing things with debug builds is
 *  good practice. In a release build any possible exceptions that the user may
 *  induce shouldn't be caught by these assertions, as no testing is done in 
 *  release builds. For messing with user input and files, make sure to test for
 *  errors properly.
 * 
 *  BUILD_TIME just combines __DATE__ and __TIME__.
 * 
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef DEBUG_H_
#define DEBUG_H_

#include <iostream>

// This defines assert
#include <cassert>

#ifdef NDEBUG
    // Don't generate any code in a release build.
    #define DEBUG_MESSAGE( text )
#else 
    #define DEBUG_MESSAGE( text ) \
        std::cout << "Debug message from " __FILE__ ":" << __LINE__ << ": "\
                  << text << std::endl
#endif

#define PRINT_STUB_MESSAGE \
    std::cout << "Not implemented yet: " __FILE__ ":" << __LINE__ << std::endl
#define BUILD_TIME __DATE__ " " __TIME__ 
#endif /*DEBUG_H_*/
