/*
 * Copyright (C) 2009 Christopho, Solarus - http://www.solarus-engine.org
 *
 * Solarus is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Solarus is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include "lowlevel/Music.h"
#include "lowlevel/SpcDecoder.h"
#include "lowlevel/ItDecoder.h"
#include "lowlevel/FileTools.h"
#include "lowlevel/Debug.h"
#include "lowlevel/StringConcat.h"
#include "lowlevel/physfsrwops.h"
#include "Configuration.h"

float Music::volume = 1.0;
Music *Music::current_music = NULL;
std::map<MusicId,Music> Music::all_musics;

const MusicId Music::none = "none";
const MusicId Music::unchanged = "same";
bool Music::initialized = false;

SDL_Thread *thread;


/**
 * @brief Empty constructor.
 */
Music::Music():
  mus(NULL), id(none) {

}

/**
 * @brief Creates a new music.
 * @param music_id id of the music (a file name)
 */
Music::Music(const MusicId &music_id):
  mus(NULL), id(music_id) {

  if (!is_initialized()) {
    return;
  }

  // compute the file name
  file_name = (std::string) "musics/" + music_id;

  // get the format
  size_t index = music_id.find_last_of(".");
  Debug::check_assertion(index != std::string::npos && index != music_id.size(),
    StringConcat() << "Invalid music file name: " << music_id);
}

/**
 * @brief Destroys the music.
 */
Music::~Music() {

  if (!is_initialized()) {
    return;
  }

  if (current_music == this) {
    stop();
  }
  if (mus != NULL)
    Mix_FreeMusic(mus);
}

/**
 * @brief Initializes the music system.
 */
void Music::initialize() {
  // get the music volume from the configuration file
  set_volume(Configuration::get_value("music_volume", 100));
  initialized=true;
}

/**
 * @brief Exits the music system.
 */
void Music::quit() {
  if (is_initialized()) {
    all_musics.clear();
  }
}

/**
 * @brief Returns whether the music system is initialized.
 * @return true if the music system is initilialized
 */
bool Music::is_initialized() {
  return initialized;
}

/**
 * @brief Returns the current volume of musis.
 * @return the volume (0 to 100)
 */
int Music::get_volume() {

  return (int) (volume * 100.0);
}

/**
 * @brief Sets the volume of musics.
 * @param volume the new volume (0 to 100)
 */
void Music::set_volume(int volume) {

  Debug::check_assertion(volume >= 0 && volume <= 100, StringConcat() << "Illegal volume for music:" << volume);

  Configuration::set_value("music_volume", volume);
  Music::volume = volume / 100.0;
  Mix_VolumeMusic(volume);
    if (volume==0 && current_music != NULL) {
      current_music->stop();
      current_music = NULL;
    }

/*  if (current_music != NULL) {
    alSourcef(current_music->source, AL_GAIN, Music::volume);
  }*/
}

/**
 * @brief Returns the music currently playing.
 * @return the current music, or NULL if no music is being played
 */
Music* Music::get_current_music() {
  return current_music;
}

/**
 * @brief Returns the id of the music currently playing.
 * @return the id of the current music, or "none" if no music is being played
 */
const MusicId& Music::get_current_music_id() {
  return current_music != NULL ? current_music->id : none;
}

/**
 * @brief Plays a music.
 *
 * If the music is different from the current one,
 * the current one is stopped.
 * The music specified can also be Music::none_id (then the current music is just stopped)
 * or even Music::unchanged_id (nothing is done in this case).
 *
 * @param music_id id of the music to play
 */
void Music::play(const MusicId &music_id) {

  if (music_id != unchanged && music_id != get_current_music_id()) {
    // the music is changed

    if (music_id == none && current_music != NULL) {

      current_music->stop();
      current_music = NULL;
    }
    else {

      // play another music
      if (current_music != NULL) {
	current_music->stop();
      }

      if (all_musics.count(music_id) == 0) {
	all_musics[music_id] = Music(music_id);
      }

      Music &music = all_musics[music_id];
      if (music.start()) {
	current_music = &music;
      }
      else {
	current_music = NULL;
      }
    }
  }
}

/**
 * @brief Updates the music system.
 *
 * When a music is playing, this function makes it update.
 */

void Music::update() {

  if (!is_initialized()) {
    return;
  }
}

/**
 * @brief Loads the file and starts playing this music.
 *
 * No other music should be playing.
 *
 * @return true if the music was loaded successfully
 */
bool Music::start() {

  if (!is_initialized()||Music::volume==0.0) {
    return false;
  }

  Debug::check_assertion(current_music == NULL, StringConcat() << "Cannot play music file '" << file_name << "': a music is already playing");

  bool success = false;

  if (mus == NULL)
    mus = Mix_LoadMUS_RW(PHYSFSRWOPS_openRead(file_name.c_str()));
  
  if (mus != NULL) {
    Mix_PlayMusic(mus, -1); 
    success = true;
  }

  current_music = this;
  return success;
}

/**
 * @brief Stops playing the music.
 */
void Music::stop() {

  if (!is_initialized()) {
    return;
  }

  if (this != current_music) {
    return;
  }

  Mix_HaltMusic();
  current_music = NULL;
}

/**
 * @brief Returns whether the music is paused.
 * @return true if the music is paused, false otherwise
 */
bool Music::is_paused() {

  if (!is_initialized()) {
    return false;
  }

  return false;
}

/**
 * @brief Pauses or resumes the music.
 * @param pause true to pause the music, false to resume it
 */
void Music::set_paused(bool pause) {

  if (!is_initialized()) {
    return;
  }

/*  if (pause) {
    alSourcePause(source);
  }
  else {
    alSourcePlay(source);
  }*/
}

