/*
 * Copyright (C) 2009 Christopho, Solarus - http://www.solarus-engine.org
 *
 * Solarus is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Solarus is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include <iostream> // std::cout
#include <cstring>  // memcpy
#include <cmath>
#include <sstream>
#include <vector>
#include "lowlevel/Sound.h"
#include "lowlevel/Music.h"
#include "lowlevel/FileTools.h"
#include "lowlevel/Debug.h"
#include "lowlevel/StringConcat.h"
#include "lowlevel/physfsrwops.h"
#include "Configuration.h"

#include <SDL/SDL.h>
#include <SDL/SDL_mixer.h>

bool Sound::initialized = false;
float Sound::volume = 1.0;
std::list<Sound*> Sound::current_sounds;
std::map<SoundId,Sound> Sound::all_sounds;

/**
 * @brief Empty constructor.
 */
Sound::Sound():
  snd(NULL), id("") {

}

/**
 * @brief Creates a new Ogg Vorbis sound.
 * @param sound_id id of the sound: name of a .ogg file in the sounds subdirectory,
 * without the extension (.ogg is added automatically)
 */
Sound::Sound(const SoundId &sound_id):
  snd(NULL), id(sound_id) {
}

/**
 * @brief Destroys the sound.
 */
Sound::~Sound() {

  if (is_initialized()) {

    // stop the sources where this buffer is attached
/*    std::list<ALuint>::iterator it;
    for (it = sources.begin(); it != sources.end(); it++) {
      ALuint source = (*it);
      alSourceStop(source);
      alSourcei(source, AL_BUFFER, 0);
      alDeleteSources(1, &source);
    }
    alDeleteBuffers(1, &buffer);*/
    current_sounds.remove(this);
  }
}

/**
 * @brief Initializes the audio (music and sound) system.
 *
 * This method should be called when the application starts.
 * If the argument -no-audio is provided, this function has no effect and
 * there will be no sound.
 *
 * @param argc command-line arguments number
 * @param argv command-line arguments
 */
void Sound::initialize(int argc, char **argv) {
 
  // check the -no-audio option
  bool disable = false;
  for (argv++; argc > 1 && !disable; argv++, argc--) {
    const std::string arg = *argv;
    disable = (arg.find("-no-audio") == 0);
  }
  if (disable) {
    return;
  }

  SDL_InitSubSystem(SDL_INIT_AUDIO);
  Mix_OpenAudio(22050, AUDIO_S16SYS, 2, 4096);

  initialized = true;

  // initialize the music system
  Music::initialize();
}

/**
 * @brief Closes the audio (music and sound) system.
 *
 * This method should be called when exiting the application.
 */
void Sound::quit() {

  if (is_initialized()) {

    // uninitialize the music subsystem
    Music::quit();

    // clear the sounds
    all_sounds.clear();

    // uninitialize SDL_Mixer

    Mix_CloseAudio();

    initialized = false;
  }
}

/**
 * @brief Returns whether the audio (music and sound) system is initialized.
 * @return true if the audio (music and sound) system is initilialized
 */
bool Sound::is_initialized() {
  return initialized;
}

/**
 * @brief Loads and decodes all sounds listed in the game database.
 */
void Sound::load_all() {

  // open the resource database file
  const std::string file_name = "project_db.dat";
  std::istream& database_file = FileTools::data_file_open(file_name);
  std::string line;

  // read each animation
  while (std::getline(database_file, line)) {

    if (line.size() == 0) {
      continue;
    }

    int resource_type;
    std::string resource_id, resource_name;
    std::istringstream iss(line);
    FileTools::read(iss, resource_type);
    FileTools::read(iss, resource_id);
    FileTools::read(iss, resource_name);

    if (resource_type == 4) { // it's a sound

      if (all_sounds.count(resource_id) == 0) {
        all_sounds[resource_id] = Sound(resource_id);
        all_sounds[resource_id].load();
      }
    }
  }
  FileTools::data_file_close(database_file);
}

/**
 * @brief Returns whether a sound exists.
 * @param sound_id id of the sound to test
 */
bool Sound::exists(const SoundId& sound_id) {

  std::ostringstream oss;
  oss << "sounds/" << sound_id << ".ogg";
  return FileTools::data_file_exists(oss.str());
}

/**
 * @brief Starts playing the specified sound.
 * @param sound_id id of the sound to play
 */
void Sound::play(const SoundId& sound_id) {

  if (all_sounds.count(sound_id) == 0) {
    all_sounds[sound_id] = Sound(sound_id);
  }

  all_sounds[sound_id].start();
}

/**
 * @brief Returns the current volume of sound effects.
 * @return the volume (0 to 100)
 */
int Sound::get_volume() {

  return (int) (volume * 100.0);
}

/**
 * @brief Sets the volume of sound effects.
 * @param volume the new volume (0 to 100)
 */
void Sound::set_volume(int volume) {

  Debug::check_assertion(volume >= 0 && volume <= 100, StringConcat() << "Illegal volume for sound effects:" << volume);

  Configuration::set_value("sound_volume", volume);
  Sound::volume = volume / 100.0;
  Mix_Volume(-1, volume);
}

/**
 * @brief Updates the audio (music and sound) system.
 *
 * This function is called repeatedly by the game.
 */
void Sound::update() {
  // also update the music
  Music::update();
}

/**
 * @brief Loads and decodes the sound into memory.
 */
void Sound::load() {

  std::string file_name = (std::string) "sounds/" + id;
  if (id.find(".") == std::string::npos) {
    file_name += ".ogg";
  }
  snd = Mix_LoadWAV_RW(PHYSFSRWOPS_openRead(file_name.c_str()),1);
  if(!snd) {
    printf("Mix_LoadWAV_RW: %s\n", Mix_GetError());
  }
}

/**
 * @brief Plays the sound.
 * @return true if the sound was loaded successfully, false otherwise
 */
bool Sound::start() {

  bool success = false;

  if (is_initialized()) {
    if (snd == NULL)
      load();

    if (snd != NULL) {
      Mix_PlayChannel(-1, snd, 0);
      success = true;
    }
  }
  return success;
}
