/*
 * bool has_entity(MapEntity *entity);
 * Copyright (C) 2009 Christopho, Solarus - http://www.solarus-engine.org
 * 
 * Solarus is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Solarus is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include "entities/MapEntities.h"
#include "entities/Hero.h"
#include "entities/Tile.h"
#include "entities/TilePattern.h"
#include "entities/Layer.h"
#include "entities/Obstacle.h"
#include "entities/CrystalSwitchBlock.h"
#include "entities/Boomerang.h"
#include "Map.h"
#include "Game.h"
#include "lowlevel/Music.h"
#include "lowlevel/Debug.h"
#include "lowlevel/StringConcat.h"
using std::list;

//#include <iostream>
//#define MAPENT_PERF_DISP

/**
 * @brief Constructor.
 * @param game the game
 * @param map the map
 */
MapEntities::MapEntities(Game &game, Map &map):
  game(game),
  map(map),
  hero(game.get_hero()),
  music_before_miniboss(Music::none) {

  Layer layer = hero.get_layer();
  this->obstacle_entities[layer].push_back(&hero);
  this->entities_displayed_y_order[layer].push_back(&hero);
  // TODO update that when the layer changes, same thing for enemies
}

/**
 * @brief Destructor.
 */
MapEntities::~MapEntities() {
  destroy_all_entities();
}

/**
 * @brief Removes all entities from the map.
 *
 * This function is called by the destructor or when the map is unloaded.
 */
void MapEntities::destroy_all_entities() {

  // delete the entities sorted by layer
  for (int layer = 0; layer < LAYER_NB; layer++) {

    for (unsigned int i = 0; i < tiles[layer].size(); i++) {
      delete tiles[layer][i];
    }

    tiles[layer].clear();
    delete[] obstacle_tiles[layer];

    entities_displayed_first[layer].clear();
    entities_displayed_y_order[layer].clear();
    obstacle_entities[layer].clear();
    stairs[layer].clear();
  }

  // delete the other entities

  list<MapEntity*>::iterator i;
  for (i = all_entities.begin(); i != all_entities.end(); i++) {
    delete *i;
  }
  all_entities.clear();

  detectors.clear();
  entities_to_remove.clear();
}

/**
 * @brief Returns the hero.
 * @return the hero
 */
Hero& MapEntities::get_hero() {
  return hero;
}

/**
 * @brief Returns the entities (other that tiles) such that the hero cannot walk on them.
 * @param layer the layer
 * @return the obstacle entities on that layer
 */
list<MapEntity*>& MapEntities::get_obstacle_entities(Layer layer) {
  return obstacle_entities[layer];
}

/**
 * @brief Returns all detectors on the map.
 * @return the detectors
 */
list<Detector*>& MapEntities::get_detectors() {
  return detectors;
}

/**
 * @brief Returns all stairs on the specified layer.
 * @param layer the layer
 * @return the stairs on this layer
 */
list<Stairs*>& MapEntities::get_stairs(Layer layer) {
  return stairs[layer];
}

/**
 * @brief Returns all crystal switch blocks on the specified layer.
 * @param layer the layer
 * @return the crystal switch blocks on this layer
 */
list<CrystalSwitchBlock*>& MapEntities::get_crystal_switch_blocks(Layer layer) {
  return crystal_switch_blocks[layer];
}

/**
 * @brief Sets the obstacle tile property of an 8*8 square of the map.
 * @param layer layer of the square
 * @param x8 x coordinate of the square (divided by 8)
 * @param y8 y coordinate of the square (divided by 8)
 * @param obstacle the obstacle property to set
 */
void MapEntities::set_obstacle(int layer, int x8, int y8, Obstacle obstacle) {

  if (x8 >= 0 && x8 < map_width8 && y8 >= 0 && y8 < map_height8) {
    int index = y8 * map_width8 + x8;
    obstacle_tiles[layer][index] = obstacle;
  }
}

/**
 * @brief Returns the entity with the specified type and name.
 *
 * The program stops if there is no such entity.
 *
 * @param type type of entity
 * @param name name of the entity to get
 * @return the entity requested
 */
MapEntity* MapEntities::get_entity(EntityType type, const std::string &name) {

  MapEntity *entity = find_entity(type, name);

  Debug::check_assertion(entity != NULL, StringConcat() << "Cannot find entity with type '" << type << "' and name '" << name << "'");

  return entity;
}

/**
 * @brief Returns the entity with the specified type and name, or NULL if it doesn't exist.
 * @param type type of entity
 * @param name name of the entity to get
 * @return the entity requested, or NULL if there is no entity with the specified type and name
 */
MapEntity* MapEntities::find_entity(EntityType type, const std::string &name) {

  list<MapEntity*>::iterator i;
  for (i = all_entities.begin(); i != all_entities.end(); i++) {

    MapEntity *entity = *i;
    if (entity->get_type() == type && entity->get_name() == name && !entity->is_being_removed()) {
      return entity;
    }
  }

  return NULL;
}

/**
 * @brief Returns all entities of the map with the specified type.
 * @param type type of entity
 * @return the entities of this type
 */
list<MapEntity*> MapEntities::get_entities(EntityType type) {

  list<MapEntity*> entities;

  list<MapEntity*>::iterator i;
  for (i = all_entities.begin(); i != all_entities.end(); i++) {

    MapEntity *entity = *i;
    if (entity->get_type() == type && !entity->is_being_removed()) {
      entities.push_back(entity);
    }
  }

  return entities;
}

/**
 * @brief Returns the entities of the map with the specified type and having the specified name prefix.
 * @param type type of entity
 * @param prefix prefix of the name
 * @return the entities of this type and having this prefix in their name
 */
list<MapEntity*> MapEntities::get_entities_with_prefix(EntityType type, const std::string &prefix) {

  list<MapEntity*> entities;

  list<MapEntity*>::iterator i;
  for (i = all_entities.begin(); i != all_entities.end(); i++) {

    MapEntity *entity = *i;
    if (entity->get_type() == type && entity->has_prefix(prefix) && !entity->is_being_removed()) {
      entities.push_back(entity);
    }
  }

  return entities;
}

/**
 * @brief Brings to front an entity that is displayed as a sprite in the normal order.
 * @param entity the entity to bring to front
 */
void MapEntities::bring_to_front(MapEntity *entity) {

  Debug::check_assertion(entity->can_be_displayed(),
      StringConcat() << "Cannot bring to front entity '" << entity->get_name() << "' since it is not displayed");

  Debug::check_assertion(!entity->is_displayed_in_y_order(),
    StringConcat() << "Cannot bring to front entity '" << entity->get_name() << "' since it is displayed in the y order");

  Layer layer = entity->get_layer();
  entities_displayed_first[layer].remove(entity);
  entities_displayed_first[layer].push_back(entity);
}

/**
 * @brief Notifies all entities of the map that the map has just become active.
 */
void MapEntities::notify_map_started() {

  list<MapEntity*>::iterator i;
  for (i = all_entities.begin(); i != all_entities.end(); i++) {
    MapEntity *entity = *i;
    entity->notify_map_started();
  }
}

/**
 * @brief Adds a tile on the map.
 *
 * This function is called for each tile when loading the map.
 * The tiles cannot change during the game.
 *
 * @param tile the tile to add
 */
void MapEntities::add_tile(Tile *tile) {

  Layer layer = tile->get_layer();

  // add the tile to the map
  tiles[layer].push_back(tile);
  tile->set_map(map);

  // update the collision list
  Obstacle obstacle = tile->get_tile_pattern().get_obstacle();

  int tile_x8 = tile->get_x() / 8;
  int tile_y8 = tile->get_y() / 8;
  int tile_width8 = tile->get_width() / 8;
  int tile_height8 = tile->get_height() / 8;

  int i, j;
 
  for (i = 0; i < tile_height8; i++) {
    for (j = 0; j < tile_width8; j++) {
      set_obstacle(layer, tile_x8 + j, tile_y8 + i, OBSTACLE_NONE);
    }
  }

  switch (obstacle) {

    /* If the obstacle property is the same for all points inside the base tile,
     * then all 8*8 squares of the extended tile have the same property.
     */
  case OBSTACLE_NONE:
  case OBSTACLE_SHALLOW_WATER:
  case OBSTACLE_DEEP_WATER:
  case OBSTACLE_HOLE:
  case OBSTACLE_LAVA:
  case OBSTACLE_PRICKLE:
  case OBSTACLE_LADDER:
  case OBSTACLE:
    for (i = 0; i < tile_height8; i++) {
      for (j = 0; j < tile_width8; j++) {
	set_obstacle(layer, tile_x8 + j, tile_y8 + i, obstacle);
      }
    }
    break;

    /* If the top right corner of the tile is an obstacle,
     * then the top right 8*8 squares are OBSTACLE, the bottom left
     * 8*8 squares are NO_OBSTACLE and the 8*8 squares on the diagonal
     * are OBSTACLE_TOP_RIGHT.
     */
  case OBSTACLE_TOP_RIGHT:
    // we traverse each row of 8*8 squares on the tile
    for (i = 0; i < tile_height8; i++) {

      // 8*8 square on the diagonal
      set_obstacle(layer, tile_x8 + i, tile_y8 + i, OBSTACLE_TOP_RIGHT);

      // right part of the row: we are in the top-right corner
      for (j = i + 1; j < tile_width8; j++) {
	set_obstacle(layer, tile_x8 + j, tile_y8 + i, OBSTACLE);
      }
    }
    break;

  case OBSTACLE_TOP_LEFT:
    // we traverse each row of 8*8 squares on the tile
    for (i = 0; i < tile_height8; i++) {

      // left part of the row: we are in the top-left corner
      for (j = 0; j < tile_width8 - i - 1; j++) {
	set_obstacle(layer, tile_x8 + j, tile_y8 + i, OBSTACLE);
      }

      // 8*8 square on the diagonal
      set_obstacle(layer, tile_x8 + j, tile_y8 + i, OBSTACLE_TOP_LEFT);
    }
    break;

  case OBSTACLE_BOTTOM_LEFT:
    // we traverse each row of 8*8 squares on the tile
    for (i = 0; i < tile_height8; i++) {

      // left part of the row: we are in the bottom-left corner
      for (j = 0; j < i; j++) {
	set_obstacle(layer, tile_x8 + j, tile_y8 + i, OBSTACLE);
      }

      // 8*8 square on the diagonal
      set_obstacle(layer, tile_x8 + j, tile_y8 + i, OBSTACLE_BOTTOM_LEFT);
    }
    break;

  case OBSTACLE_BOTTOM_RIGHT:
    // we traverse each row of 8*8 squares on the tile
    for (i = 0; i < tile_height8; i++) {

      // 8*8 square on the diagonal
      set_obstacle(layer, tile_x8 + tile_width8 - i - 1, tile_y8 + i, OBSTACLE_BOTTOM_RIGHT);

      // right part of the row: we are in the bottom-right corner
      for (j = tile_width8 - i; j < tile_width8; j++) {
	set_obstacle(layer, tile_x8 + j, tile_y8 + i, OBSTACLE);
      }
    }
    break;

  case OBSTACLE_EMPTY:
    Debug::die("Illegal obstacle property for this tile");
    break;
  }
}

/**
 * @brief Adds an entity to the map.
 *
 * This function is called when loading the map. If the entity
 * specified is NULL (because some entity creation functions
 * may return NULL), nothing is done.
 *
 * @param entity the entity to add (can be NULL)
 */
void MapEntities::add_entity(MapEntity *entity) {

  if (entity == NULL) {
    return;
  }

  if (entity->get_type() == TILE) {
    add_tile((Tile*) entity);
  }
  else {
    Layer layer = entity->get_layer();

    // update the detectors list
    if (entity->can_detect_entities()) {
      detectors.push_back((Detector*) entity);
    }

    // update the obstacle list
    if (entity->can_be_obstacle()) {

      if (entity->has_layer_independent_collisions()) {
	// some entities handle collisions on any layer (e.g. stairs inside a single floor)
        obstacle_entities[LAYER_LOW].push_back(entity);
        obstacle_entities[LAYER_INTERMEDIATE].push_back(entity);
        obstacle_entities[LAYER_HIGH].push_back(entity);
      }
      else {
	// but usually, an entity collides with only one layer
        obstacle_entities[layer].push_back(entity);
      }
    }

    // update the sprites list
    if (entity->is_displayed_in_y_order()) {
      entities_displayed_y_order[layer].push_back(entity);
    }
    else if (entity->can_be_displayed()) {
      entities_displayed_first[layer].push_back(entity);
    }

    // update the specific entities lists
    switch (entity->get_type()) {

      case STAIRS:
	stairs[layer].push_back((Stairs*) entity);
	break;

      case CRYSTAL_SWITCH_BLOCK:
	crystal_switch_blocks[layer].push_back((CrystalSwitchBlock*) entity);
	break;

      case BOOMERANG:
	this->boomerang = (Boomerang*) entity;
	break;

      default:
      break;
    }

    // update the list of all entities
    all_entities.push_back(entity);
  }

  // notify the entity
  entity->set_map(map);
}

/**
 * @brief Removes an entity from the map and schedules it to be destroyed.
 * @param entity the entity to remove
 */
void MapEntities::remove_entity(MapEntity *entity) {

  entities_to_remove.push_back(entity);
  entity->notify_being_removed();

  if (entity == (MapEntity*) this->boomerang) {
    this->boomerang = NULL;
  }
}

/**
 * @brief Removes an entity from the map and schedules it to be destroyed.
 * @param type type of the entity to remove
 * @param name name of the entity
 */
void MapEntities::remove_entity(EntityType type, const std::string &name) {
  remove_entity(get_entity(type, name));
}

/**
 * @brief Removes all entities of a type whose name starts with the specified prefix.
 * @param type a type of entities
 * @param prefix prefix of the name of the entities to remove
 */
void MapEntities::remove_entities_with_prefix(EntityType type, const std::string& prefix) {

  std::list<MapEntity*> entities = get_entities_with_prefix(type, prefix);
  std::list<MapEntity*>::iterator it;
  for (it = entities.begin(); it != entities.end(); it++) {
    remove_entity(*it);
  }
}

/**
 * @brief Removes and destroys the entities placed in the entities_to_remove list. 
 */
void MapEntities::remove_marked_entities() {

  list<MapEntity*>::iterator it;

  // remove the marked entities
  for (it = entities_to_remove.begin();
       it != entities_to_remove.end();
       it++) {

    MapEntity *entity = *it;
    Layer layer = entity->get_layer();

    // remove it from the obstacle entities list if present
    if (entity->can_be_obstacle()) {

      if (entity->has_layer_independent_collisions()) {
	for (int i = 0; i < LAYER_NB; i++) {
	  obstacle_entities[i].remove(entity);
	}
      }
      else {
        obstacle_entities[layer].remove(entity);
      }
    }

    // remove it from the detectors list if present
    if (entity->can_detect_entities()) {
      detectors.remove((Detector*) entity);
    }

    // remove it from the sprite entities list if present
    if (entity->is_displayed_in_y_order()) {
      entities_displayed_y_order[layer].remove(entity);
    }
    else if (entity->can_be_displayed()) {
      entities_displayed_first[layer].remove(entity);
    }

    // remove it from the whole list
    all_entities.remove(entity);

    // destroy it
    delete entity;
  }
  entities_to_remove.clear();
}

/**
 * @brief Suspends or resumes the movement and animations of the entities.
 *
 * This function is called by the map when the game
 * is being suspended or resumed.
 *
 * @param suspended true to suspend the movement and the animations,
 * false to resume them
 */
void MapEntities::set_suspended(bool suspended) {

  // the hero first
  hero.set_suspended(suspended);

  // other entities
  list<MapEntity*>::iterator i;
  for (i = all_entities.begin();
       i != all_entities.end();
       i++) {

    (*i)->set_suspended(suspended);
  }

  // note that we don't suspend the animated tiles
}

/**
 * @brief Updates the position, movement and animation each entity.
 */
void MapEntities::update() {

  // first update the hero
  hero.update();

  // update the tiles and the dynamic entities
  list<MapEntity*>::iterator it;
  for (int layer = 0; layer < LAYER_NB; layer++) {

    for (unsigned int i = 0; i < tiles[layer].size(); i++) {
      tiles[layer][i]->update();
    }

    // sort the entities displayed in y order
    entities_displayed_y_order[layer].sort(compare_y);
  }

  for (it = all_entities.begin();
       it != all_entities.end();
       it++) {

    if (!(*it)->is_being_removed()) {
      (*it)->update();
    }
  }

  // remove the entities that have to be removed now
  remove_marked_entities();
}

/**
 * @brief Displays the entities on the map surface.
 */
void MapEntities::display() {

  for (int layer = 0; layer < LAYER_NB; layer++) {

    // put the tiles
#ifdef MAPENT_PERF_DISP
   std::cout << " [" << layer << "] = " << tiles[layer].size();
#endif
    for (unsigned int i = 0; i < tiles[layer].size(); i++) {
      tiles[layer][i]->display_on_map();
    }

    // put the first sprites
    list<MapEntity*>::iterator i;
    for (i = entities_displayed_first[layer].begin();
	 i != entities_displayed_first[layer].end();
	 i++) {

      MapEntity *entity = *i;
      /*
      Debug::check_assertion(entity->get_layer() == layer,
	  StringConcat() << "Trying to display entity " << entity << " on layer "
	  << layer << " but it is actually on layer " << entity->get_layer());
      */
      if (entity->is_enabled()) {
        entity->display_on_map();
      }
    }

    // put the sprites displayed at the hero's level, in the order
    // defined by their y position (including the hero)
    for (i = entities_displayed_y_order[layer].begin();
	 i != entities_displayed_y_order[layer].end();
	 i++) {

      MapEntity *entity = *i;
      if (entity->is_enabled()) {
        entity->display_on_map();
      }
    }
  }
#ifdef MAPENT_PERF_DISP
   std::cout << std::endl;
#endif
}

/**
 * @brief Compares the y position of two entities.
 * @param first an entity
 * @param second another entity
 * @return true if the y position of the first entity is lower
 * than the second one
 */
bool MapEntities::compare_y(MapEntity *first, MapEntity *second) {

  // before was: first->get_top_left_y() < second->get_top_left_y(); but doesn't work for bosses
  return first->get_top_left_y() + first->get_height() < second->get_top_left_y() + second->get_height();
}

/**
 * @brief Changes the layer of an entity.
 *
 * Only some specific entities should change their layer.
 *
 * @param entity an entity
 * @param layer the new layer
 */
void MapEntities::set_entity_layer(MapEntity *entity, Layer layer) {

  Layer old_layer = entity->get_layer();

  if (layer != old_layer) {

    entity->set_layer(layer);

    // update the obstacle list
    if (entity->can_be_obstacle() && !entity->has_layer_independent_collisions()) {
      obstacle_entities[old_layer].remove(entity);
      obstacle_entities[layer].push_back(entity);
    }

    // update the sprites list
    if (entity->is_displayed_in_y_order()) {
      entities_displayed_y_order[old_layer].remove(entity);
      entities_displayed_y_order[layer].push_back(entity);
    }
    else if (entity->can_be_displayed()) {
      entities_displayed_first[old_layer].remove(entity);
      entities_displayed_first[layer].push_back(entity);
    }
  }
}

/**
 * @brief Returns whether a rectangle overlaps with a raised crystal switch block.
 * @param layer the layer to check
 * @param rectangle a rectangle
 * @return true if this rectangle overlaps a raised crystal switch block
 */
bool MapEntities::overlaps_raised_blocks(Layer layer, const Rectangle &rectangle) {

  bool overlaps = false;
  std::list<CrystalSwitchBlock*> blocks = get_crystal_switch_blocks(layer);

  std::list<CrystalSwitchBlock*>::iterator it;
  for (it = blocks.begin(); it != blocks.end() && !overlaps; it++) {
    overlaps = (*it)->overlaps(rectangle) && (*it)->is_raised();
  }

  return overlaps;
}

/**
 * @brief Returns true if the player has thrown the boomerang.
 * @return true if the boomerang is present on the map
 */
bool MapEntities::is_boomerang_present() {
  return boomerang != NULL;
}

/**
 * @brief Removes the boomerang from the map, if it is present.
 */
void MapEntities::remove_boomerang() {

  if (boomerang != NULL) {
    remove_entity(boomerang);
    boomerang = NULL;
  }
}

