/* 
    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
extern "C" {
#include "wavpack/wavpack.h"
}
#include "../plugin.h"
#include "../util.h"

static struct sound_plugin plugin;
static WavpackContext *wpc;
static FILE* fp = NULL;
static char wperror [80];
static int playing;
static int bps;
static int32_t *tmpbuf;
static int writepos;
static int readpos;
static int total_samples;
static string fieldname[5];
static string fielddata[5];

static int close();

static int32_t read_bytes (void *buff, int32_t bcount)
{
    return fread(buff, 1, bcount, fp);
}

static int init_file(char *fname)
{
    playing = 0;
#ifdef A320
    plugin.clockfreq = 336;
#else
    plugin.clockfreq = 150;
#endif
    
    if(!(fp = fopen(fname, "rb")))
    {
        fprintf(stderr, "wavpackplugin:init_file - Couldn't open file: %s\n",fname);
        return -1;
    }
    
    wpc = WavpackOpenFileInput(read_bytes, wperror);

    plugin.channels = WavpackGetReducedChannels(wpc);
    plugin.freq     = WavpackGetSampleRate(wpc);
    total_samples   = WavpackGetNumSamples(wpc);
    plugin.length   = (int)((1000.0*total_samples)/plugin.freq);
    bps = WavpackGetBytesPerSample(wpc);

    if (!wpc) {
        fprintf(stderr, "wavpackplugin:init_file - Error: %s\n",wperror);
        close();
        return -1;
    }

    if (!(tmpbuf = (int*)malloc(2048*sizeof(int)*plugin.channels)))
    {
        fprintf(stderr, "wavpackplugin:init_file - Couldn't allocate memory\n");
        close();
        return -1;
    }

    int x = 0;
    fieldname[x] = "File";
    fielddata[x] = strrchr(fname,'/')+1;
    x++;
    fieldname[x] = "Format";
    fielddata[x] = "Wavpack";
    x++;
    plugin.nfields = x;
    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;

    writepos        = 0;
    readpos         = 0;
    playing         = 1;
    printf("bps = %d, total_samples = %d\n",bps,total_samples);
    return 0;
}

static int close()
{
    if (fp)
    {
        fclose(fp);
        fp = NULL;
    }
    if (tmpbuf)
    {
        free(tmpbuf);
        tmpbuf = NULL;
    }
    for(int i = 0; i < 5; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }
    playing = plugin.length = plugin.nfields = 0;
    return 0;
}

static void bufcpy(signed short *dest, int32_t *src, int samples)
{
    if (bps == 1)
        for (int i = 0; i < samples; i++)
            dest[i] = src[i]<<8;
    else if (bps == 2)
        for (int i = 0; i < samples; i++)
            dest[i] = src[i];
    else if (bps == 3)
        for (int i = 0; i < samples; i++)
            dest[i] = src[i]>>8;
    else if (bps == 4)
        for (int i = 0; i < samples; i++)
            dest[i] = src[i]>>16;
}

static int fill_buffer(signed short *dest, int len)
{
    if(playing)
    {
        int written = 0;
        int towrite = len>>1;
        for (;;)
        {
            int left = towrite - written;
            int buf  = writepos - readpos;
            //printf("towrite = %04d - written = %04d = %04d, " \
                    "read = %04d - write = %04d = %04d\n",\
                towrite,written,left,readpos,writepos,buf);
            if (left <= buf)
            {
                //memcpy((char*)dest + writtenbytes, (char*)tmpbuf + readposbytes, bytesleft);
                bufcpy(&dest[written], &tmpbuf[readpos], left);
                readpos += left;
                written = towrite;
                break;
            }
            else
            {
                //memcpy((char*)dest + writtenbytes, (char*)tmpbuf + readposbytes, bufbytes);
                bufcpy(&dest[written], &tmpbuf[readpos], buf);
                written += buf;
                readpos = 0;
                writepos = WavpackUnpackSamples(wpc, tmpbuf, 2048)*plugin.channels;
                if (!writepos)
                    break;
            }
        }
        //samples_done += written/(2*plugin.channels);
        //printf("written = %d\n",written<<1);
        return written<<1;
    }
    return 0;
}

static int can_handle(const char *name)
{
    return (is_ext(name, ".wv")); 
}

/*static int set_position(int msecs, int subtune)
{
    if (playing)
    {
        int skip_samples = (int)(msecs*plugin.freq*msecs/1000);
        if ((int)samples_done + skip_samples < 0)
            skip_samples = -samples_done;
        samples_done += skip_samples;
        seek_absolute(samples_done);
        readposbytes = 0;

        return (int)(1000.0*skip_samples/plugin.freq);
    }
    return 0;
}*/

extern "C" {

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN wavpack_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
    memset(&plugin, 0, sizeof(plugin));
    plugin.plugname = "Wavpack";
    //plugin.init_data = init_data;
    plugin.init_file = init_file;
    //plugin.request_format = NULL;
    //plugin.set_position = set_position;
    plugin.fill_buffer = fill_buffer;
    plugin.can_handle = can_handle;
    plugin.close = close;
    plugin.tune = 0;
    plugin.subtunes = 1;
    plugin.replaygain = 1;
    return &plugin;
}

}

