/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <string>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <tremor/ivorbisfile.h>
#include "../plugin.h" 
#include "../util.h"

static struct sound_plugin plugin;
static int playing;
static int current_section;
static string fieldname[5];
static string fielddata[5];
static OggVorbis_File vf;
static vorbis_info *vi;
static FILE *fp;

#define TREMORDEBUG

static int init_file(char *fname)
{
    playing = 0;
    current_section = 0;
#ifdef A320
    plugin.clockfreq = 336;
#else
    plugin.clockfreq = 150; 
#endif

#ifdef TREMORDEBUG
    fprintf(stderr, "vorbisplugin::init_file - Opening file\n");
#endif
    fp = fopen(fname, "rb");
    if(ov_open(fp, &vf, NULL, 0) < 0)
    {
        fprintf(stderr, "vorbisplugin::init_file - Couldn't open file\n");
        fclose(fp);
        return -1;
    }
#ifdef TREMORDEBUG
    fprintf(stderr, "vorbisplugin::init_file - File ok!\n");
#endif
    vi = ov_info(&vf,-1);
    plugin.channels = vi->channels;
    plugin.freq = plugin.channels * vi->rate / 2;
    long samples = ov_pcm_total(&vf,-1);
    plugin.length = (int)((1000.0*samples) * plugin.channels 
                                / (plugin.freq * 2));
#ifdef TREMORDEBUG
    fprintf(stderr, "vorbisplugin::init_file - %d chn, %d rate, %d len\n",
                    plugin.channels, plugin.freq, plugin.length);
#endif

    vorbis_comment *vc = ov_comment(&vf,-1);
#ifdef TREMORDEBUG
    fprintf(stderr, "vorbisplugin::init_file - Comments found: %d\n",
                    vc->comments);
#endif
    // Comment parsing
    string tmpname   = "";
    string tmpartist = "";
    string tmpperf   = "";
    string tmpalbum  = "";
    string tmpgenre  = "";
    string tmptrack  = "";
    string tmpdesc   = "";
    string tmpcopy   = "";
    string tmpdate   = "";
    string tmpvers   = "";
    string tmploc    = "";
    for (int i = 0; i<vc->comments; i++)
    {
        char* comment = vc->user_comments[i];
             if (!strncasecmp(comment,"name=",5) && strlen(comment+5))
            tmpname   = comment+5;
        else if (!strncasecmp(comment,"title=",6) && strlen(comment+6))
            tmpname   = comment+6;
        else if (!strncasecmp(comment,"performer=",10) && strlen(comment+10))
            tmpperf   = comment+10;
        else if (!strncasecmp(comment,"artist=",7) && strlen(comment+7))
            tmpartist = (comment+7) + string(" ");
        else if (!strncasecmp(comment,"album=",6) && strlen(comment+6))
            tmpalbum  = comment+6;
        else if (!strncasecmp(comment,"genre=",6) && strlen(comment+6))
            tmpgenre  = comment+6;
        else if (!strncasecmp(comment,"track=",6) && strlen(comment+6))
            tmptrack  = comment+6;
        else if (!strncasecmp(comment,"tracknumber=",12) && strlen(comment+12))
            tmptrack  = comment+12;
        else if (!strncasecmp(comment,"description=",12) && strlen(comment+12))
            tmpdesc   = comment+12;
        else if (!strncasecmp(comment,"comment=",8) && strlen(comment+8))
            tmpdesc   = comment+8;
        else if (!strncasecmp(comment,"copyright=",10) && strlen(comment+10))
            tmpcopy   = comment+10;
        else if (!strncasecmp(comment,"date=",5) && strlen(comment+5))
            tmpdate   = comment+5;
        else if (!strncasecmp(comment,"version=",8) && strlen(comment+8))
            tmpvers   = comment+8;
        else if (!strncasecmp(comment,"location=",9) && strlen(comment+9))
            tmploc    = comment+9;
    }

    int x = 0; // I can fit four of these + format
    if (tmpname.length())
    {
        fieldname[x] = "Title";
        fielddata[x] = tmpname;
        if (tmpvers.length())
            fielddata[x] += " (" + tmpvers + ")";
    }
    else
    {
        fieldname[x] = "File";
        fielddata[x] = strrchr(fname,'/')+1;
    }
    x++;
    if (tmpalbum.length()) 
    {
        fieldname[x] = "Album";
        fielddata[x] = tmpalbum;
        x++;
    } // max 2 so far
    if (tmpartist.length())
    {
        fieldname[x] = "Artist";
        fielddata[x] = tmpartist;
        x++;
    } // max 3
    if (tmpperf.length())
    {
        fieldname[x] = "Performer";
        fieldname[x] = tmpperf;
        x++;
    } // max 4 (have to start checking)
    if (x < 4 && tmpgenre.length())
    {
        fieldname[x] = "Genre";
        fielddata[x] = tmpgenre;
        x++;
    }
    if (x < 4 && tmptrack.length())
    {
        fieldname[x] = "Track";
        fielddata[x] = tmptrack;
        x++;
    }
    else if (tmptrack.length() && tmpname.length())
        fielddata[0] = string(tmptrack) + " - " + fielddata[0];
    if (x < 4 && tmpdesc.length())
    {
        fieldname[x] = "Description";
        fielddata[x] = tmpdesc;
    }
    if (x < 4 && tmpcopy.length())
    {
        fieldname[x] = "Copyright";
        fielddata[x] = tmpcopy;
        x++;
    }
    if (x < 4 && tmpdate.length())
    {
        fieldname[x] = "Date";
        fielddata[x] = tmpdate;
        x++;
    }
    if (x < 4 && tmploc.length())
    {
        fieldname[x] = "Location";
        fielddata[x] = tmploc;
        x++;
    }
    fieldname[x] = "Format";
    fielddata[x] = "OGG";
    x++;
    plugin.nfields = x;
    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;
    playing = 1;
    return 0;
}

static int close()
{
    ov_clear(&vf);
    //fclose(fp); // Already done in ov_clear. Not the way it should work i think
    for(int i = 0; i < 5; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }
    playing = plugin.length = plugin.nfields = 0;
    current_section = 0;
    return 0;
}

static int fill_buffer(signed short *dest, int len)
{
    if(playing)
    {
        long read = 0;
        for (int left = len; left > 0; left = len - read)
        {
            long read2 = ov_read(&vf,(char*)dest+read, left, &current_section);
            if (read2 > 0)
                read += read2;
            else if (read2 < 0)
                printf("vorbisplugin:fill_buffer - Error %d\n",read2);
            else
            {
                playing = 0;
                plugin.length = 0;
                return read;
            }
        }
        return read;
    }
    return 0;
}

static int set_position(int msecs, int subtune)
{
    if (!playing)
        return 0;
    if (msecs)
    {
        int ogg_time = (int)ov_time_tell(&vf);
        int seek_time = msecs;
        if (seek_time < 0 && ogg_time < -seek_time)
            seek_time = 0;
        else
            seek_time = ogg_time + seek_time;
        if(ov_time_seek(&vf,seek_time))
            return 0;
        return (seek_time - ogg_time);
    }
    return 0;
}

static int can_handle(const char *name)
{
    return (is_ext(name, ".ogg"));
}

extern "C" {

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN vorbis_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
    memset(&plugin, 0, sizeof(plugin));
    plugin.plugname = "Vorbis";
    //plugin.init_data = init_data;
    plugin.init_file = init_file;
    plugin.close = close;
    //plugin.request_format = NULL;
    plugin.set_position = set_position;
    plugin.fill_buffer = fill_buffer;
    plugin.can_handle = can_handle;
    plugin.tune = 0;
    plugin.subtunes = 1;
    plugin.replaygain = 1;
    return &plugin;
}

}

