/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

extern "C" {
#include "vgmstream/vgmstream.h"
}

#include "../plugin.h"
#include "../util.h"

#include "../FormatsDatabase.h"

static struct sound_plugin plugin;
static VGMSTREAM *s = NULL;
static int32_t total_samples = 0;
static int32_t current_sample = 0;

static FormatsDatabase formatsdb;

int ignore_loop = 0;
int force_loop = 0;
int loop_count = 1;

static int playing;
static string fieldname[3];
static string fielddata[3];
static string filename;
static int set_position(int msecs, int subtune);

static int init_file(char *fname)
{
    playing = 0;
    plugin.tune = 0;
    plugin.subtunes = 1;

    s = init_vgmstream(fname);
    if( !s ) {
        printf("Error: Cannot load %s file\n", fname);
        return -1;
    }
	
    // loop flag
    /* force only if there aren't already loop points */
    if (force_loop && !s->loop_flag) {
        /* this requires a bit more messing with the VGMSTREAM than I'm
			* comfortable with... */
        s->loop_flag = 1;
        s->loop_start_sample = 0;
        s->loop_end_sample = s->num_samples;
        s->loop_ch = (VGMSTREAMCHANNEL *)calloc(s->channels,sizeof(VGMSTREAMCHANNEL));
    }

    if (ignore_loop) s->loop_flag=0;
	
    plugin.channels = s->channels;
    plugin.freq = s->sample_rate;
    current_sample = 0;
    total_samples = get_vgmstream_play_samples((double)loop_count, 0, 0, s);
    plugin.length = (total_samples*1000) / s->sample_rate;
    if( plugin.length <= 0 ) plugin.length = 200000;

    char desc[512], encoding[128], layout[128], metadata[128];
    describe_vgmstream(s, desc, 512);

    int x = 0;
    char *tmp = strstr(desc, "encoding: ");
    if( tmp ) {
        strncpy(encoding, tmp+10, 128);
        *(strchr(encoding, '\n')) = '\0';
        fieldname[x] = "Encoding";
        fielddata[x++] = string(encoding);
    }

    tmp = strstr(desc, "layout: ");    
    if( tmp ) {
        strncpy(layout, tmp+8, 128);
        *(strchr(layout, '\n')) = '\0';
        
        fieldname[x] = "Layout";
        fielddata[x++] = string(layout);
    }

    tmp = strstr(desc, "metadata from: ");    
    if( tmp ) {
        strncpy(metadata, tmp+15, 128);
        fieldname[x] = "Metadata";
        fielddata[x++] = string(metadata);
    }

    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;
    plugin.nfields = x;

    playing = 1;
    return 0;
}

static int close()
{
    if( s ) {
        close_vgmstream(s);
        s = NULL;
    }

    for(int i = 0; i < plugin.nfields; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }

    playing = plugin.tune = plugin.subtunes = plugin.length = plugin.nfields = 0;
    return 0;
}

static int fill_buffer(signed short *dest, int len)
{
    if(playing)
    {
        // Audio write func
        // have we finished decoding?
        current_sample += len / (plugin.channels << 1);
        if( current_sample >= total_samples ) {
            playing = 0;
            return 0;
    	}
		
    	render_vgmstream((sample *)dest, len / (plugin.channels << 1), s);
    	return len;
    }

    return 0;
}

static int can_handle(const char *name)
{
    const char *sext = strrchr(name, '.');
    return formatsdb.Find(sext ? sext+1 : "");
}

extern "C" {

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN vgmstream_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
    // Load formats database
    if( !formatsdb.Load("[VGMSTREAM]", "formats.db") )
        exit(-1);

    memset(&plugin, 0, sizeof(plugin));
    plugin.plugname = "vgmstream r6xx";
    plugin.init_file        = init_file;
    //plugin.init_data      = init_data;
    //plugin.request_format = NULL;
    //plugin.set_position     = set_position;
    plugin.fill_buffer      = fill_buffer;
    plugin.can_handle       = can_handle;
    plugin.close            = close;
    plugin.channels   = 2;
    plugin.freq       = 44100;
    plugin.replaygain = 1;

#ifdef A320
    plugin.clockfreq = 384; 
#else
    plugin.clockfreq = 150; 
#endif

    return &plugin;
}

}

