/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <SDL/SDL.h>
#include <string.h>

#include "MusicPlayer.h"
#include "PlayerApp.h"
#include "fex/fex.h"
#include "id3tag.h"

#ifndef _WIN32
#define stricmp strcasecmp
#define strnicmp strncasecmp
#include <unistd.h>
#else
#include <direct.h>
#endif

using namespace std;

void error( fex_err_t err )
{
	if ( err != NULL )
	{
		const char* str = fex_err_str( err );
		fprintf( stderr, "Error: %s\n", str );
		exit( EXIT_FAILURE );
	}
}

int is_ext(const char *s, const char *ext)
{
	const char *sext = strrchr(s, '.');
    return sext && stricmp(ext, sext) == 0;
}

int is_fex(const char *s)
{
    static const char exts [] [6] = {
		".7z",
		".arc",
		".arj",
		".bz2",
		".cab",
		".dmg",
		".gz",
		".lha",
		".lz",
		".lzh",
		".lzma",
		".lzo",
		".lzx",
		".pea",
		".rar",
        ".rsn",
		".sit",
		".sitx",
		".tgz",
		".tlz",
		".z",
		".zip",
		".zoo",
		""
	};
	
	const char (*ext) [6] = exts;
	for ( ; **ext; ext++ )
	{
		if ( is_ext( s, *ext) )
			return 1;
	}
	return 0;
}

int is_playlist(const char *s)
{
    return is_ext(s,".m3u") || is_ext(s,".pls") || is_ext(s,".opl");
}

int compare(const void* a, const void* b)
{
    return strcasecmp((const char *)a, (const char *)b);
}

int read_file(char *fbuf, const char* fname, int fsize)
{
    FILE* fp = fopen(fname,"rb"); 
    if (fp)
    {
        int fsize = fread(fbuf, sizeof(char), fsize, fp);
        fclose(fp);
        return fsize;
    }
    else
        return -1;
}

int read_zip(char *dest, const char* zipname, int length, int index)
{
    fex_t *fex;
    error( fex_open( &fex, zipname ) );
    
    // Search for file by it's index
    // TODO - Do another check for security?
    int i = 0;
    while( !fex_done( fex ) ) {
        if( i == index ) break;
        error( fex_next( fex ) );
        i++;
    }

    error( fex_stat( fex ) );
    fprintf(stderr, "read_zip: %s [%d / %d]\n", fex_name( fex ), length, fex_size( fex ));
    
    error( fex_read( fex, dest, length ) );
    fex_close( fex );
    return length;
}

int plugin_filter_cb(void *data, const char *name)
{
	PlayerApp *app = (PlayerApp *)data;
	if(app->music_player->CanHandle(name))
		return 1;
	if(is_fex(name) || is_ext(name, ".m3u") || \
       is_ext(name, ".pls") || is_ext(name, ".opl"))
		return 1;
	return 0;
} 

int plugin_filter_zip_cb(void *data, const char *name)
{
    if (is_fex(name))
        return 0;
    return plugin_filter_cb(data, name);
}

int filter_opl_cb(void *data, const char *name)
{
    return is_ext(name,".opl");
}

char *get_string(struct id3_tag *id3tag, const char *what)
{
	struct id3_frame *frame = id3_tag_findframe(id3tag, what, 0);
	if(frame)
	{
		id3_field *field = &frame->fields[1];
		int nstrings = id3_field_getnstrings(field);
		for (int j=0; j<nstrings; ++j)
		{
			const id3_ucs4_t *ucs4 = id3_field_getstrings(field, j);
			id3_latin1_t *latin1 = id3_ucs4_latin1duplicate(ucs4);
			if(latin1)
				return (char*)latin1;
            //int len = strlen((char *)latin1);
		}
	}
	return NULL;
}

int parse_id3v1(char* fname, string* fieldname, string* fielddata, int max)
{
	id3_file *id3file = id3_file_open(fname, ID3_FILE_MODE_READONLY);
	struct id3_tag *id3tag = id3_file_tag(id3file);

	char *s;
    int x = 0;
	if (s = get_string(id3tag, ID3_FRAME_TITLE))
    {
        fieldname[x] = "Title";
        fielddata[x] = s;
        if (s = get_string(id3tag, ID3_FRAME_TRACK))
            fielddata[x] = string(s) +". "+fielddata[x];
        x++;
    }
    else
    {
        fieldname[x] = "File";
        fielddata[x++] = strrchr(fname,'/')+1;
    } // 1
    if (x <= max && (s = get_string(id3tag, ID3_FRAME_ALBUM)))
    {
        fieldname[x] = "Album";
        fielddata[x] = s;
        if ((s = get_string(id3tag, ID3_FRAME_YEAR)))
            fielddata[x] += " ("+string(s)+")";
        x++;
    } // 2
	if (x <=max && (s = get_string(id3tag, ID3_FRAME_ARTIST)))
    {
        fieldname[x] = "Artist";
        fielddata[x++] = s;
    } // 3
	if (x <= max && (s = get_string(id3tag, ID3_FRAME_GENRE)))
    {
        fieldname[x] = "Genre";
        fielddata[x++] = s;
    } // 4
	if (x <= max && (s = get_string(id3tag, ID3_FRAME_COMMENT)))
    {
        fieldname[x] = "Comment";
        fielddata[x++] = s;
    }
	id3_file_close(id3file);
    return x;
}

