/*  
    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <stdarg.h>
#include <SDL/SDL.h>
#include <SDL/SDL_thread.h>
#include "../plugin.h"
#include "../util.h"

extern "C"
{
#include "eagleplayer.h"
#include "uadecontrol.h"
#include "uadeconf.h"
#include "songdb.h"

#include "uadeipc.h"
//int uade_init();
//void uade_go();
}

#define INSIZE 16384

static uint8_t space[UADE_MAX_MESSAGE_SIZE];
static struct sound_plugin plugin;
static struct eagleplayerstore *eaglestore;
static struct eagleplayer *ep;
static struct uade_ipc uadeipc;
static struct uade_song *us;
static struct uade_config uc;
static struct uade_msg *um = (struct uade_msg *) space;
static char current_format[80] = "";
static int     left        = 0;
static int     subsong_end = 0;
static int     tailbytes   = 0;
static int     new_subsong = -1;
static int     uade_song_end_trigger = 0;
static int     playbytes;
static short   soundbuffer[32768];
static short  *soundptr = soundbuffer;
enum uade_control_state state = UADE_S_STATE;
static string fieldname[5];
static string fielddata[5];

unsigned int htonl(unsigned int l)
{ return ((l>>24) & 0xff) | ((l>>8) & 0xFF00) | ((l<<8) & 0xFF0000) | (l<<24); }

unsigned int ntohl(unsigned int l)
{ return ((l>>24) & 0xff) | ((l>>8) & 0xFF00) | ((l<<8) & 0xFF0000) | (l<<24); }

short ntohs(short l)
{ return ((l>>8) & 0xff) | ((l<<8) & 0xFF00); }

extern "C"
{
void client_sleep() { SDL_Delay(5); }
int release_write_mutex(void **m) { return (SDL_mutexV((SDL_mutex*)*m) == 0); }
int release_read_mutex(void **m)  { return (SDL_mutexV((SDL_mutex*)*m) == 0); }
int get_write_mutex(void **m)
{
	if(!*m)
		*m = SDL_CreateMutex();
	return (SDL_mutexP((SDL_mutex*)*m) == 0);

}
int get_read_mutex(void **m)
{
	if(!*m)
		*m = SDL_CreateMutex();
	return (SDL_mutexP((SDL_mutex*)*m) == 0);
}
} // extern "C"

int uade_thread(void *data)
{
	fprintf(stderr, "uadeplugin::uade_thread - Starting uade core\n");
	//uade_init();
	//uade_go();
	return 0;
}

static int run_client()
{
	if(state == UADE_S_STATE)
	{
		if(uade_song_end_trigger)
			return -1;
		/*if(uade_song_end_trigger)
		{
			next_song = 1;
			if(uade_send_short_message(UADE_COMMAND_REBOOT, &uadeipc))
			{
				fprintf(stderr, "\nCan not send reboot\n");
				return 0;
			}
		}
		else */

		if(new_subsong >= 0)
		{
			uade_change_subsong(new_subsong, &uadeipc);
			new_subsong = -1;
		}
		
		left = uade_read_request(&uadeipc); 
        // Request another batch of sample data from uadecore

		if(uade_send_short_message(UADE_COMMAND_TOKEN, &uadeipc))
		{
			fprintf(stderr, "uadeplugin:run_client - Can not send token\n");
			return 0;
		}
		state = UADE_R_STATE;
	}

	/* receive state */
	if(state == UADE_R_STATE)
	{	
		uint16_t *sm;

		if (uade_receive_message(um, sizeof(space), &uadeipc) <= 0)
		{
			fprintf(stderr, "uadeplugin:run_client - Can not receive events from uade\n");
			return 0;
		}

		switch (um->msgtype)
		{
		case UADE_COMMAND_TOKEN:
			state = UADE_S_STATE;
			break;

		case UADE_REPLY_DATA:
			sm = (uint16_t *)um->data;

			if(subsong_end)
			{
				playbytes = tailbytes; // Determined by UADE_REPLY_SONG_END
				tailbytes = 0;
			}
			else
				playbytes = um->size;

			if(plugin.freq == 44100)
			{
				for (int i = 0; i < um->size/2; i++)
					soundptr[i] = ntohs(sm[i]);

				soundptr += (playbytes/2);
			}
			else if(plugin.freq == 22050)
			{
				for (int i = 0; i < um->size/2; i+=4)
				{
					soundptr[i/2] = (ntohs(sm[i]) + ntohs(sm[i+2])) / 2;
					soundptr[i/2+1] = (ntohs(sm[i+1]) + ntohs(sm[i+3])) / 2;
				}

				soundptr += (playbytes/4);
			}
			//printf("Got %d bytes sampledata\n", playbytes);
			//time_bytes += playbytes;
			assert(left >= um->size);
			left -= um->size;
			break;

		case UADE_REPLY_FORMATNAME:
			uade_check_fix_string(um, 128);
			fprintf(stderr, "uadeplugin:run_client - Format name: %s\n", (char*) um->data);
			strcpy(current_format, (char *)um->data);
			break; 

		case UADE_REPLY_MODULENAME:
			uade_check_fix_string(um, 128);
			fprintf(stderr, "uadeplugin:run_client - Module name: %s\n", (char*) um->data);
			break; 

		case UADE_REPLY_MSG:
			uade_check_fix_string(um, 128);
			fprintf(stderr, "uadeplugin:run_client - Message: %s\n", (char *) um->data);
			break;

		case UADE_REPLY_PLAYERNAME:
			uade_check_fix_string(um, 128);
			fprintf(stderr, "uadeplugin:run_client - Player name: %s\n", (char*) um->data);
			strcpy(current_format, (char *)um->data);
			break; 

		case UADE_REPLY_SONG_END:
			if (um->size < 9)
			{
				fprintf(stderr, "uadeplugin:run_client - Invalid song end reply\n");
				exit(-1);
			}
			tailbytes = ntohl(((uint32_t *) um->data)[0]);
			uade_song_end_trigger = 1;
			fprintf(stderr, "uadeplugin:run_client - Song end signal received\n");
			break;

		case UADE_REPLY_SUBSONG_INFO:
			if(um->size != 12)
			{
				fprintf(stderr, "uadeplugin:run_client - subsong info: too short a message\n");
			}
			{
				unsigned int *u32ptr = (unsigned int *)um->data;
				int min_sub = ntohl(u32ptr[0]);
				int max_sub = ntohl(u32ptr[1]);
				int cur_sub = ntohl(u32ptr[2]);
				plugin.subtunes = max_sub ? max_sub : 1;
				plugin.tune = cur_sub;
				fprintf(stderr, "uadeplugin:run_client - subsong: %d from range [%d, %d]\n", 
                                cur_sub, min_sub, max_sub);
			}
			break;
		case UADE_REPLY_CANT_PLAY:
		case UADE_REPLY_CAN_PLAY:


		default:
			fprintf(stderr, "uadeplugin:run_client - Expected sound data. got %d.\n", um->msgtype);
			return 0;
		}
	}
    return 0;
}

static int fill_buffer(signed short *dest, int len)
{
	if(run_client() < 0)
		return 0;

    int samp_len = len>>1;
	int filled = soundptr - soundbuffer;
	while(filled < len)
	{
		if(run_client() < 0)
			return 0;
		filled = soundptr - soundbuffer;
	}
    memcpy(dest, soundbuffer, len);
    memmove(soundbuffer, soundbuffer+samp_len, filled*2-len);
    soundptr -= samp_len;
	return len;
}

static int init_file(char *songname) // Use same naming convention as uade123.c
{
	char playername[256];
	char modulename[256];   
    strncpy(modulename, songname, sizeof modulename);

    if ((ep = uade_analyze_file_format(modulename, &uc)) = NULL)
    {
        uade_unalloc_song(us);
        fprintf(stderr, "uadeplugin:init_file - Can't handle file\n");
        return -1;
    }

    if (strcmp(ep->playername, "custom") == 0)
    {
        strncpy(playername, modulename, sizeof playername);
        modulename[0] = 0;
    } 
    else 
    {
        snprintf(playername, sizeof playername, "players/%s", ep->playername);
    }
    sprintf(playername, "players/%s", ep->playername);

    if ((us = uade_alloc_song(songname)) == NULL)
    {
        fprintf(stderr, "uadeplugin:init_file - Can not read %s\n", songname);
        return -1;
    }

    fprintf(stderr, "uadeplugin:init_file - player %s, module %s\n", playername, modulename);
    int rc = uade_song_initialization(
                            "score", 
                            playername, 
                            modulename, 
                            us,
                            &uadeipc,
                            &uc);
    
	plugin.length = -1;
	plugin.subtunes = 1;
	plugin.tune = 0;

    // Info section
    int x = 0;
    fieldname[x] = "File";
    fielddata[x++] = strrchr(songname, '/')+1;
    fieldname[x] = "Format";
    fielddata[x++] = strlen(current_format) ? current_format : "UADE";
    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;
    plugin.nfields   = x;
    // Info section 

	uade_song_end_trigger = 0;

	if(rc) return rc;

	run_client();
	return 0;
}

static int wait_token()
{
	do
	{
		int ret = uade_receive_message(um, sizeof(space), &uadeipc);
		if(ret < 0)
		{
			fprintf(stderr, "uadeplugin:wait_token - Can not receive events (TOKEN) from uade.\n");
			return 0;
		}
		if (ret == 0)
		{
			fprintf(stderr, "uadeplugin:wait_token - End of input after reboot.\n");
			return 0;
		}
	} while (um->msgtype != UADE_COMMAND_TOKEN);
	return 1;
}

static int close()
{
	if(state == UADE_R_STATE)
		wait_token();

	if(uade_send_short_message(UADE_COMMAND_REBOOT, &uadeipc))
	{
		fprintf(stderr, "uadeplugin:close - Can not send reboot\n");
		return 0;
	}
    if (uade_send_short_message(UADE_COMMAND_TOKEN, &uadeipc))
	{
		fprintf(stderr, "uadeplugin:close - Can not send token\n");
		return 0;
	}

	wait_token();

	state = UADE_S_STATE;

	return 0;
}

static int can_handle(const char *name)
{
	return (uade_analyze_file_format(name, &uc) != NULL);
}

extern "C"
{

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN uade_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
	uade_set_peer(&uadeipc, 1, "client", "server");
	SDL_Thread *thread = SDL_CreateThread(uade_thread, NULL);
	if (uade_send_string(UADE_COMMAND_CONFIG, "uaerc", &uadeipc))
    {
		fprintf(stderr, "uadeplugin - Can not send config name\n");
		SDL_KillThread(thread);
	    return NULL;
	} 

    uade_config_set_defaults(&uc);
    uade_set_filter_type(&uc, 0);//uadeconf.filter_type = 0;
 
	memset(&plugin, 0, sizeof(plugin));

	plugin.plugname     = "uade";
	plugin.init_file    = init_file;
	plugin.init_data    = NULL; //init_data;
	plugin.fill_buffer  = fill_buffer;
	plugin.can_handle   = can_handle;
	plugin.close        = close;
	plugin.set_position = NULL; //set_position;
	plugin.freq         = 22050;
	plugin.channels     = 2;
	plugin.clockfreq    = 250;

	return &plugin;
}

}  // extern "C"
