/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <stdarg.h>
#include <SDL/SDL.h>
#include <SDL/SDL_thread.h>
#include "../plugin.h"
#include "../util.h"

#include "../FormatsDatabase.h"

extern "C"
{
#include "amifilemagic.h"
#include "uadecontrol.h"
#include "uadeconf.h"
#include "uade/include/uadeipc.h"

#include "uade4all.h"

void uade_init(const char *optionsfile);
void uade_go();
void uae_quit();
}

static FormatsDatabase formatsdb;

char current_format[80] = "";
struct uade_config uadeconf;
uint8_t space[UADE_MAX_MESSAGE_SIZE];
struct uade_msg *um = (struct uade_msg *) space;
struct uade_ep_options ep_options;

static struct uade_ipc uadeipc;
static int first_run = 1;

static string fieldname[3]; 
static string fielddata[3];

#ifndef WIN32
#define stricmp strcasecmp
#define strnicmp strncasecmp
#endif

/* Extra Functions */

/* TODO - Move these back to eagleplayer.c? */

#define LINESIZE (1024)
#define WS_DELIMITERS " \t\n"
#define OPTION_DELIMITER ","

char **split_line(size_t *nitems, size_t *lineno, FILE *f,
			 const char *delimiters)
{
	char line[LINESIZE], templine[LINESIZE];
	char **items = NULL;
	size_t pos;
	char *sp, *s;

	*nitems = 0;

	while (fgets(line, sizeof line, f) != NULL) {
		if (lineno != NULL)
			(*lineno)++;

		/* Skip, if a comment line */
		if (line[0] == '#')
			continue;

		/* strsep() modifies line that it touches, so we make a copy of it, and
		then count the number of items on the line */
		strncpy(templine, line, sizeof(templine));
		sp = templine;
		while ((s = strsep(&sp, delimiters)) != NULL) {
			if (*s == 0)
				continue;
			(*nitems)++;
		}

		if (*nitems > 0)
			break;
	}

	if (*nitems == 0)
		return NULL;

	if ((items = (char **)malloc(sizeof(items[0]) * (*nitems + 1))) == NULL) {
		fprintf(stderr, "split_line - No memory for nws items.\n");
		exit(-1);
	}

	sp = line;
	pos = 0;
	while ((s = strsep(&sp, delimiters)) != NULL) {
		if (*s == 0)
			continue;

		if ((items[pos] = strdup(s)) == NULL) {
			fprintf(stderr, "No memory for an nws item.\n");
			exit(-1);
		}
		pos++;
	}

	items[pos] = NULL;
	if( pos != *nitems ) return NULL;

	return items;
}



char *uade_get_eagleplayer(const char *ext, const char *config, int *ep_attributes)
{
	FILE *f;
	char playername[128];
	int found = 0;
	size_t lineno = 0;
	size_t exti;
	size_t i;

	// Extensions
	int len = strlen(ext), tmp;

	f = fopen(config, "r");
	if (f == NULL)
		goto error;

	while (1) {
		char **items;
		size_t nitems;

		if ((items = split_line(&nitems, &lineno, f, WS_DELIMITERS)) == NULL)
			break;

		if( nitems <= 0 ) {
			break;
		}

		strcpy(playername, items[0]);

        if( ep_attributes ) {
		    // Reset eagleplayer attributes and options
    		*ep_attributes = 0;
    		memset(&ep_options, 0, sizeof(struct uade_ep_options));
        }

		for (i = 1; i < nitems; i++) {
			if (strncasecmp(items[i], "prefixes=", 9) == 0) {
				char prefixes[LINESIZE];
				char *prefixstart = items[i] + 9;
				char *sp, *s;
				size_t pos;

				pos = 0;
				sp = prefixstart;
				while ((s = strsep(&sp, OPTION_DELIMITER)) != NULL) {
					if (*s == 0) continue;
					//fprintf(stderr, "uade_get_eagleplayer - ext: %s, prefix: %s\n", ext, s);
					tmp = strlen(s);
					if( tmp == len )
						if( strcasecmp(ext, s) == 0 ) {
							found = 1;
							break;
						}
					pos++;
				}

                continue;
			}
 
            if( ep_attributes ) { 
    			// Get attributes
                if (strcasecmp(items[i], "content_detection") == 0) {
    				*ep_attributes |= EP_CONTENT_DETECTION;
    			}
	    		else if (strcasecmp(items[i], "detect_format_by_content") == 0) {
	    			*ep_attributes |= EP_CONTENT_DETECTION;
	    		}
	    		else if (strcasecmp(items[i], "a500") == 0) { 
	    			*ep_attributes |= EP_A500;
	    		} else if (strcasecmp(items[i], "a1200") == 0) {
	    			*ep_attributes |= EP_A1200;
	    		} else if (strcasecmp(items[i], "always_ends") == 0) {
	    			*ep_attributes |= EP_ALWAYS_ENDS;
	    		} else if (strcasecmp(items[i], "speed_hack") == 0) {
	    			*ep_attributes |= EP_SPEED_HACK;
	    		} else if (strcasecmp(items[i], "ignore_player_check") == 0) {
	    			*ep_attributes |= ES_IGNORE_PLAYER_CHECK;
	    		} else if (strcasecmp(items[i], "broken_song_end") == 0) {
	    			*ep_attributes |= ES_BROKEN_SONG_END;
	    		} else if (strcasecmp(items[i], "detect_format_by_name") == 0) {
	    			*ep_attributes |= ES_NAME_DETECTION;
	    		} else if (strncasecmp(items[i], "epopt=", 6) == 0) {
	    			uade_add_ep_option(&ep_options, items[i] + 6);
	    		} else if (strncasecmp(items[i], "comment:", 8) == 0) {
	    			break;
	    		} else {
	    			fprintf(stderr, "uade_get_eagleplayer - Unrecognized option: %s\n", items[i]);
	    		}
            }
		}

		// Free memory
		for (i = 0; i < nitems; i++)
			free(items[i]);
		free(items);
		if( found ) break;
    }

	fclose(f);
	if( found ) {
		return strdup(playername);
	}

	return NULL;

error:
	if( playername ) free(playername);
	if (f != NULL) fclose(f);
	return NULL;
}

/* SUPPORT */
size_t atomic_fread(void *ptr, size_t size, size_t nmemb, FILE *stream)
{
  uint8_t *dest = (uint8_t *)ptr;
  size_t readmembers = 0;
  size_t ret;

  while (readmembers < nmemb) {
    ret = fread(dest + size * readmembers, size, nmemb - readmembers, stream);
    if (ret == 0)
      break;
    readmembers += ret;
  }

  assert(readmembers <= nmemb);
  return readmembers;

}



char *uade_analyze_format(const char *path, const char *name, int *ep_attributes)
{
	if( path == NULL | name == NULL ) return NULL;

	fprintf(stderr, "uade_analyze_format - Analizing (%s - %s)\n", path, name);

	const char *sext = strrchr(name, '.');
	char prefix[strlen(name)+1];

	char ext[128];
	size_t bufsize, bytesread;
	uint8_t buf[8192];

    if( ep_attributes ) {
    	// Get some magic
	    FILE *f;
	    int size;
	    if ((f = fopen(path, "rb")) == NULL) {
	    	fprintf(stderr, "uade_analyze_format - Cannot load %s\n", path);
	    	return NULL;
	    }

	    // Get file size
	    size = fseek(f, 0, SEEK_END);
	    fseek(f, 0, SEEK_SET);

	    bufsize = sizeof buf;
	    bytesread = atomic_fread(buf, 1, bufsize, f);
	    fclose(f);

	    if (bytesread == 0)
	    	return NULL;

	    memset(&buf[bytesread], 0, bufsize - bytesread);

	    uade_filemagic(buf, bytesread, ext, size, path, 1);

	    if (strcmp(ext, "reject") == 0)
	    	return NULL;

	    if (ext[0] != 0)
	    	fprintf(stderr, "uade_analyze_format - Content recognized: %s (%s)\n", ext, path);

	    if (strcmp(ext, "packed") == 0)
	    	return NULL;
    }
    
    // Try prefix and extension first
    if( !sext )
	{
		sext = "";
        	prefix[0] = '\0';
    	}
	else
    	{
		sext++;

    	const char* name2 = strrchr(name,'/');
        name2 = name2 ? name2+1 : name;

        strcpy(prefix,name2);
        *strchr(prefix,'.') = '\0';

#ifdef __PSP
	    /* HACK */
	    char *tmp = strchr(prefix, '~');
	    if( tmp ) *tmp = '\0';
#endif
    }


	/* First do filename detection (we'll later do content detection) */
	char *name_pl = uade_get_eagleplayer(prefix, "eagleplayer.conf", ep_attributes);
	if( name_pl == NULL ) {
		name_pl = uade_get_eagleplayer(sext, "eagleplayer.conf", ep_attributes);
	}
	else {
	}

    if( ep_attributes == NULL ) return name_pl;

	// If name detection skip content detection
	if( name_pl && (*ep_attributes & ES_NAME_DETECTION) ) return name_pl;

	/* If filemagic found a match, we'll use player plugins associated with
	   that extension */

	if (ext[0]) {
		int _ep_attributes = 0;
		char *content_pl = uade_get_eagleplayer(ext, "eagleplayer.conf", &_ep_attributes);
		if( content_pl ) {
			*ep_attributes = _ep_attributes;
			return content_pl;
		}
	}

	return name_pl;
}


/* TODO - remove all of this since they
          are not needed anymore */
int cprintf(char *fmt, ...);
unsigned int htonl(unsigned int l)
{
	return ((l>>24) & 0xff) | ((l>>8) & 0xFF00) | ((l<<8) & 0xFF0000) | (l<<24);
}

unsigned int ntohl(unsigned int l)
{
	return ((l>>24) & 0xff) | ((l>>8) & 0xFF00) | ((l<<8) & 0xFF0000) | (l<<24);
}

short ntohs(short l)
{
	return ((l>>8) & 0xff) | ((l<<8) & 0xFF00);
}

static struct sound_plugin plugin; 


extern "C"
{

void client_sleep()
{
	SDL_Delay(5);
}

int get_write_mutex(void **m)
{
	if(!*m) *m = SDL_CreateMutex();
	return (SDL_mutexP((SDL_mutex*)*m) == 0);

}

int release_write_mutex(void **m)
{
	return (SDL_mutexV((SDL_mutex*)*m) == 0);
}

int get_read_mutex(void **m)
{
	if(!*m) *m = SDL_CreateMutex();
	return (SDL_mutexP((SDL_mutex*)*m) == 0);
}

int release_read_mutex(void **m)
{
	return (SDL_mutexV((SDL_mutex*)*m) == 0);
}

} // extern "C"

static void *uade_thread(void *data)
{
	uade_go();
    pthread_exit ( 0 );
}


extern "C" {
	void fill_audio_buffer (Uint8 *stream, int len);
    void flush_audio_buffer();
    void uade_change_subsong(int subsong);

	extern int quit_program;

    void uade_set_subtunes(int mins, int maxs, int curs)
    {
        plugin.subtunes = maxs ? maxs : 1;
    	plugin.tune = curs ? curs-1 : 0;				
    }

    /* To allow uade automatically change subsong */
    int uade_request_change_subtune() 
    {
        if( plugin.tune < (plugin.subtunes-1) ){
            fprintf(stderr, "uadeplugin - changing subtune %d\n", plugin.tune+1);
            plugin.tune++;
            uade_change_subsong(plugin.tune);
            return 1;
        } 

        return 0;
    }

    void uade_free_memory (void); 
    void uade_reset_memory (void);
    void uade_free_table68k (void);
    void uade_reset_table68k (void);
}

static int fill_buffer(signed short *dest, int len)
{
    if( quit_program ) return 0;
 
    fill_audio_buffer((Uint8 *)dest, len);
	return len;
}
 
pthread_t uade_thread_id = 0;
pthread_attr_t pattr;
sched_param param;

static int init_file(char *fname)
{
	char *playername;
	char tmp[256], tmp2[256];
	char *src = fname;
	char *dst = tmp2;

    /* Save memory for other plugins */
    if( first_run ) {
       	uade_init("uaerc");
        first_run = 0;
    } else {
        uade_reset_memory();
        uade_reset_table68k();
    }

#ifdef A320
	plugin.clockfreq = 336;
    if( strstr(fname, "mdat") ) plugin.clockfreq = 384;
#else
	plugin.clockfreq = 250;
#endif

	int rc;

	int ep_attributes = 0;
	const char *name = strrchr(fname, '/');
	playername = uade_analyze_format(fname, name ? name+1 : NULL, &ep_attributes);
	if( playername == NULL ) {
        fprintf(stderr, "init_file - could not find player for: %s\n", fname);
        return -1;
    }

	fprintf(stderr, "init_file - player: %s\n", playername);
	sprintf(tmp, "players/%s", playername);

	while(*src)
	{
		if(*src == '\\')
			*dst++ = '/';
		else
			*dst++ = *src;
		src++;
	}

	*dst = 0;

	strcpy(current_format, "");
	if(strcmp("custom", playername) == 0)
	{
        rc = uade_load_song(NULL, "score", tmp2);
		strcpy(current_format, "Amiga Custom");
	}
	else
		rc = uade_load_song(tmp2, "score", tmp);

	if(rc == 0) return -2;

	plugin.length = -1;
	plugin.subtunes = 1;
	plugin.tune = 0;

	int x = 0;
    fieldname[x] = "File";
    fielddata[x++] = strrchr(fname, '/')+1;
    fieldname[x] = "Format";
    fielddata[x++] = strlen(current_format) ? current_format : "UADE";
    fieldname[x] = "Player";
    fielddata[x++] = string(playername);

    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;
    plugin.nfields   = x;

    if( playername ) free( playername );

	uade_config_set_defaults(&uadeconf);

	// Set Eagleplayer attributes
	if (ep_attributes & ES_IGNORE_PLAYER_CHECK) {
		uade_set_command(UADE_COMMAND_IGNORE_CHECK);
    }

    if (ep_attributes & ES_BROKEN_SONG_END) {
		uade_set_command(UADE_COMMAND_SONG_END_NOT_POSSIBLE);
    }

	// if (uade_send_short_message(UADE_COMMAND_SET_NTSC, &uadeipc) < 0 ) return -3;

	// Speed hack
	if (ep_attributes & ES_SPEED_HACK) {
		uade_set_command(UADE_COMMAND_SPEED_HACK);
    }

	// Send ep options
	if (uade_send_ep_options(&ep_options, &uadeipc))
		return -3;
	
    // Create UADE Thread
	//SDL_Thread *thread = SDL_CreateThread(uade_thread, NULL);
    pthread_attr_init(&pattr);
	pthread_attr_getschedparam (&pattr, &param);

	/* set the priority; others are unchanged */
	param.sched_priority = 0;/* -20 high to 19 low. 0 == default */

   	/* setting the new scheduling param */
	pthread_attr_setschedparam (&pattr, &param);
    pthread_create( &uade_thread_id, &pattr, uade_thread, NULL);

	return 0;
}

static int close()
{
    if( uade_thread_id ) {
        // Kill UADE thread
    	uae_quit();
        
        flush_audio_buffer();

        pthread_join(uade_thread_id, NULL);
        uade_thread_id = 0;
    } 

    uade_free_memory();
    uade_free_table68k();
    
    for(int i = 0; i < plugin.nfields; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }

    plugin.tune = plugin.subtunes = plugin.length = plugin.nfields = 0;
	return 0;
}

int set_position(int msec, int subtune)
{
    printf("uadeplugin:set_position - %d, %d\n",msec,subtune);
    if (!msec) {
    	uade_change_subsong(subtune);
        return 1;
    }
    else
        return 0; // No seeking
}

static int can_handle(const char *name)
{
    char tmp[256], *tmp2;
    tmp2 = strrchr(name, '/');
    strncpy(tmp, tmp2 ? tmp2+1 : name, 256);
 
    char *prefix = strtok(tmp, ".");
	const char *sext = strrchr(name, '.');

    // Try with prefix first
    int ret = 0;
    ret = formatsdb.Find(prefix ? prefix: "");    

    return ret ? ret : formatsdb.Find(sext ? sext+1 : "");
}

extern "C"
{

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN uade_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
     // Load formats database
    if( !formatsdb.Load("[UADE]", "formats.db") )
        exit(-1);

    /* TODO- this shouldn't be necessary anymore */
   	uade_set_peer(&uadeipc, 1, "client", "server");

	memset(&plugin, 0, sizeof(plugin));

	plugin.plugname = "uade";
	plugin.freq = 44100;
	plugin.channels = 2; 
	plugin.init_file = init_file;
	//plugin.init_data = init_data;
	plugin.fill_buffer = fill_buffer;
	plugin.can_handle = can_handle;
	plugin.close = close;
	plugin.set_position = set_position;

	plugin.replaygain = 1;

	return &plugin;
}

}  // extern "C"
