/* 
    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
extern "C" {
#include "tta/ttalib.h"
}
#include "../plugin.h"
#include "../util.h"

static struct sound_plugin plugin;
static tta_info info;
static int playing;
signed short *ttabuf = NULL;
static int buffsize;
static int writeposbytes;
static int readposbytes;
static int samples_done;

static string fieldname[5];
static string fielddata[5];

static int close();
static int init_file(char *fname)
{
    playing = 0;
    buffsize = 0;
#ifdef A320
    plugin.clockfreq = 360;
#else
    plugin.clockfreq = 200; 
#endif

    // TODO: read ID3v2 headers and calc. it's offset. Or don't I have to?
    if (open_tta_file(fname, &info, 0))
    {
        fprintf(stderr, "ttaplugin:init_file - Couldn't open file\n");
        return -1;
    }
    if (player_init(&info))
    {
        fprintf(stderr,  "ttaplugin:init_file - player_init(info) failed\n");
        close_tta_file(&info);
        return -1;
    }
    id3_info id3info = info.ID3;
    
    int x = 0;
    if (strlen((const char*)id3info.name))
    {
        fieldname[x] = "Name";
        fielddata[x] = (const char*)id3info.name;
        x++;
    }
    if (strlen((const char*)id3info.title))
    {
        fieldname[x] = "Title";
        fielddata[x] = (const char*)id3info.title;
        if (strlen((const char*)id3info.track))
            fielddata[x] = (const char*)id3info.track + fielddata[x];
        x++;
    }
    if (strlen((const char*)id3info.album))
    {
        fieldname[x] = "Album";
        fielddata[x] = (const char*)id3info.album;
        x++;
    }
    if (strlen((const char*)id3info.genre))
    {
        fieldname[x] = "Genre";
        fielddata[x] = (const char*)id3info.genre;
        x++;
    }
    if (x < 4 && strlen((const char*)id3info.comment))
    {
        fieldname[x] = "Comment";
        fielddata[x] = (const char*)id3info.comment;
        x++;
    }
    fieldname[x] = "Format";
    fielddata[x] = "TTA";
    x++;
    plugin.nfields = x;
    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;

    plugin.freq     = info.SAMPLERATE;
    plugin.channels = info.NCH;
    plugin.length   = info.LENGTH*1000;
    buffsize        = PCM_BUFFER_LENGTH*plugin.channels;

    printf("%d, %d, %d, %d\n", \
        plugin.freq,plugin.channels,plugin.length,buffsize);
    if(!(ttabuf = (signed short*)malloc(buffsize*2)))
    {
        fprintf(stderr, "ttaplugin:init_file - Cannot allocate memory\n");
        close();
        return -1;
    }
    playing = 1;
    writeposbytes = 0;
    readposbytes  = 0;
    samples_done  = 0;
    return 0;
}

static int close()
{
    fprintf(stderr, "foo\n");
    close_tta_file(&info);
    player_stop();

    fprintf(stderr, "foo 1\n");
    if (ttabuf)
    {
        free(ttabuf);
        ttabuf = NULL;
    }

    fprintf(stderr, "foo 2\n");
    for(int i = 0; i < 5; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }
    playing = plugin.length = plugin.nfields = 0;
    return 0;
}

static void bufcpy(signed short *dest, signed short* src, int len)
{
    memcpy(dest,src,len);
}

static int fill_buffer(signed short *dest, int len)
{
    if(playing)
    {        
        int writtenbytes = 0;
        for (;;)
        {
            int bytesleft = len - writtenbytes;
            int bufbytes  = writeposbytes - readposbytes;
            //printf("len = %04d - written = %04d = %04d, " \
                    "read = %04d - write = %04d = %04d\n",\
                len,writtenbytes,bytesleft,readposbytes,writeposbytes,bufbytes);
            if (bytesleft <= bufbytes)
            {
                //memcpy((char*)dest + writtenbytes, (char*)ttabuf + readposbytes, bytesleft);
                bufcpy(&dest[writtenbytes>>1], &ttabuf[readposbytes>>1], bytesleft);
                readposbytes += bytesleft;
                writtenbytes = len;
                break;
            }
            else
            {
                //memcpy((char*)dest + writtenbytes, (char*)ttabuf + readposbytes, bufbytes);
                bufcpy(&dest[writtenbytes>>1], &ttabuf[readposbytes>>1], bufbytes);
                writtenbytes += bufbytes;
                writeposbytes = 0;
                writeposbytes = get_samples((unsigned char*)ttabuf)*2*plugin.channels; 
                readposbytes = 0;
                if (!writeposbytes)
                    break;
            }
        }
        samples_done += writtenbytes/(2*plugin.channels);
        //printf("written = %d\n",writtenbytes);
        return writtenbytes;
    }
    return 0;
}

static int set_position(int msecs, int subtune)
{
    if (playing && msecs)
    {
        //seek position = seek_time_ms / SEEK_STEP
        int now_pos = (int)((double)samples_done*1000/(plugin.freq*SEEK_STEP));
        int to_pos = msecs / SEEK_STEP;
        if (now_pos + to_pos <= 0)
            to_pos = 0;
        else
            to_pos += now_pos;

        set_position(to_pos);
        readposbytes = 0;

        int samples_done_new = (int)((double)plugin.freq*to_pos*SEEK_STEP/1000);
        int timejump = (int)((samples_done_new - samples_done)*1000.0/plugin.freq);
        samples_done = samples_done_new;
        return timejump;

    }
}

static int can_handle(const char *name)
{
    return (is_ext(name, ".tta")); 
}

extern "C" {

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN tta_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
    memset(&plugin, 0, sizeof(plugin));
    plugin.plugname = "TTA";
    //plugin.init_data = init_data;
    plugin.init_file = init_file;
    //plugin.request_format = NULL;
    plugin.set_position = set_position;
    plugin.fill_buffer = fill_buffer;
    plugin.can_handle = can_handle;
    plugin.close = close;
    plugin.tune = 0;
    plugin.subtunes = 1;
    plugin.replaygain = 1;
    return &plugin;
}

}

