/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "stsound/StSoundLibrary.h"

#include "../plugin.h"
#include "../util.h"

static struct sound_plugin plugin;
YMMUSIC *pMusic = NULL;

static int playing;
static string fieldname[1];
static string fielddata[1];
static string filename;
static int set_position(int msecs, int subtune);

static int init_file(char *fname)
{
    playing = 0;
    plugin.tune = 0;
    plugin.subtunes = 1;

    pMusic = ymMusicCreate();
	if( !ymMusicLoad(pMusic, fname) )
		return -1;	

    ymMusicInfo_t info;
	ymMusicGetInfo(pMusic, &info);

    int x = 0;
	if( info.pSongName ) {
        fieldname[x] = "Song";
        fielddata[x++] = string(info.pSongName);
    }
    
	if( info.pSongAuthor ) {
        fieldname[x] = "Author";
        fielddata[x++] = string(info.pSongAuthor);
    }

    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;
    plugin.nfields  = x;
    
	plugin.length = info.musicTimeInMs > 0 ? info.musicTimeInMs : 80000;
	
	ymMusicSetLoopMode(pMusic, YMFALSE);
	
	ymMusicStop(pMusic);
	ymMusicPlay(pMusic);

    playing = 1;
    return 0;
}

static int close()
{
    if( pMusic ) {
		ymMusicDestroy(pMusic);
		pMusic = NULL;
	}

    for(int i = 0; i < plugin.nfields; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }

    playing = plugin.tune = plugin.subtunes = plugin.length = plugin.nfields = 0;
    return 0;
}

static int fill_buffer(signed short *dest, int len)
{
    if(playing)
    {
        ymMusicCompute(pMusic, (ymsample *)dest, len/2);
    	return len;
    }

    return 0;
}

static int set_position(int msecs, int subtune)
{
    if (!playing)
        return 0;

    
    if( msecs && ymMusicIsSeekable(pMusic) )	{
        ymMusicSeek(pMusic, msecs);
        return msecs;
    }

    return 0;
}


static int can_handle(const char *name)
{
    return is_ext(name, ".ym");
}

extern "C" {

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN stsound_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
    memset(&plugin, 0, sizeof(plugin));
    plugin.plugname = "ST-Sound";
    plugin.init_file        = init_file;
    //plugin.init_data      = init_data;
    //plugin.request_format = NULL;
    plugin.set_position     = set_position;
    plugin.fill_buffer      = fill_buffer;
    plugin.can_handle       = can_handle;
    plugin.close            = close;
    plugin.channels   = 1;
    plugin.freq       = 44100;
    plugin.replaygain = 1;

#ifdef A320
    plugin.clockfreq = 264; 
#else
    plugin.clockfreq = 150; 
#endif

    return &plugin;
}

}

