/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
    */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sidplay/player.h>
#include <sidplay/fformat.h>
#include <sidplay/myendian.h>

#include "../plugin.h"

extern "C" {
#include "../md5.h"
}

#define SID_DOUBLE_MONO

class SidState
{
    public:
        SidState(emuEngine *e = NULL, sidTune *t = NULL)
        {
            engine = e;
            tune = t;
            playing = false;
        }

        emuEngine *engine;
        sidTune *tune;
        bool playing;
        int song;
};

static SidState state;
static struct sound_plugin plugin;
static string fieldname[5];
static string fielddata[5];

static void init_engine()
{
    SidState *s = &state;

    s->playing = false;

    if(!s->engine)
    {
        s->engine = new emuEngine();

        if(!s->engine->verifyEndianess())
        {
            exit(1);
        }

        // Get the default configuration.
        struct emuConfig myEmuConfig;
        s->engine->getConfig(myEmuConfig);
        myEmuConfig.frequency = plugin.freq;
        myEmuConfig.channels = plugin.channels == 2 ? SIDEMU_STEREO : SIDEMU_MONO;
        myEmuConfig.bitsPerSample = SIDEMU_16BIT;
        myEmuConfig.sampleFormat = SIDEMU_SIGNED_PCM;
        myEmuConfig.volumeControl = s->engine->setConfig(myEmuConfig);
    }
}

static int total = 0;

void string2val(char *str, unsigned int *csum)
{
    for(int i=0; i<4; i++)
    {
        char c = str[8];
        str[8] = 0;
        csum[i] = strtoul(str, NULL, 16);
        str[8] = c;
        str += 8;
    }
}

static int total_songs = 0;
static unsigned int *song_md5;
static unsigned char *song_len;

#define TO32(ptr) ((ptr[0]<<24) | (ptr[1]<<16) | (ptr[2]<<8) | (ptr[3]))

unsigned char *get_song_length(unsigned char *data, int len)
{
    uint8_t md5[16];
    MD5_CTX ctx;

    unsigned int csum[4];

    int hsize = 0x76;

    if(data[5] == 2)
        hsize += 6;

    if((data[8] | data[9]) == 0)
        hsize += 2;

    MD5Init(&ctx);
    MD5Update(&ctx, data+hsize, len-hsize);
    MD5Update(&ctx, &data[0xB], 1);
    MD5Update(&ctx, &data[0xA], 1);
    MD5Update(&ctx, &data[0xD], 1);
    MD5Update(&ctx, &data[0xC], 1);
    MD5Update(&ctx, &data[0xF], 1);
    MD5Update(&ctx, &data[0xE], 1);

    int songs          = data[0xf];
    unsigned char *ptr = &data[0x12];
    int speed          = TO32(ptr);
    unsigned char i8;

    int i;
    for(i = 0; i < songs ; i++)
    {
        i8 = (speed & (1 << i)) ? 60 : 0;
        MD5Update(&ctx, &i8, 1);
    }

    if(data[5] == 2 )
    {
        i8 = (data[0x76+1] >> 2) & 3;
        if (i8 == 2)
            MD5Update(&ctx, &i8, 1);
    }

    MD5Final(md5, &ctx);
    //string2val((char *)md5, csum);

    for(i=0; i<16; i+=4)
        csum[i/4] = (md5[i]<<24) | (md5[i+1]<<16) | (md5[i+2]<<8) | (md5[i+3]);

    unsigned int *mptr = song_md5;

    for(i=0; i<total_songs; i++)
    {
        if(csum[0] == mptr[0] && csum[1] == mptr[1] && csum[2] == mptr[2] && 
                csum[3] == mptr[3])
            return &song_len[mptr[4]]; //song_len[i] * 1000;
        mptr += 5;
    }
    
    return NULL;
}

static unsigned char *sidlenptr = NULL;

void set_len()
{
    if(sidlenptr)
    {
        int i = plugin.tune;
        int j = 0;
        while(i--)
        {
            if(sidlenptr[j] == 0xff)
                j += 3;
            else
                j++;
        }
        int len = sidlenptr[j];
        if(len == 0xff)
            len = (sidlenptr[j+1]<<8) | sidlenptr[j+2];
        plugin.length = len * 1000;
    }
    else
        plugin.length = -1;
}

static void write_fields(char *fname, sidTuneInfo *mySidInfo)
{
    int x = 0;
    if (mySidInfo->nameString && strlen(mySidInfo->nameString))
    {
        fieldname[x] = "Name";
        fielddata[x] = mySidInfo->nameString;
    }
    else
    {
        fieldname[x] = "File";
        char *name = strrchr(fname, '/');
        fielddata[x] = name ? name+1 : fname;
    }
    x++;
    if (mySidInfo->authorString && strlen(mySidInfo->authorString))
    {
        fieldname[x] = "Author";
        fielddata[x] = mySidInfo->authorString;
        x++;
    }
    if (mySidInfo->copyrightString && strlen(mySidInfo->copyrightString))
    {
        fieldname[x] = "Copyright";
        fielddata[x] = mySidInfo->copyrightString;
        x++;
    }
    if (mySidInfo->speedString && strlen(mySidInfo->speedString))
    {
        fieldname[x] = "Speed";
        fielddata[x] = mySidInfo->speedString;
        x++;
    }
    if (mySidInfo->formatString && strlen(mySidInfo->formatString))
    {
        fieldname[x] = "Format";
        fielddata[x] = mySidInfo->formatString;
    }
    else 
    {
        fieldname[x] = "Format";
        fielddata[x] = "SID";
    }
    x++;
    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;
    plugin.nfields = x;
}

static int init_data(char *name, void *data, int size)
{
    // Initialize the SID-Emulator Engine to defaults.
    init_engine();
    SidState *s = &state;
    if(s->tune)
        delete s->tune;
    s->tune = NULL;

    sidTune *myTune = new sidTune((const ubyte *)data, size);
    if (!myTune)  
        return -1;

    if(!myTune->getStatus())
    {
        delete myTune;
        return -1;
    }

    struct sidTuneInfo mySidInfo;
    myTune->getInfo(mySidInfo);

    write_fields(name, &mySidInfo);

    plugin.subtunes = mySidInfo.songs;
    plugin.tune = mySidInfo.startSong - 1;

    sidlenptr = get_song_length((unsigned char *)data, size);

    set_len();

    s->song = mySidInfo.startSong;
    s->tune = myTune;

    if(!s->tune)
        return -2;

    if(!sidEmuInitializeSong(*(s->engine), *(s->tune), s->song))
        return -3;

    s->engine->resetSecondsThisSong();
    s->playing = true;

    total = 0;

    return 0;
}

static int init_file(char *fname)
{
    // Initialize the SID-Emulator Engine to defaults.
    init_engine();
    SidState *s = &state;
    if(fname)
    {
        if(s->tune)
            delete s->tune;
        s->tune = NULL;

        sidTune *myTune = new sidTune(fname);
        if (!myTune)  
            return -1;

        struct sidTuneInfo mySidInfo;
        myTune->getInfo(mySidInfo);

        write_fields(fname, &mySidInfo);

        plugin.subtunes  = mySidInfo.songs;
        plugin.tune      = mySidInfo.startSong - 1;

        s->song = mySidInfo.startSong;
        s->tune = myTune;
    }

    if(!s->tune)
        return -2;

    if(!sidEmuInitializeSong(*(s->engine), *(s->tune), s->song))
        return -3;

    s->engine->resetSecondsThisSong();
    s->playing = true;

    total = 0;

    return 0;
}

static int fill_buffer(signed short *dest, int len)
{
    if(state.playing)
    {
#ifdef DOUBLE_MONO_SID
        signed short tmp[len>>2];
        sidEmuFillBuffer(*(state.engine), *(state.tune), tmp, len>>1);
        for (int i=0; i < len>>1; i++)
        {
            *dest++ = tmp[i];
            *dest++ = tmp[i];
        }
#else
        sidEmuFillBuffer(*(state.engine), *(state.tune), dest, len);
#endif
        //total += len;
        //if(total > 44100*5)
        //	return 0;
        return len;
    }
    return 0;
}

static int set_position(int msecs, int subtune)
{
    if (!msecs) {
        total = 0;
        state.song = subtune+1;
        init_file(NULL);
        plugin.tune = subtune;
        set_len();
        return subtune;
    }
    return 0;
}

static int is_ext(const char *s, const char *ext)
{
    const char *sext = strrchr(s, '.');
    if(!sext)
        sext = "";
    return (strcmp(ext, sext) == 0);
}

static int can_handle(const char *name)
{
    return (is_ext(name, ".sid"));
}

extern "C" {

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN sidplay_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
    memset(&plugin, 0, sizeof(plugin));

    plugin.plugname   = "sidplay";
    plugin.freq       = 44100;
#ifdef DOUBLE_MONO_SID
    plugin.channels   = 2;
#else
    plugin.channels   = 1;
#endif
#ifdef A320
    plugin.clockfreq  = 384;
#else
    plugin.clockfreq  = 100;
#endif
    plugin.replaygain = 1;
    //plugin.init_file   = init_file;
    plugin.init_data    = init_data;
    plugin.fill_buffer  = fill_buffer;
    plugin.set_position = set_position;
    plugin.can_handle   = can_handle;

    FILE *fp = fopen("Songlengths.txt", "rb");
    if(!fp)
        fp = fopen("songlengths.txt", "rb");
    if(fp)
    {
        char line[2048];
        char *rc = fgets(line, sizeof(line), fp);
        if(strncmp(line, "[Database]", 10) == 0)
        {
            song_md5 = (unsigned int *)malloc(sizeof(int) * 32000 * 5);
            song_len = (unsigned char *)malloc(2 * 32000);
            unsigned int *csum = song_md5;
            unsigned char *lenptr = song_len;
            while(rc)
            {
                rc = fgets(line, sizeof(line), fp);
                rc = fgets(line, sizeof(line), fp);
                if(rc)
                {
                    string2val(line, csum);
                    char *min, *sec;
                    if(min = strchr(line, '='))
                    {
                        csum[4] = lenptr - song_len;
                        min++;
                        while(sec = strchr(min, ':'))
                        {
                            *sec++ = 0;
                            sec[2] = 0;
                            int len = atoi(min) * 60 + atoi(sec);
                            if(len > 254)
                            {
                                *lenptr++ = 0xFF;
                                *lenptr++ = len>>8;
                            }
                            *lenptr++ = len&0xff;
                            min = &sec[3];
                            while(*min && !isdigit(*min)) min++;
                        }
                        csum += 5;
                        total_songs++;
                    }
                }
            }
            lenptr = lenptr;
        }
        fclose(fp);
    }
    return &plugin;
}

}

