/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

extern "C" {
#include "api68/api68.h"
}

#include "../plugin.h"
#include "../util.h"

static struct sound_plugin plugin;
api68_t *sc68 = NULL;

static int playing;
static string fieldname[5];
static string fielddata[5];
static string filename;
static int set_position(int msecs, int subtune);

// SC68 ALLOC FUNCTIONS
void *myAlloc(unsigned int n)
{
	unsigned int *data = new unsigned int [n];
	return data;
}

void myFree(void * data)
{
	delete(data);
}


static int init_file(char *fname)
{
    int x = 0;
	api68_music_info_t info;

    playing = 0;
    plugin.tune = 0;
    plugin.subtunes = 1;

   	if (api68_verify_file(fname) < 0) goto error;
	if (api68_load_file(sc68, fname)) goto error;
	if (api68_play(sc68, 1)) goto error;

	plugin.freq = api68_sampling_rate(sc68, plugin.freq);

	if( !api68_music_info(sc68, &info, 0, 0) ) {
    	if( info.title ) {
            fieldname[x] = "Song";
            fielddata[x++] = string(info.title);
        }
    
    	if( info.composer ) {
            fieldname[x] = "Composer";
            fielddata[x++] = string(info.composer);
        }

        if( info.author ) {
            fieldname[x] = "Author";
            fielddata[x++] = string(info.author);
        }

		plugin.subtunes = info.tracks;
		plugin.length = info.time_ms ? info.time_ms : 200000;
	}
	else {
        char *tmp = strrchr(fname, '/');
        fieldname[x] = "Song";
        fielddata[x++] = tmp ? string(tmp+1) : string(fname);
        
		plugin.subtunes = 1;
		plugin.length = 120000;
	}

    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;
    plugin.nfields  = x;
    
    playing = 1;

    return 0;

error:
	api68_shutdown(sc68);
	return -1;
}

static int close()
{
    api68_close(sc68);

    for(int i = 0; i < plugin.nfields; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }

    playing = plugin.tune = plugin.subtunes = plugin.length = plugin.nfields = 0;
    return 0;
}

static int fill_buffer(signed short *dest, int len)
{
	if(playing)
    {
		unsigned int code = api68_process(sc68, dest, len >> 2);

		if (code == API68_MIX_ERROR) {
			playing = 0;
            return 0;
		}

		if ( (code & API68_END) ) {
			playing = 0;
            return 0;
		}

		return len;
    }

    return 0;
}

static int set_position(int msecs, int subtune)
{
    if (!playing)
        return 0;

    if( !msecs ) {
       	api68_stop(sc68);
    	api68_play(sc68, subtune);

    	api68_music_info_t info;
    	if( !api68_music_info(sc68, &info, subtune, 0) ) {                
            int x = 0;
        	if( info.title ) {
                fieldname[x] = "Song";
                fielddata[x++] = string(info.title);
            }
        
        	if( info.composer ) {
                fieldname[x] = "Composer";
                fielddata[x++] = string(info.composer);
            }
    
            if( info.author ) {
                fieldname[x] = "Author";
                fielddata[x++] = string(info.author);
            }

            plugin.fieldname = fieldname;
            plugin.fielddata = fielddata;
            plugin.nfields  = x;

    		plugin.subtunes = info.tracks;
    		plugin.length = info.time_ms ? info.time_ms : 200000;
    	}
    	else {
    		plugin.subtunes = 1;
    		plugin.length = 120000;
    	}
    }
    else {
   		return api68_seek(sc68, msecs);
    }

    return 0;
}


static int can_handle(const char *name)
{
    return is_ext(name, ".sc68");
}

extern "C" {

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN sc68_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
	api68_init_t init68;
	memset(&init68, 0, sizeof(init68));
	init68.alloc = myAlloc;
	init68.free = myFree;
	init68.user_path = "./";
	sc68 = api68_init(&init68);
    if( !sc68 ) {
        fprintf(stderr, "sc68plugin - couldn't init sc68 api :(.\n");
        exit(-1);
    }

    memset(&plugin, 0, sizeof(plugin));
    plugin.plugname = "SC68";
    plugin.init_file        = init_file;
    //plugin.init_data      = init_data;
    //plugin.request_format = NULL;
    plugin.set_position     = set_position;
    plugin.fill_buffer      = fill_buffer;
    plugin.can_handle       = can_handle;
    plugin.close            = close;
    plugin.channels   = 2;
    plugin.freq       = 44100;
    plugin.replaygain = 1;

#ifdef A320
    plugin.clockfreq = 336; 
#else
    plugin.clockfreq = 200; 
#endif

    return &plugin;
}

}

