/*  
    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include "pokecubed/pokecubed.h"
#include "../plugin.h"
#include "../util.h"

static struct sound_plugin plugin;
static pokecubed *cubeplayer;

static string fieldname[5];
static string fielddata[5];

static int init_file(char *fname)
{
    cubeplayer = new pokecubed();
    cubeplayer->init();
    if (!cubeplayer->loadFile(fname))
    {
        fprintf(stderr,"pokecubeplugin:init_file - Couldn't load file");
        return -1;
    }

    plugin.length = cubeplayer->getLength();
    if (plugin.length <= 0)
        plugin.length = 150000;
    plugin.subtunes = 1;
    plugin.tune = 0;
    plugin.channels = cubeplayer->getNumberOfChannels();
    plugin.freq = cubeplayer->getFrequency();

    int x = 0;
    fieldname[x] = "File";
    fielddata[x++] = strrchr(fname,'/')+1;
    fieldname[x] = "Channels";
    fielddata[x++] = plugin.channels == 1 ? "Mono" : "Stereo";
    char tmp[20];
    sprintf(tmp,"%d.%d KHz",plugin.freq/1000,(plugin.freq%1000)/100);
    fieldname[x] = "Frequency";
    fielddata[x++] = tmp;
    fieldname[x] = "Format";
    fielddata[x++] = "Game Cube";
    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;
    plugin.nfields = x;
    if (is_ext(fname,".ast"))
        plugin.clockfreq = 250;
    return 0;
}

static int close()
{
    delete cubeplayer;
    for(int i = 0; i < 5; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }
    plugin.tune = plugin.subtunes = plugin.length = plugin.nfields = 0;
    return 0;
}

static int fill_buffer(signed short *dest, int len)
{
    int samp_per_chn = cubeplayer->getNSamples(dest,len/(plugin.channels<<1));
    return samp_per_chn*plugin.channels<<1;
}

static int set_position(int msecs, int subtune)
{
    if (msecs <= 0)
        return 0;
    else
        return cubeplayer->seek((unsigned int)msecs/1000);
}

static int can_handle(const char *name)
{
    return (is_ext(name, ".adx") || is_ext(name,".adp") 
         || is_ext(name, ".dsp") || is_ext(name,".gcm")
         || is_ext(name, ".afc") || is_ext(name,".ast")
         || is_ext(name, ".hps") || is_ext(name,".idsp")
         || is_ext(name, ".spt") || is_ext(name,".spd")
         || is_ext(name, ".mss") || is_ext(name,".mpdsp")
         || is_ext(name, ".ish") || is_ext(name,".ymf")
         || is_ext(name, ".rsf") || is_ext(name,".vag")
         || is_ext(name, ".rsd") || is_ext(name,".rsp"));
}

extern "C" {

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN pokecubed_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
    memset(&plugin, 0, sizeof(plugin));
    plugin.plugname = "pokecubed";
    //plugin.init_data = init_data;
    plugin.init_file = init_file;
    //plugin.request_format = NULL;
    plugin.set_position = set_position;
    plugin.fill_buffer = fill_buffer;
    plugin.can_handle = can_handle;
    plugin.close = close;
    plugin.clockfreq = 200;
    plugin.replaygain = 1;
    return &plugin;
}

}

