/* These are routines for handling the screen, either for a specific hardware (PSP, NDS, GBA etc)
 * or for a general framebuffer. The screen is always handled like 8bit paletted.
 * Define either GBA, DS, PSP, FB8, FB16 or FB32 before compiling
 * Then call screen_init() to set everything up to default
 * Then call screen_set_target() if you have a general framebuffer or need to write offscreen
 * Then call screen_setcolors() to set the current palette. For DIRECT modes this will change the
 * single palette. For 16 and 32 bit framebuffers it will just change the translation table
 */
#include "pogo/stdcore.h"

#ifdef DS
  #include <ds_defs.h>
#elif (defined GBA)
  #include <gba_defs.h>
#elif (defined PSP)
  #include <psp.h>
#endif

// Current location and size of framebuffer
static int g_screenWidth;
static int g_screenHeight;
static void *g_screenPtr;

#ifdef GBA

typedef uchar Pixel;
#define DIRECT
#define colors ((uint16*)0x05000000) 

#else

#undef DIRECT

#if (defined FB8)
  typedef uchar Pixel;
  #define DIRECT
#elif (defined FB16)
  typedef uint16 Pixel;
#elif (defined FB32)
  typedef uint32 Pixel;
#else
  // Default
  typedef uint16 Pixel;
#endif

#endif

#define SCREEN_WIDTH g_screenWidth
#define SCREEN_HEIGHT g_screenHeight
#define SCREEN_PTR g_screenPtr


#define MAX_COLOR 256


#if (defined PSP) || (defined GBA)
  #define UINT2PIXEL(x) ((x>>19)&0x1F) | (((x>>11)&0x1F)<<5) | (((x>>3)&0x1F)<<10)
#else
  #if (defined FB32)
  #define UINT2PIXEL(x) (x)
  #else
  #define UINT2PIXEL(x) (((x>>19)&0x1F)<<11) | (((x>>10)&0x3F)<<5) | ((x>>3)&0x1F)
  #endif
#endif

#ifdef DIRECT
  #define COPY_PIXEL(dst, src) *(dst) = *(src)
#else
  static Pixel colors[MAX_COLOR];
  #define COPY_PIXEL(dst, src) *(dst) = colors[*(src)]
#endif

#ifdef PSP
  #define OFFSET 32
#else
  #define OFFSET 0
#endif


#undef SCREEN_ROTATION
#define SCREEN_ROTATION 0
#include "screen_rot.h"
#undef SCREEN_ROTATION
#define SCREEN_ROTATION 1
#include "screen_rot.h"
#undef SCREEN_ROTATION
#define SCREEN_ROTATION 2
#include "screen_rot.h"
#undef SCREEN_ROTATION
#define SCREEN_ROTATION 3
#include "screen_rot.h"


int screen_setcolors(unsigned int *pal, int start, int size)
{
	int i;
	unsigned int col;

	if((start+size) > MAX_COLOR)
		size = (MAX_COLOR - start);
	
	for(i=0; i<size; i++)
	{
		col = pal[i];
		colors[i+start] = UINT2PIXEL(col);
	}
	
	return size;
}

void (*screen_copy)(int x, int y, uchar *src, int width, int height, int sw, int solid);
void (*screen_set)(int x, int y, int width, int height, int col);

#ifdef DS

void screen_init()
{
	*POWER_CR = POWER_ALL_2D | 0x8000;
	*DISPCNT = 0x00020000; // FRAMEBUF, BANK_A
	*BANK_A_CR = 0x80; // Map LCDC to 0x06800000, bank A
	g_screenPtr = 0x06000000;
	g_screenWidth = 256;
	g_screenHeight = 192;
	
	screen_set = screen_set_tl;
	screen_copy = screen_copy_tl;
	
}

#elif (defined GBA)

void screen_init()
{
	SETW(REG_DISPCNT, DISP_LCDC_OFF);	
	SETW(REG_DISPCNT, DISP_MODE_4 | DISP_BG2_ON | DISP_OBJ_ON | DISP_OBJ_CHAR_1D_MAP);
	SETW(REG_BG2CNT, 0);
 
	g_screenPtr = (void *)0x06000000;
	g_screenWidth = 240;
	g_screenHeight = 160;
	
	screen_set = screen_set_tl;
	screen_copy = screen_copy_tl;
	
}


#elif (defined PSP)

void screen_init()
{
	int i =512*272;
	uint16 *ptr;
	sceDisplaySetMode(0, 480, 272);
	sceDisplaySetFrameBuf((void *)0x04000000, 512, 1, 1 );
	
	screen_set = screen_set_tl;
	screen_copy = screen_copy_tl;

	g_screenPtr = (void *)0x44000000;
	
	ptr = g_screenPtr;
	while(i--)
		*ptr++ = 0x01f;

	g_screenWidth = 512;
	g_screenHeight = 272;
}

#else

void screen_init()
{
	int i;
	uint32 color = 0;

	for(i=0; i<256; i++)
	{
		screen_setcolors(&color, i, 1);
		color += 0x00010101;
	}

	g_screenPtr = NULL;
	g_screenWidth = g_screenHeight = 0;
}

#endif

void screen_set_target(void *ptr, int width, int height, int rotation)
{
	if(ptr)
		g_screenPtr = ptr;
	if(width)
		g_screenWidth = width;
	if(height)
		g_screenHeight = height;
	
	switch(rotation)
	{
	case 0:
		screen_set = screen_set_tl;
		screen_copy = screen_copy_tl;
		break;
	case 1:
		screen_set = screen_set_tr;
		screen_copy = screen_copy_tr;
		break;
	case 2:
		screen_set = screen_set_br;
		screen_copy = screen_copy_br;
		break;
	case 3:
		screen_set = screen_set_bl;
		screen_copy = screen_copy_bl;
		break;
	}
	
}
