/*
 * Core device handling functions (font.c)
 * Part of libpogo, a c-library replacent for GBA
 * Programmed by Jonas Minnberg (Sasq)
 *
 * DESCRIPTION
 * Functions for handling devices - allowing users to
 * register new devices that gets called through the
 * open(),close(),read() etc functions defined here.
 *
 **/

#include "pogo/stdcore.h"
#include "pogo/device.h"

#if (defined DS)
  #include <ds_defs.h>
#elif (defined GBA)
  #include <gba_defs.h>
#endif

typedef struct {
	uint32 fd;
	PogoDevice *dev;
} FdMap;

typedef struct {
	PogoDevice *device;
	char *name;
	void (*irqfunc)(void);
} RegDevice;

#define MAX_DEVICE 8
#define MAX_FD 32

static int dev_count = 0;
static RegDevice deviceList[MAX_DEVICE];

static int fdmap_count;
static FdMap fdList[MAX_FD];

static int default_dev = 0;


void device_doirq(void)
{
	int i;
	for(i=0; i<dev_count; i++)
		if(deviceList[i].irqfunc)
				deviceList[i].irqfunc();
}

vuint32 frame_counter;

#ifdef DS
void InterruptProcess(void)
{
	uint32 i = *IF;

	if(i & IRQ_VBLANK)
		frame_counter++;
	device_doirq();
	*IF = *IF;
}
#elif (defined GBA)
void InterruptProcess(void)
{
	int i = GETW(REG_IF);
	if(i & V_COUNT_INTR_FLAG)
		frame_counter++;
	device_doirq();
	SETW(REG_IF, 0xFFFF);
}
#endif

void pogo_device_init(void)
{
#ifdef DS
	*IME = 0;
	*DISPSTAT = 0x08;
	*Arm9IrqPointer = (uint32)InterruptProcess;
	*IE = IRQ_VBLANK;
	*IF = ~0;
	*IME = 1;
#endif

	fdList[0].fd = -1;
	fdList[0].dev = NULL;
	fdList[1].fd = -1;
	fdList[1].dev = NULL;
	fdList[2].fd = -1;
	fdList[2].dev = NULL;
	fdmap_count = 0;
}

static PogoDevice *dev_fromname(const char *name, char **cutname)
{
	int i,l;
	char *cn = *cutname;
	for(i=0; i<dev_count; i++) {
		l = strlen(deviceList[i].name);
		if(strncmp(name, deviceList[i].name, l) == 0)
		{
			cn = (char *)&name[l];
			while(*cn == '/')
				cn++;
			*cutname = cn;
			return deviceList[i].device;
		}
	}
	*cutname = (char *)name;
	return deviceList[0].device;
}

static PogoDevice *dev_fromhandle(int *fd)
{
	PogoDevice *dev = fdList[*fd].dev;
	*fd = fdList[*fd].fd;
	return dev;
}

int pogo_device_register(PogoDevice *dev, char *name, void (*irqfunc)(void), int fd)
{
	RegDevice *d = &deviceList[dev_count++];
	d->name = name;
	d->irqfunc = irqfunc;
	d->device = dev;

	/*if(fd > -1 && fd < 3)
	{
		fdList[fd].dev = dev;
		fdList[fd].fd = 0;
	}
	else */
		default_dev = dev_count-1;

	return dev_count-1;
}

int pogo_open(const char *name, int flags)
{
	int fd;
	PogoDevice *dev;
	char *cutname;
	
	if(!name)
	{
		fdList[fdmap_count].dev = NULL;
		fdList[fdmap_count].fd = 0;
		return fdmap_count++;
	}

	if((dev = dev_fromname(name, &cutname)))
		if(dev->open)
		{
			fd = dev->open(cutname, flags);
			if(fd < 0)
				return fd;
			fdList[fdmap_count].dev = dev;
			fdList[fdmap_count].fd = fd;
			return fdmap_count++;
		}
	return -1;
}

int pogo_dopen(const char *name)
{
	int fd;
	PogoDevice *dev;
	char *cutname;
	
	if((dev = dev_fromname(name, &cutname)))
	{
		if(dev->dopen)
		{
			fd = dev->dopen(cutname);
			if(fd < 0)
				return fd;
			fdList[fdmap_count].dev = dev;
			fdList[fdmap_count].fd = fd;
			return fdmap_count++;
		}
		else
		if(dev->open)
			return pogo_open(name, O_RDONLY);
	}
	return -1;
}

int pogo_read(int fd, void *dest, int size)
{
	PogoDevice *dev;
	if((dev = dev_fromhandle(&fd)))
		if(dev->read)
			return dev->read(fd, dest, size);
	return -1;
}

int pogo_dread(int fd, void *dest, int size)
{
	PogoDevice *dev;
	if((dev = dev_fromhandle(&fd)))
	{
		if(dev->dread)
			return dev->dread(fd, dest, size);
		else
		if(dev->read)
			return dev->read(fd, dest, size);
	}
		
	return -1;
}

int pogo_write(int fd, const void *dest, int size)
{
	PogoDevice *dev;
	if((dev = dev_fromhandle(&fd)))
		if(dev->write)
			return dev->write(fd, dest, size);
	return -1;
}

int pogo_close(int fd)
{
	FdMap *f;
	int oldfd = fd;
	PogoDevice *dev;
	if((dev = dev_fromhandle(&fd)))
	{
		if(dev->close)
		{
			fdList[oldfd].dev = NULL;
			f = &fdList[fdmap_count-1];
			while(fdmap_count && f->dev == NULL) {
				fdmap_count--;
				f = &fdList[fdmap_count-1];
			}

			dev->close(fd);
			return 0;
		}
	}

	return -1;
}

int pogo_dclose(int fd)
{
	FdMap *f;
	int oldfd = fd;
	PogoDevice *dev;
	if((dev = dev_fromhandle(&fd)))
	{
		if(dev->dclose)
		{
			fdList[oldfd].dev = NULL;
			f = &fdList[fdmap_count-1];
			while(fdmap_count && f->dev == NULL)
			{
				fdmap_count--;
				f = &fdList[fdmap_count-1];
			}
			dev->dclose(fd);
			return 0;
		}
		else
		if(dev->close)
		{
			dev->close(fd);
			return 0;
		}	
	}
	return -1;
}


int pogo_lseek(int fd, int offset, int origin)
{
	PogoDevice *dev;
	if((dev = dev_fromhandle(&fd)))
		if(dev->lseek)
			return dev->lseek(fd, offset, origin);
	return -1;
}

int pogo_ioctl(int fd, int request, ...)
{
	int rc = -1;
	PogoDevice *dev;
	va_list vl;
	va_start(vl, request);

	if((dev = dev_fromhandle(&fd)))
		if(dev->ioctl)
			rc = dev->ioctl(fd, request, vl);
	va_end(vl);
	return rc;
}

int pogo_stat(const char *name, struct pogo_stat *buffer)
{
	PogoDevice *dev;
	char *cutname;
	if((dev = dev_fromname(name, &cutname)))
	{
		if(dev->stat)
			return dev->stat(cutname, buffer);
	}
	return -1;
}



int pogo_remove(const char *name)
{
	PogoDevice *dev;
	char *cutname;
	if((dev = dev_fromname(name, &cutname)))
		if(dev->remove)
			return dev->remove(cutname);
	return -1;
}

int pogo_tell(int fd)
{
	return pogo_lseek(fd, 0, SEEK_CUR);
}
