/*
 * Text console device (console.c)
 * Part of libpogo, a c-library replacent for GBA
 * Programmed by Jonas Minnberg (Sasq)
 *
 * DESCRIPTION
 * Device for text output. Supports fixed and proportional fonts,
 * multiple windows etc.
 *
 **/

#include "pogo/stdcore.h"
#include "pogo/device.h"
#include "pogo/font.h"
#include "pogo/screen.h"
#include "pogo/console.h"

//#define TO_RGB16(r,g,b) ( ((b << 7) & 0x7C00) | ((g << 2) & 0x03E0) | (r >> 3) )
//#define TO_RGB16(r,g,b) ( ((b << 8) & 0xF800) | ((g << 3) & 0x07E0) | (r >> 3) )
#define TO_RGB16(r,g,b) ( ((r << 8) & 0xF800) | ((g << 3) & 0x07E0) | (b >> 3) )

#define TO_RGB32(r,g,b) ((r<<16) | (g<<8) | b)

#define FLUSH() console_putchar(0);

const static uint16 ansi_colors[16*3] =
{ 
	TO_RGB16(0x00,0x00,0x00),
	TO_RGB16(0x80,0x00,0x00),
	TO_RGB16(0x00,0x80,0x00),
	TO_RGB16(0x80,0x80,0x00),
	TO_RGB16(0x00,0x00,0x80),
	TO_RGB16(0x80,0x00,0x80),
	TO_RGB16(0x00,0x80,0x80),
	TO_RGB16(0x80,0x80,0x80),

	TO_RGB16(0xC0,0xC0,0xC0),
	TO_RGB16(0xFF,0x00,0x00),
	TO_RGB16(0x00,0xFF,0x00),
	TO_RGB16(0xFF,0xFF,0x00),
	TO_RGB16(0x00,0x00,0xFF),
	TO_RGB16(0xFF,0x00,0xFF),
	TO_RGB16(0x00,0xFF,0xFF),
	TO_RGB16(0xFF,0xFF,0xFF),
};

const static uint32 ansi_colors32[16*3] =
{ 
	TO_RGB32(0x00,0x00,0x00),
	TO_RGB32(0x80,0x00,0x00),
	TO_RGB32(0x00,0x80,0x00),
	TO_RGB32(0x80,0x80,0x00),
	TO_RGB32(0x00,0x00,0x80),
	TO_RGB32(0x80,0x00,0x80),
	TO_RGB32(0x00,0x80,0x80),
	TO_RGB32(0x80,0x80,0x80),

	TO_RGB32(0xC0,0xC0,0xC0),
	TO_RGB32(0xFF,0x00,0x00),
	TO_RGB32(0x00,0xFF,0x00),
	TO_RGB32(0xFF,0xFF,0x00),
	TO_RGB32(0x00,0x00,0xFF),
	TO_RGB32(0xFF,0x00,0xFF),
	TO_RGB32(0x00,0xFF,0xFF),
	TO_RGB32(0xFF,0xFF,0xFF),
};


void (*more_callback)(void)  = NULL;


enum { FONT_NORMAL, FONT_ITALIC, FONT_BOLD };

typedef struct 
{
	Font *fonts[4];		// Fonts for this window
	uint16 *pixels;		// Top left of pixels

	uint16 width;		// Width of window in pixels
	uint16 height;		// Height of window in pixels
	uint16 stride;		// Width of screen
	uint16 current_font;

	char *buffer;		// Contents of window
	
	char *startptr;		// Location of char at top left corner
	
	uint32 *offsets;	// screen offset for every char
	uint16 pos;			// Last character (where to append in buffered window)
	
	uint16 columns;		// Width of one grid line or entire buffer for stream buffers
	uint16 lines;		// Ignored for stream buffers (considered 1)
	
	uint16 xstep;
	uint16 lineh;		// Height of a single textline, usually font_height or height / lines
	
	uint16 type;		// Fixed grid or stream buffer
	int flags;			// Flags like LINEWRAP, AUTOSCROLL
	int x,y;			// Current cursor position
	
	int fgcol;
	int bgcol;

	int mark_start;
	int mark_stop;

	char *wordbuf;
	int wordpos;
	int wordlen;

	int linesout;		// Lines printed (used for output pausing)
	int nextmore;

	int visible;
	
} Window;

#define MAXWIN 16
static Window *windows[MAXWIN];
static Window *awin;	// Active window

#ifdef LIBPOGO
static Device condev;
#endif

#if (defined PSP)
void *consolePixels = (void *)0x44000000;
int consoleStride = 512;
#elif (defined DS)
void *consolePixels = (void *)0x06800000;
int consoleStride = 256;
#elif (defined GBA)
void *consolePixels = (void *)0x06000000;
int consoleStride = 240;
#else
void *consolePixels;
int consoleStride;
#endif

void console_set_target(void *pixels, int stride)
{
	consolePixels = pixels;
	consoleStride = stride;
}

static int console_mark_internal(int start, int stop)
{
	int x0, y0;

	if(awin->type == CONWIN_STREAM)
	{
		 x0 = awin->offsets[start] & 0xffff;
		 y0 = awin->offsets[start] >> 16;
		//int x1 = awin->offsets[stop] & 0xffff;
		//int y1 = awin->offsets[stop] >> 16;	
	}
	else
	{
		 x0 = start % awin->columns;
		 y0 = start / awin->columns;
	}
		
	if(start == stop)
	{
		int h = awin->lineh;
		uint16 *pixels =  &awin->pixels[x0 * awin->xstep + y0 * awin->lineh * awin->stride];
		
		while(h--)
		{
			pixels[0] = pixels[0] ^ 0xFFFF;
			pixels[1] = pixels[1] ^ 0xFFFF;
			pixels += awin->stride;
		}
	}
	
	awin->mark_start = start;
	awin->mark_stop = stop;
	
	return 0;

}

static int console_mark(int start, int stop)
{
	if(awin->mark_start >= 0)
		console_mark_internal(awin->mark_start, awin->mark_stop);
	if(start >= 0)
		console_mark_internal(start, stop);

	return 0;
}


static void console_clear(void)
{
	uint16 *p = awin->pixels;
	uint16 col = ansi_colors[awin->bgcol];
	int h = awin->height;
	
	while(h--)
	{
		int i = awin->width;
		while(i--)
			*p++ = col;
		p += (awin->stride - awin->width);
	}
	
	if(awin->type == CONWIN_FIXED)
	{
		int i;
		for(i=0; i<(awin->columns * awin->lines); i++)
			awin->buffer[i] = ' ';
	}
	else
		*(awin->buffer) = 0;

	awin->pos = 0;
	awin->x = awin->y = 0;
}

static void console_gotoxy(int x, int y)
{
	awin->x = x;
	awin->y = y;
}

static void console_setcolor(int fg, int bg)
{
	awin->fgcol = fg;
	awin->bgcol = bg;
}


static void console_scroll(int y)
{
	int w,h;
	int modulo = awin->stride - awin->width;
	uint16 bgcol = ansi_colors[awin->bgcol];
	uint16 *dest = awin->pixels;
	uint16 *src = &awin->pixels[awin->stride * y * awin->lineh];

	if(awin->mark_start >= 0)
		console_mark_internal(awin->mark_start, awin->mark_stop);
	awin->mark_start = -1;
	
	h = awin->height - (y * awin->lineh);
	
	while(h--)
	{
		w = awin->width;
		while(w--)
			*dest++ = *src++;
		dest += modulo;
		src += modulo;
	}
	
	h = y * awin->lineh;
	while(h--)
	{
		w = awin->width;
		while(w--)
			*dest++ = bgcol;
		dest += modulo;
	}

	if(awin->type == CONWIN_STREAM)
	{
		int i = 0;
		while((int)(awin->offsets[i] >> 16) < y)
			i++;

		awin->pos -= i;

		memmove(awin->buffer, &awin->buffer[i], awin->pos);
		memmove(awin->offsets, &awin->offsets[i], (awin->pos+1) * sizeof(uint32));
	
		for(i=0; i<awin->pos; i++)
			awin->offsets[i] = (awin->offsets[i] - (y<<16));
	}
	else
	{
		memmove(awin->buffer, &awin->buffer[y * awin->columns], awin->columns*(awin->lines-y));
		memset(&awin->buffer[awin->columns*(awin->lines-y)], ' ', awin->columns * y);
	}
	
	awin->y -= y;

}



static void console_erase(int n)
{
	int clearw,w,h;
	uint16 *ptr;

	uint16 bgcol = ansi_colors[awin->bgcol];
	int oldy = awin->y;
	
	if(!n)
		return;

	if(awin->mark_start >= 0)
		console_mark_internal(awin->mark_start, awin->mark_stop);
	awin->mark_start = -1;

	if(awin->type == CONWIN_STREAM)
	{
		if(n > awin->pos)
			awin->pos = 0;
		else
			awin->pos -= n;
	
		awin->x = awin->offsets[awin->pos] & 0xff;	
		awin->y = awin->offsets[awin->pos] >> 16;
		
		ptr = &awin->pixels[awin->x * awin->xstep + awin->y * awin->lineh * awin->stride];		
		h = awin->lineh;
		clearw = awin->width - awin->x;
		
	}
	else
	{
		while(n--)
		{
			awin->x--;
			if(awin->x < 0)
			{
				awin->x = awin->columns - 1;
				awin->y--;
			}
		}

		ptr = &awin->pixels[awin->x * awin->xstep + awin->y * awin->lineh * awin->stride];		
		h = awin->lineh;
		clearw = awin->width - (awin->x * awin->xstep);
	}


	// Clear area

	while(h--)
	{
		w = clearw;
		while(w--)
			*ptr++ = bgcol;
		ptr += (awin->stride - clearw);
	}
	
	ptr -= (awin->width - clearw);
	
	h = (oldy - awin->y) * awin->lineh;

	while(h--)
	{
		w = awin->width;
		while(w--)
			*ptr++ = bgcol;
		ptr += (awin->stride - awin->width);
	}

}

static int console_getcharcount()
{
	if(awin->type == CONWIN_FIXED)
		return awin->x + awin->y * awin->columns;
	else
		return awin->pos;
}


static void console_putchar(unsigned char c)
{
	int w = 0;

	Font *font = awin->fonts[awin->current_font];

	// If there are chars to print we remove mark first
	if(awin->wordpos || c)
	{
		if(awin->mark_start >= 0)
			console_mark_internal(awin->mark_start, awin->mark_stop);
		awin->mark_start = -1;
	}

	if(c == 12)
	{
		console_clear();
		return;
	}

	font_settarget(awin->pixels, awin->stride, 192, 0);

	if(awin->type == CONWIN_FIXED)
	{
		switch(c)
		{
		case 13:
		case 10:
			awin->x = 0;
			awin->y++;
			awin->linesout++;
			break;
		case 9:
			break;
		default:				
			break;
		}

		if((awin->x + 1) > awin->columns)
		{
			if(0)//awin->flags & CM_LINEWRAP)
			{
				awin->x = 0;
				awin->y++;
				awin->linesout++;
			}
			else
				return;
		}

		while(awin->y >= awin->lines)
		{
			console_scroll(1);
		}
		
		if(c >= 0x20)
		{
			awin->buffer[awin->x + awin->y * awin->columns] = c;
			font_putchar(font, c, awin->x * awin->xstep, awin->y * awin->lineh);
			awin->x++;
		}
	}
	else
	{
		int i;
		//int wordwrap = 0;
		if(c && awin->flags & CM_WORDWRAP)
		{
			awin->wordbuf[awin->wordpos] = c;
			//awin->offsets[awin->endpos] = awin->x | (awin->y << 16);
			awin->wordpos++;
			w = font_putchar(font, c, -1, -1);
			awin->wordlen += w;

			if(c != ' ' && c != 13 && c != 10)
				return;

			if(awin->wordlen + awin->x > awin->width)
			{
				awin->x = 0;
				awin->y++;
				awin->linesout++;

			}
		}
		else
		if(c)
			awin->wordbuf[awin->wordpos++] = c;

		if(more_callback && (awin->linesout == awin->nextmore))
		{
			int w;
			awin->nextmore = awin->linesout + awin->lines - 1;
			while(awin->y >= awin->lines)
				console_scroll(1);
			w = font_text(awin->fonts[0], "[MORE]", 0, (awin->lines-1) * awin->lineh);
			more_callback();
			screen_set(0, awin->y * awin->lineh, w, awin->fonts[0]->height, awin->bgcol);
		}

		//while(awin->pos < awin->endpos)
		for(i=0; i<awin->wordpos; i++)
		{
			c = awin->wordbuf[i];
			awin->buffer[awin->pos] = c;
			awin->offsets[awin->pos] = awin->x | (awin->y << 16);
			awin->pos++;
			
			
			switch(c)
			{
			case 13:
			case 10:
				w = 0;
				awin->x = 0;
				awin->y++;
				awin->linesout++;
				break;
			case 9:
				w = 0;
				break;
			default:
				w = font_putchar(font, c, -1, -1);
				break;
			}

			awin->offsets[awin->pos] = (awin->x + w) | (awin->y << 16);

			if(w + awin->x > awin->width)
			{
				//if(awin->flags & CM_LINEWRAP)
				//{
					awin->x = 0;
					awin->y++;
					awin->linesout++;
				//}
				//else
				//	return;
			}

			while(awin->y >= awin->lines)
				console_scroll(1);

			if(w)
			{
				font_putchar(font, c, awin->x, awin->y * awin->lineh);
				awin->x += w;
			}
		}
		awin->wordpos = 0;
		awin->wordlen = 0;

		if(more_callback && (awin->linesout == awin->nextmore))
		{
			int w;
			awin->nextmore = awin->linesout + awin->lines - 1;
			w = font_text(awin->fonts[0], "[MORE]", 0, (awin->lines-1) * awin->lineh);
			more_callback();
			screen_set(0, awin->y * awin->lineh, w, awin->fonts[0]->height, awin->bgcol);
		}

	}

}

static void console_setfont(int num)
{
	//awin->fonts[num] = font;
	awin->current_font = num;
}

static void window_calc(Window *win)
{
	int i;
	int fw = 0;
	int fh = 0;

	for(i=0; i<4; i++)
	{
		if(win->fonts[i])
		{
			if(win->fonts[i]->charwidth > fw)
				fw = win->fonts[i]->charwidth;
			if(win->fonts[i]->height > fh)
				fh = win->fonts[i]->height;
		}
	}

	win->columns = win->width / fw;
	win->lines = win->height / fh;
	win->lineh = win->fonts[0]->height;

	if(win->type == CONWIN_FIXED)
		win->xstep = win->width / win->columns;
	else
		win->xstep = 1;
}

void console_window_close(int id)
{
	Window *save = awin;
	awin = windows[id];
	console_clear();
	free(awin);
	windows[id] = NULL;
	awin = save;
}

static int console_window_create(void *pixels, int w, int h, int stride, Font **fonts, int type)
{
	Window *win;

	int i;
	//int fw = 0;
	//int fh = 0;
	int id = -1;

	for(i=0; i<MAXWIN; i++)
		if(!windows[i])
		{
			id = i;
			break;
		}

	if(id < 0)
		return id;


	win = (Window *)malloc(sizeof(Window));

	win->width = w;
	win->height = h;
	win->stride = stride;

	for(i=0; i<4; i++)
		win->fonts[i] = fonts[i];
		
	win->current_font = 0;

	//win->startpos = win->buffer;
	//win->endpos = win->buffer;
	win->pos = 0;
	
	win->pixels = (uint16*)pixels;//(uint16 *)(0x06800000 + x*2 + 512 * y);

	win->x = 0;
	win->y = 0;

	win->type = type; //CONWIN_FIXED;
	win->flags = CM_LINEWRAP | CM_WORDWRAP;
	
	win->mark_start = -1;

	windows[id] = win;
	
	window_calc(win);

	win->linesout = 0;
	win->nextmore = win->lines - 1;

	win->buffer = (char *)malloc(win->columns * win->lines * 12);
	win->offsets = (uint32 *)malloc(win->columns * win->lines * sizeof(uint32) * 12);

	win->wordbuf = (char *)malloc(256);
	win->wordpos = win->wordlen = 0;
	awin = win;
	
	console_setcolor(15, 0);
	console_clear();

	return id;

}

int console_write(int fd, const void *dest, int size)
{
	int i;

	uchar *s = (uchar *)dest;

	if(!awin)
		return -1;

	font_setcolor(ansi_colors32[awin->fgcol], ansi_colors32[awin->bgcol]);
	for(i=0; i<size; i++)
		console_putchar(s[i]);

	return size;
}

int console_read(int fd, void *dest, int size)
{
	return 0;
}

int console_lseek(int fd, int offset, int origin)
{
	return -1;
}


static void console_move(int x, int y, int neww, int newh)
{
	font_setcolor(ansi_colors32[awin->fgcol], ansi_colors32[awin->bgcol]);
	
	//if(awin->mark_start >= 0)
	//	console_mark_internal(awin->mark_start, awin->mark_stop);
	//awin->mark_start = -1;

	
	if(x >= 0 && y >= 0)
		awin->pixels = (uint16 *)((uchar *)consolePixels + x*2 + consoleStride * 2 * y);

	awin->x = awin->y = 0;
	awin->stride = consoleStride;
	
	if(awin->type == CONWIN_FIXED)
	{

	}
	else
	{
		int oldpos, oldnextmore, oldlinesout;
		int i;
		//char c;
		//int w = 0;
		//Font *font = awin->fonts[awin->current_font];
		font_settarget(awin->pixels, awin->stride, awin->height, 0);

		if(neww >=0)
			awin->width = neww;

		if(newh >=0)
			awin->height = newh;

		window_calc(awin);

		{
			uint16 *p = awin->pixels;
			uint16 col = ansi_colors[awin->bgcol];
			int h = awin->height;
			while(h--)
			{
				int i = awin->width;
				while(i--)
					*p++ = col;
				p += (awin->stride - awin->width);
			}			
		}

		oldpos = awin->pos;
		awin->pos = 0;
		
		//awin->linesout = 0;
		//awin->nextmore = awin->lines - 1;
		oldnextmore = awin->nextmore;
		oldlinesout = awin->linesout;
		awin->nextmore = awin->linesout + 1000;

		for(i=0; i<oldpos; i++)
		{
			console_putchar(awin->buffer[i]);
		}
		
		awin->nextmore = oldnextmore;
		awin->linesout = oldlinesout;
	}

	//if(awin->mark_start >= 0)
	//	console_mark_internal(awin->mark_start, awin->mark_stop);
}

void console_show(int show)
{
}


int console_open(const char *name, int mode)
{
	return 0;
}

int console_ioctl_v(int fd, int req, va_list vl)
{
	char *s;
	int c;

	if((!awin) && (req != CC_OPENWIN))
		return -1;

	switch(req) {
	case CC_GETTEXTWIDTH:
		s = va_arg(vl, char*);
		return font_text(awin->fonts[awin->current_font], s, -1, -1);
	case CC_GETCHARWIDTH:
		c = va_arg(vl, int);
		return font_putchar(awin->fonts[awin->current_font], c, -1, -1);
	case CC_SETFONT:
		console_setfont(va_arg(vl, int));
		return 0;
	case CC_GETFONT:
		return (int)awin->fonts[va_arg(vl, int)];
	case CC_GETX:
		return awin->x;
	case CC_GETY:
		return awin->y;
	case CC_LINESOUT:
		return awin->linesout;
		break;
	case CC_GOTOXY:
		{
			int x = va_arg(vl, int);
			int y = va_arg(vl, int);
			FLUSH();
			console_gotoxy(x, y);
		}
		return 1;
	case CC_MARK:
		{
			int x = va_arg(vl, int);
			int y = va_arg(vl, int);
			FLUSH();
			console_mark(x, y);
		}
		return 1;
	case CC_ERASE:
		FLUSH();
		console_erase(va_arg(vl, int));
		return 1;
	case CC_SETCOLOR:
		{
			int x = va_arg(vl, int);
			int y = va_arg(vl, int);
			FLUSH();
			console_setcolor(x, y);
		}
		return 1;
	case CC_GETXY:
	case CC_GETPOS:
		FLUSH();
		return console_getcharcount();
	case CC_CLEAR:
		FLUSH();
		console_clear();
		return 1;
	case CC_GETHEIGHT:
		return awin->lines;
	case CC_SCROLL:
		FLUSH();
		console_scroll(va_arg(vl, int));
		return 1;
	case CC_FLUSH:
		console_putchar(0);
		return 1;
	case CC_SETWIN:
		{
			int w = va_arg(vl, int);
			if(awin != windows[w])
			{
				FLUSH();
				awin = windows[w];
			}
		}
		return 1;
	case CC_OPENWIN:
		{
			int x = va_arg(vl, int);
			int y = va_arg(vl, int);
			int w = va_arg(vl, int);
			int h = va_arg(vl, int);
			Font **fonts = (va_arg(vl, Font**));
			int type = va_arg(vl, int);

			uint16 *pixels = (uint16 *)((uchar *)consolePixels + x*2 + consoleStride * 2 * y);
			return console_window_create(pixels, w, h, consoleStride, fonts, type);
		}
	case CC_CLOSEWIN:
		console_window_close(va_arg(vl, int));
		return 1;
	case CC_RESIZEWIN:
		{
			int w = va_arg(vl, int);
			int h = va_arg(vl, int);
			FLUSH();
			console_move(-1, -1, w, h);
		}		
		return 1;
	case CC_MOVEWIN:
		{
			int x = va_arg(vl, int);
			int y = va_arg(vl, int);
			int w = va_arg(vl, int);
			int h = va_arg(vl, int);
			FLUSH();
			console_move(x, y, w, h);
		}
		return 1;
	case CC_SETMORECALLBACK:
		more_callback = (void (*)(void))va_arg(vl, int);
		break;
	case CC_RESETMORE:
		awin->nextmore = awin->linesout + awin->lines - 1;
		break;
	}
	return -1;
}

int console_ioctl(int fd, int request, ...)
{
	int rc = -1;
	va_list vl;
	va_start(vl, request);
	rc = console_ioctl_v(fd, request, vl);
	va_end(vl);
	return rc;
}

static PogoDevice condev;

void console_init()
{
	int i;	
	for(i=0; i<MAXWIN; i++)
		windows[i] = NULL;

	more_callback = NULL;

	memset(&condev, 0, sizeof(condev));
	condev.open = console_open;
	condev.write = console_write;
	condev.ioctl = console_ioctl_v;
	condev.read = console_read;
	condev.lseek = console_lseek;

	pogo_device_register(&condev, "/dev/console", NULL, 1);

}

