/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h> 
#include <string>
#include <sys/stat.h>
#include <stdarg.h>

#include "mpg123/mpg123.h"

#include "../plugin.h"
#include "../util.h"


static struct sound_plugin plugin;

mpg123_handle *player = NULL;

static int playing = 0;
static int fsize;

signed short *tmpbuf = NULL;

static int readposbytes;
static size_t writeposbytes;
static int samples_done;
static bool quit_thread;

static string fieldname[5];
static string fielddata[5];

static int fill_buffer(signed short *dest, int len)
{
	if(playing)
	{
        int writtenbytes = 0;
        for (;;)
        {
            int bytesleft = len - writtenbytes;
            int bufbytes  = writeposbytes - readposbytes;
            //printf("len = %04d - written = %04d = %04d, " \
                    "read = %04d - write = %04d = %04d\n",\
                len,writtenbytes,bytesleft,readposbytes,writeposbytes,bufbytes);
            if (bytesleft <= bufbytes)
            {
                memcpy((char*)dest + writtenbytes, (char*)tmpbuf + readposbytes, bytesleft);
                readposbytes += bytesleft;
                writtenbytes = len;
                break;
            }
            else
            {
                memcpy((char*)dest + writtenbytes, (char*)tmpbuf + readposbytes, bufbytes);
                writtenbytes += bufbytes;
                readposbytes = writeposbytes = 0;
                
                /* TODO - find safest max buffer size here */
                int ret = mpg123_read(player, (unsigned char*)tmpbuf, 1024, &writeposbytes);
            	if (ret == MPG123_DONE) break;

                if (!writeposbytes)
                    break;
            }
        }
        samples_done += writtenbytes/(2*plugin.channels);
        return writtenbytes;
	}

	return 0;
}

static int close()
{
   	mpg123_close(player);
	mpg123_delete(player);

    if (tmpbuf) {
		free(tmpbuf);
	}

    for(int i = 0; i < 5; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }

    playing = plugin.tune = plugin.subtunes = plugin.length = plugin.nfields = 0;
	return 0;
}

static int init_file(char *fname)
{
	plugin.subtunes = 1;
	plugin.tune = 0;
	plugin.length = -1;

	fprintf(stderr, "mpg123plugin:init_file - Decoder Init\n");

    /* TODO - check inbuf size */
    tmpbuf = new signed short [1024*8];

	player = mpg123_new(NULL, NULL);
    if( !player ) return -1;

    int x = parse_id3v1(fname, fieldname, fielddata, 4);
    fieldname[x] = "Format";
    fielddata[x++] = is_ext(fname,".mp3") ? "MP3" : is_ext(fname,".mp2") ? "MP2" : "MP1";

    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;
    plugin.nfields   = x;

    if (mpg123_open(player, fname) != MPG123_OK) {
        mpg123_delete(player);
        return -2;
    }

    long rate;
    int encoding, channels;
    if (mpg123_getformat(player, &rate, &channels, &encoding) != MPG123_OK) {
        // we will have to wait till format ...
        size_t        dummy;
		unsigned char dumbuf[1024];
        int rc;
        while ( (rc = mpg123_read(player, dumbuf, 1024, &dummy)) != MPG123_NEW_FORMAT) {
            if( rc == MPG123_DONE || rc == MPG123_ERR || 
                rc > MPG123_OK ) {
                mpg123_close(player);
                mpg123_delete(player);
                return -3;
            }
        }
            
        mpg123_getformat(player, &rate, &channels, &encoding);
    }

    plugin.freq = rate;
    plugin.channels = channels;
    plugin.length = mpg123_length(player) / (plugin.freq/1000);

    if( tmpbuf == NULL ) {
        size_t size = mpg123_safe_buffer(); 
        /* fprintf(stderr, "mpg123plugin - Safe buffer size = %d\n", size); */
        tmpbuf = (signed short *)malloc(size * sizeof(signed short));
    }

    writeposbytes = 0;
    readposbytes  = 0;
    samples_done  = 0;
	playing = 1;
	return 0;
}

static int set_position(int msecs, int subtune)
{
	if (msecs)
    {
        struct mpg123_frameinfo mi;
        mpg123_info(player, &mi);
        int seeksamples = msecs * mi.rate / 1000;

        int cursamples = mpg123_tell(player);
        if (seeksamples < cursamples)
       		mpg123_seek(player, seeksamples, SEEK_SET);
		else
            mpg123_seek(player, seeksamples - cursamples, SEEK_CUR);
        return msecs;
    }
    return 0;
}

static int can_handle(const char *name)
{
	return (is_ext(name, ".mp3") || is_ext(name, ".mp2") || is_ext(name, ".mp1"));
}

extern "C" {
#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN mpg123_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
	if (mpg123_init() != MPG123_OK) {
    	fprintf(stderr, "mpg123plugin - Init failed.\n");
        exit(-1);
    }

	memset(&plugin, 0, sizeof(plugin));

	plugin.plugname     = "mpg123";
	plugin.init_file    = init_file;
	plugin.close        = close;
	//plugin.init_data  = init_data;
	plugin.fill_buffer  = fill_buffer;
	plugin.set_position = set_position;
	plugin.can_handle   = can_handle;
#ifdef A320
	plugin.clockfreq    = 300;
#else
	plugin.clockfreq    = 150;
#endif
    plugin.replaygain   = 1;

	return &plugin;
}
}

