/* 
    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
extern "C" {
#include "mpc/mpcdec.h"
#include "mpc/reader.h"
}
#include "../plugin.h"
#include "../util.h"

typedef struct reader_data_t {
    FILE      *file;
    long       size;
    mpc_bool_t seekable;
} reader_data;

static struct sound_plugin plugin;
static reader_data data;
static mpc_reader reader;
static mpc_demux* demux;
static mpc_streaminfo mpcinfo;

static FILE *file = NULL;

static int playing;
static MPC_SAMPLE_FORMAT *tmpbuf = NULL;
static int writepos;
static int readpos;
static int samples_done;
static int m_shift  = 16 - MPC_FIXED_POINT_SCALE_SHIFT;
static int clip_min = -1 << (16 - 1);
static int clip_max = (1 << (16 - 1)) - 1;
static string fieldname[5];
static string fielddata[5];

static mpc_int32_t read_impl(mpc_reader *p_reader, void *ptr, mpc_int32_t size)
{
    reader_data *d = (reader_data *) p_reader->data;
    return fread(ptr, 1, size, d->file);
}

static mpc_bool_t seek_impl(mpc_reader *p_reader, mpc_int32_t offset)
{
    reader_data *d = (reader_data *) p_reader->data;
    return d->seekable ? !fseek(d->file, offset, SEEK_SET) : false;
}

static mpc_int32_t tell_impl(mpc_reader *p_reader)
{
    reader_data *d = (reader_data *) p_reader->data;
    return ftell(d->file);
}

static mpc_int32_t get_size_impl(mpc_reader *p_reader)
{
    reader_data *d = (reader_data *) p_reader->data;
    return d->size;
}

static mpc_bool_t canseek_impl(mpc_reader *p_reader)
{
    reader_data *d = (reader_data *) p_reader->data;
    return d->seekable;
}

static int close();
static int init_file(char *fname)
{
    playing = 0;
    
    mpc_status err;
    if (!(file = fopen(fname, "r"))) return -1;

    /* initialize our reader_data tag the reader will carry around with it */
	data.file = file;
	data.seekable = true;
	fseek(data.file, 0, SEEK_END);
	data.size = ftell(data.file);
	fseek(data.file, 0, SEEK_SET);

	/* set up an mpc_reader linked to our function implementations */
	reader.read = read_impl;
	reader.seek = seek_impl;
	reader.tell = tell_impl;
	reader.get_size = get_size_impl;
	reader.canseek = canseek_impl;
	reader.data = &data;

    demux = mpc_demux_init(&reader);
    if(!demux) return -2;

    mpc_demux_get_info(demux,  &mpcinfo);

    int x = 0;
    fieldname[x] = "Encoder";
    fielddata[x++] = string(mpcinfo.encoder);

    fieldname[x] = "Format";
    fielddata[x++] = "MPC";

    plugin.nfields = x;
    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;

    if (!(plugin.freq     = mpcinfo.sample_freq))
        plugin.freq = 44100;
    if (!(plugin.channels = mpcinfo.channels))
        plugin.channels = 2;
    plugin.length   = mpc_streaminfo_get_length(&mpcinfo);

    printf("%d Hz, %d chn, %d msecs, %d buffsize\n",
        plugin.freq,plugin.channels,plugin.length,MPC_DECODER_BUFFER_LENGTH);

    if(!(tmpbuf = (MPC_SAMPLE_FORMAT*)malloc(sizeof(MPC_SAMPLE_FORMAT)*MPC_DECODER_BUFFER_LENGTH)))
    {
        fprintf(stderr, "mpcplugin:init_file - Cannot allocate memory\n");
        close();
        return -1;
    }

#ifdef A320
    plugin.clockfreq = 360;
#else
    plugin.clockfreq = 200;
#endif
    writepos      = 0;
    readpos       = 0;
    samples_done  = 0;
    playing = 1;
    //printf("BOOYAH! shift = %d min = %d max = %d\n",m_shift, clip_min, clip_max);
    return 0;
}

static int close()
{
    mpc_demux_exit(demux);
    mpc_reader_exit_stdio(&reader);

    for(int i = 0; i < 5; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }
    playing = plugin.length = plugin.nfields = 0;
    return 0;
}

#if 1
#define shift_signed(x,y) ((int)x)>>(-y);
#else
static int shift_signed(int val, int shift)
{
    if (shift > 0)
        val <<= shift;
    else if (shift < 0)
        val >>= -shift;
    return (int)val;
}
#endif

void bufcpy(signed short *dest, MPC_SAMPLE_FORMAT *src, int samp)
{
    for (int i = 0; i < samp; i++)
    {
        int tmp = src[i] >> MPC_FIXED_POINT_FRACTPART;
		if (tmp > ((1 << 15) - 1)) tmp = ((1 << 15) - 1);
		if (tmp < -(1 << 15)) tmp = -(1 << 15);
        dest[i] = tmp;
#if 0
        int val = shift_signed(src[i], m_shift);
        if (val < clip_min)
            val = clip_min;
        else if (val > clip_max)
            val = clip_max;
        dest[i] = val;
#endif
    }
}

static int fill_buffer(signed short *dest, int len)
{
    mpc_frame_info frame;
  	mpc_status err;

    if(playing)
    {
        int towrite = len>>1;
        int written = 0;
        for (;;)
        {
            int left = towrite - written;
            int buf  = writepos - readpos;
            if (left <= buf)
            {
                bufcpy(&dest[written], &tmpbuf[readpos], left);
                readpos += left;
                written = towrite;
                break;
            }
            else
            {
                bufcpy(&dest[written], &tmpbuf[readpos], buf);
                written += buf;

                frame.buffer = tmpbuf;
                mpc_demux_decode(demux, &frame);
        
                if (frame.bits == -1) {
                    writepos = 0;
                    playing = 0;
                    break;
                }

                writepos = frame.samples * sizeof(MPC_SAMPLE_FORMAT) * plugin.channels;
                fprintf(stderr, "writepos = %d\n", writepos);
                // writepos = mpc_decoder_decode(&mpcdecoder, tmpbuf, 0, 0)*plugin.channels;
                readpos = 0;
                /*if (writepos <= 0)
                    break;
                */
            }
        }

        samples_done += written/plugin.channels;
        return written<<1;
    }

    return len;
}

static int can_handle(const char *name)
{
    return (is_ext(name, ".mpc") || is_ext(name, ".mp+")); 
}

static int set_position(int msecs, int subtune)
{
    if (playing)
    {
        int skip_samples = (int)((double)plugin.freq*msecs/1000);
        if (samples_done + skip_samples < 0)
            skip_samples = -samples_done;
        samples_done += skip_samples;
        mpc_status ok = mpc_demux_seek_sample(demux, samples_done);
        readpos = writepos = 0;

        return (int)(1000.0*skip_samples/plugin.freq);
    }
}

extern "C" {

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN mpc_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
    memset(&plugin, 0, sizeof(plugin));
    plugin.plugname = "MPC";
    //plugin.init_data = init_data;
    plugin.init_file = init_file;
    //plugin.request_format = NULL;
    plugin.set_position = set_position;
    plugin.fill_buffer = fill_buffer;
    plugin.can_handle = can_handle;
    plugin.close = close;
    plugin.tune = 0;
    plugin.subtunes = 1;
    plugin.replaygain = 1;
    return &plugin;
}

}

