/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "modplug/modplug.h"
#include "../plugin.h"
#include "../util.h"

#ifndef WIN32
#define stricmp strcasecmp
#endif

static char *types[] = { "", "Protracker", "ScreamTracker", "FastTracker II", 
"OctaMed","MTM", "ImpulseTracker", "669 Composer", "UltraTracker", "STM", 
"Farandole", "Wave", "Asylum/Advanced MF", "AMS", "DSM/DSIK", "DigiTracker", 
"Oktalyzer", "Midi", "DMF/X-Tracker", "PolyTracker", "DigiBooster Pro", 
"MadTracker 2.0", "Asylum/Advanced MF0", "PSM", "J2B" };

static struct sound_plugin plugin;
static ModPlugFile *mpfile = NULL;
static ModPlug_Settings settings;
static string fieldname[5];
static string fielddata[5];
//static int total_samples;
//static int song_end;

static int init_settings(char *fname)
{
    ModPlug_GetSettings(&settings);
    settings.mResamplingMode = MODPLUG_RESAMPLE_LINEAR;
    settings.mChannels = 2;
    settings.mBits = 16;
    settings.mFrequency = plugin.freq;
    settings.mFlags = 0;
    settings.mFlags |= MODPLUG_ENABLE_OVERSAMPLING;
    settings.mFlags |= MODPLUG_ENABLE_NOISE_REDUCTION;
    settings.mFlags |= MODPLUG_ENABLE_REVERB;
    settings.mFlags |= MODPLUG_ENABLE_MEGABASS;
    settings.mFlags |= MODPLUG_ENABLE_SURROUND;
    settings.mReverbDepth   = 40;
    settings.mReverbDelay   = 80;
    settings.mBassAmount    = 40;
    settings.mBassRange     = 40;
    settings.mSurroundDepth = 40;
    settings.mSurroundDelay = 10;
    ModPlug_SetSettings(&settings);
}

static int init_data(char *fname, void *data, int size)
{
    if(mpfile)
        ModPlug_Unload(mpfile);
    
    init_settings(fname);

    if (!(mpfile = ModPlug_Load(data, size)))
    {
        fprintf(stderr, "modplugin::init_data - Couldn't load data\n");
        return -1;
    }

    int t = ModPlug_GetModuleType(mpfile);
    int mt = 0;
    while(t)
    {
        t = t>>1;
        mt++;
    }
    
    const char* name = ModPlug_GetName(mpfile);
    int x = 0;
    if (name && strlen(name))
    {
        fieldname[x] = "Name";
        fielddata[x] = name;
    }
    else
    {
        fieldname[x] = "File";
        fielddata[x] = fname;
    }
    x++;
    fieldname[x] = "Format";
    fielddata[x] = types[mt];
    x++;
    
    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;
    plugin.nfields = x;

    plugin.length = ModPlug_GetLength(mpfile);
    plugin.subtunes = 1;
    plugin.tune = 0;
#ifdef A320
    plugin.clockfreq = 264;
#else
    plugin.clockfreq = 100;
#endif

    //song_end = plugin.length > 0 ? (plugin.length << 2)/1000*plugin.freq : 0;
    //total_samples = 0;
    //char *d = (char *)data;

    return 0;
}

static int close()
{
    if(mpfile)
    {
        ModPlug_Unload(mpfile);
        mpfile = NULL;
    }
    for(int i = 0; i < 5; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }
    plugin.tune = plugin.subtunes = plugin.length = plugin.nfields = 0;
    return 0;
}

static int fill_buffer(signed short *dest, int len)
{
    ////if(song_end > 0 && total_samples > song_end)
    //  return 0;
    if(mpfile)
    {
        if (settings.mChannels == 2)
            ModPlug_Read(mpfile, dest, len);
        else
        {
            ModPlug_Read(mpfile, dest, len>>1);
            for (int i = (len>>2)-1, j = (len>>1)-1; i >= 0; ) 
            {
                dest[j--] = dest[i];
                dest[j--] = dest[i--];
            }
        }
        return len;
    }
    return 0;
}

static int can_handle(const char *name)
{
    return (is_ext(name, ".669") || is_ext(name, ".amf") || 
            is_ext(name, ".ams") || is_ext(name, ".dbm") || 
            is_ext(name, ".dmf") || is_ext(name, ".dsm") || 
            is_ext(name, ".far") || is_ext(name, ".it")  || 
            is_ext(name, ".j2b") || is_ext(name, ".mdl") || 
            is_ext(name, ".med") || is_ext(name, ".mod") || 
            is_ext(name, ".mt2") || is_ext(name, ".mtm") || 
            is_ext(name, ".okt") || is_ext(name, ".psm") || 
            is_ext(name, ".ptm") || is_ext(name, ".s3m") || 
            is_ext(name, ".stm") || is_ext(name, ".ult") || 
            is_ext(name, ".umx") || is_ext(name, ".xm"));
}

extern "C" {

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN modplay_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
    memset(&plugin, 0, sizeof(plugin));
    //plugin.init_file = init_file;
    //plugin.set_position = set_position;
    plugin.init_data = init_data;
    plugin.fill_buffer = fill_buffer;
    plugin.can_handle = can_handle;
    plugin.close = close;
    plugin.plugname = "modplug";
    plugin.freq = 44100;
    plugin.channels = 2;
    plugin.replaygain = 1;

    return &plugin;
}

}

