/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h> 
#include <string>
#include <sys/stat.h>
#include <stdarg.h>
#include <mad.h>
#include <id3tag.h>

#include "../plugin.h"
#include "../util.h"

// Adds support for VBRI headers from FhG (haven't found any)
//#define VBRI_MAD

#define IN_SIZE 4096
static int sampperframe[] = {0, 384, 1152, 1152};

FILE *mp3_file = NULL;
static struct sound_plugin plugin;
struct mad_stream   *stream = NULL;
struct mad_frame    *frame = NULL;
struct mad_synth    *synth = NULL;
volatile int asked = 0;
static int playing = 0;
static int fsize;

// VBR stuff
static int xing;
#ifdef VBRI_MAD
static int vbri;
#endif
int* toc = NULL;
static int toc_scale;
static int toc_size;
static int samples_done;
// VBR stuff 

signed short *tmpbuf = NULL;
unsigned char *inbuf = NULL;

static int offset;
static int readposbytes;
static int writeposbytes;
static bool quit_thread;
static string fieldname[5];
static string fielddata[5];

void mad_stream_buffer_clear(struct mad_stream *stream)
{
	int left = stream->bufend - stream->this_frame;
	int size = stream->this_frame - stream->buffer;
	
    memcpy((void *)stream->buffer, (void *)stream->this_frame, left);

	stream->bufend = stream->buffer + left;
	stream->this_frame -= size;
	stream->next_frame -= size;
	stream->ptr.byte -= size;
}

int handle_input(struct mad_stream *stream)
{
	mad_stream_buffer_clear(stream);
	int len = fread((void *)stream->bufend, 1, IN_SIZE, mp3_file);
	if(len < IN_SIZE)
		playing = 0;
	stream->bufend += len;
    return len;
}

int handle_output(struct mad_pcm *pcm)
{
    //printf("handle output -> %d hz\n",pcm->samplerate);
	signed int sample;
	int nSamples  = pcm->length; // Per channel
	mad_fixed_t *left_ch   = pcm->samples[0];
    mad_fixed_t *right_ch;

    if (plugin.channels == 2)
        right_ch  = pcm->samples[1];

	short *outp = tmpbuf;
	while(nSamples--)
	{
		sample = *left_ch++;
		sample += (1L << (MAD_F_FRACBITS - 16));
		if (sample >= MAD_F_ONE)
			sample = MAD_F_ONE - 1;
		else if (sample < -MAD_F_ONE)
			sample = -MAD_F_ONE;
        signed int s = sample >> (MAD_F_FRACBITS + 1 - 16); 
        *outp++ = s; 
	    
        if (plugin.channels == 2)
        {
            sample = *right_ch++;
            sample += (1L << (MAD_F_FRACBITS - 16));
            if (sample >= MAD_F_ONE)
                sample = MAD_F_ONE - 1;
            else if (sample < -MAD_F_ONE)
                sample = -MAD_F_ONE;
            *outp++ = sample >> (MAD_F_FRACBITS + 1 - 16);
        }
	}
	return pcm->length*2*plugin.channels;
}

static int fill_buffer(signed short *dest, int len)
{
	if(playing)
	{
        int writtenbytes = 0;
        for (;;)
        {
            int bytesleft = len - writtenbytes;
            int bufbytes  = writeposbytes - readposbytes;
            //printf("len = %04d - written = %04d = %04d, " \
                    "read = %04d - write = %04d = %04d\n",\
                len,writtenbytes,bytesleft,readposbytes,writeposbytes,bufbytes);
            if (bytesleft <= bufbytes)
            {
                memcpy((char*)dest + writtenbytes, (char*)tmpbuf + readposbytes, bytesleft);
                readposbytes += bytesleft;
                writtenbytes = len;
                break;
            }
            else
            {
                memcpy((char*)dest + writtenbytes, (char*)tmpbuf + readposbytes, bufbytes);
                writtenbytes += bufbytes;
                readposbytes = writeposbytes = 0;
                for (;;)
                {
                    if ((stream->bufend - stream->next_frame) < IN_SIZE)
                        if (handle_input(stream) <= 0)
                            return writtenbytes;
                    
                    if (mad_frame_decode(frame, stream) != -1)
                        break;
                }
                mad_synth_frame(synth, frame);
                writeposbytes = handle_output(&synth->pcm);
                if (!writeposbytes)
                    break;
            }
        }
        samples_done += writtenbytes/(2*plugin.channels);
        //printf("written = %d\n",writtenbytes);
        return writtenbytes;
	}

	return 0;
}

static int close()
{
fprintf(stderr, "foo\n");
    if (mp3_file)
    {
        fclose(mp3_file);
        mp3_file = NULL;
    }

fprintf(stderr, "foo 1\n");
    if (toc)
    {
        free(toc);
        toc = NULL;
    }

fprintf(stderr, "foo 3\n");
    if (tmpbuf) {
		delete [] tmpbuf;
	}

fprintf(stderr, "foo 2\n");
    if (inbuf) {
		delete [] inbuf;
	}

fprintf(stderr, "foo 4\n");
    inbuf = NULL;
    tmpbuf = NULL;

fprintf(stderr, "foo 5\n");
    mad_synth_finish(synth);
	mad_frame_finish(frame);
	mad_stream_finish(stream);

fprintf(stderr, "foo 6\n");
	if( stream ) delete stream;
	if( frame ) delete frame;
	if( synth ) delete synth;

    stream = NULL;
    frame = NULL;
    synth = NULL;

fprintf(stderr, "foo 4\n");
    for(int i = 0; i < 5; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }
fprintf(stderr, "foo 5\n");
    playing = plugin.tune = plugin.subtunes = plugin.length = plugin.nfields = 0;
	return 0;
}

static int init_file(char *fname)
{
	plugin.subtunes = 1;
	plugin.tune = 0;
	plugin.length = -1;

	fprintf(stderr, "madplugin:init_file - Decoder Init\n");

    /* TODO - check inbuf size */
    tmpbuf = new signed short [1024*8];
	inbuf = new unsigned char [IN_SIZE*6];
	if( inbuf == NULL || tmpbuf == NULL ) return -1;

    stream = new struct mad_stream;
	frame = new struct mad_frame;
	synth = new struct mad_synth;
	if( synth == NULL || stream == NULL || frame == NULL ) return -1;

    mad_stream_init(stream);
    mad_frame_init(frame);
    mad_synth_init(synth);

    int x = parse_id3v1(fname, fieldname, fielddata, 4);
    fieldname[x] = "Format";
    fielddata[x++] = is_ext(fname,".mp3") ? "MP3" : "MP2";

    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;
    plugin.nfields   = x;

    mp3_file = fopen(fname, "rb");
	fseek(mp3_file, 0, SEEK_END);
	fsize = ftell(mp3_file);
	fseek(mp3_file, 0, SEEK_SET);

    int len = fread(inbuf, 1, IN_SIZE, mp3_file);
    mad_stream_buffer(stream, (const unsigned char *)inbuf, len);	 

    // Track length calculation
    
	int bitrate;
	readposbytes = writeposbytes = 0;
    //handle_input(&stream);
    for (;;)
    {
        if ((stream->bufend - stream->next_frame) < IN_SIZE)
            if (handle_input(stream) <= 0)
            {
                printf("madplugin:init_file - Couldn't find frames to decode\n");
                close();
                return -1;
            }
    
        if (mad_frame_decode(frame, stream) != -1)
        {
            printf("decode!\n");
            mad_synth_frame(synth, frame);
            plugin.channels = synth->pcm.channels;
            plugin.freq     = synth->pcm.samplerate;
            bitrate         = frame->header.bitrate;
            writeposbytes = handle_output(&synth->pcm);
            break;
        }
    }

    // Guessing the offset to first frame depending on channels. I simply
    // looked into mp3-files and compared.
    int headersize = plugin.channels == 1 ? 0x15 : 0x24;
    int id3size = 0;
    if (!strncmp((const char*)inbuf,"ID3",3))
    {
        while (inbuf[id3size] != 0xFF || inbuf[id3size+1] != 0xFB)
            id3size++;
        id3size--; // just to save some operations
    }
    offset = headersize + id3size;
    char *tmpinbuf = (char*)inbuf + offset;
    xing = toc_size = 0;
#ifdef VBRI_MAD
    vbri = 0;
#endif
    int frames = 0;
    if (strcmp(tmpinbuf,"Xing") == 0)
    {
        fprintf(stderr, "madplugin:init_file - Xing frame found\n");
        xing = 1;
        if (tmpinbuf[7] & 0x01) // Has framecount
            frames = (int)tmpinbuf[11] | (int)tmpinbuf[10]<<8 | 
                    (int)tmpinbuf[9]<<16 | (int)tmpinbuf[8]<<24;
        if (tmpinbuf[7] & 0x04) // Has TOC
        {
            toc = (int*)malloc(100*sizeof(int));
            toc_size = 100;
            int tocread = 8;
            if (tmpinbuf[7] & 0x01)
                tocread += 4;
            if (tmpinbuf[7] & 0x02)
                tocread +=4;
            for (int i = 0; i < 100; i++)
                toc[i] = tmpinbuf[tocread+i];
            toc_scale = 256;
            //for (int i = 0; i < 100; i++)
                //printf("toc[%d] = %d\n",i,toc[i]);
        }
    }
#ifdef VBRI_MAD
    else // Xing is good enough. VBRI sucks ass anyway.
    {
        tmpinbuf = (char*)inbuf + offset;
        for (int i = offset; i < IN_SIZE*2; i++,tmpinbuf++) // This is really really ugly
        {
            if (*tmpinbuf != 'V')
                continue;
            if (strcmp(tmpinbuf,"VBRI") == 0)
            {
                fprintf(stderr, "madplugin:init_file - VBRI frame found\n");
                vbri = 1;
                frames    = (int)tmpinbuf[17] | (int)tmpinbuf[16]<<8 | 
                            (int)tmpinbuf[15]<<16 | (int)tmpinbuf[14]<<24;
                toc_size  = (int)tmpinbuf[19] | (int)tmpinbuf[18]<<8;
                toc_scale = (int)tmpinbuf[21] | (int)tmpinbuf[20]<<8;
                int entry_size = (int)tmpinbuf[23] | (int)tmpinbuf[22]<<8;
        
                toc =  (int*)malloc(toc_size*sizeof(int));
                for (int i = 0; i < toc_size; i++)
                {
                    toc[i] = 0;
                    for (int j = 0; j < entry_size; j++)
                        toc[i] |= (int)tmpinbuf[i*entry_size]<<(8*(entry_size-j-1));
                }
                break;
            }
        }
    }
#endif

	if(!frames)
        if (!xing)
            plugin.length = (int)((double)8*1000*fsize/bitrate);
        else
            plugin.length = 0; // If this happens, my only choice is to read all headers
    else
        plugin.length = (int)(1000.0*frames*sampperframe[frame->header.layer] /
                                plugin.freq);

    writeposbytes = 0;
    readposbytes  = 0;
    samples_done  = 0;
	playing = 1;
	return 0;
}

static int set_position(int msecs, int subtune)
{
	if (msecs)
    {
        if (toc && plugin.length > 0)
        {
            double seek_frac_time = (1000.0*samples_done/plugin.freq+msecs)/plugin.length;
            int toc_index = (int)(seek_frac_time * toc_size);
            if (toc_index > toc_size-1) 
                toc_index = toc_size-1; // Could very well just return msecs, but that's ugly
            if (toc_index < 0)
                toc_index = 0;
            
            int seek_bytes = (int)((double)fsize*toc[toc_index]/toc_scale);
            if (seek_bytes > fsize)
                seek_bytes = fsize;
            if (seek_bytes < 0)
                seek_bytes = 0;
            
            fseek(mp3_file,seek_bytes,SEEK_SET);

            int seeked_msecs = (int)((double)toc_index/toc_size*plugin.length
                    - 1000.0*samples_done/plugin.freq);
            samples_done = (int)((double)toc_index/toc_size*plugin.length*plugin.freq/1000);
            return seeked_msecs;
        }
        int bitrate   = frame->header.bitrate;
        int seekbytes = (int)((double)bitrate*msecs/(8*1000)); 
        int curpos    = ftell(mp3_file);
        //printf("Seeking : %d : %d : %d\n",seekbytes,frame.header.bitrate,msecs);
        if (seekbytes < 0 && -seekbytes > curpos - offset)
            fseek(mp3_file,offset,SEEK_SET);
        else
            fseek(mp3_file, seekbytes, SEEK_CUR);
        int newpos = ftell(mp3_file);
        // Reduce the delay? but how much? unsure... this sucks ass.
        //mad_stream_skip(&stream, 48000);
        readposbytes = writeposbytes = 0;
        return (int)(8*1000.0*(newpos-curpos)/bitrate);
    }
    return 0;
}

static int can_handle(const char *name)
{
	return (is_ext(name, ".mp3") || is_ext(name, ".mp2"));
}

extern "C" {
#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN mad_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
	memset(&plugin, 0, sizeof(plugin));

	plugin.plugname     = "mad";
	plugin.init_file    = init_file;
	plugin.close        = close;
	//plugin.init_data  = init_data;
	plugin.fill_buffer  = fill_buffer;
	plugin.set_position = set_position;
	plugin.can_handle   = can_handle;
#ifdef A320
	plugin.clockfreq    = 300;
#else
	plugin.clockfreq    = 150;
#endif
    plugin.replaygain   = 1;

	return &plugin;
}
}

