/* 
    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "mac/All.h"
#include "mac/GlobalFunctions.h"
#include "mac/MACLib.h"
#include "mac/CharacterHelper.h"
#include "mac/APETag.h"
#include "../plugin.h"
#include "../util.h"

static struct sound_plugin plugin;
IAPEDecompress *pAPEDecompress;
static int charsperblock;
static int samples_done; // Counting total samples for all channels
static int playing;
static string fieldname[5];
static string fielddata[5];

static int close();

static int init_file(char *fname)
{
    playing = 0;
    plugin.clockfreq = 200;
    CSmartPtr<wchar_t> spInput;
    spInput.Assign(GetUTF16FromANSI(fname), true);
    printf("Foo!\n");
    int nRetVal;
    if (!(pAPEDecompress = CreateIAPEDecompress(spInput, &nRetVal)))
    {
        fprintf(stderr, "macplugin:init_file - Error opening %s (error code %d)\n", \
            fname, nRetVal);
        return -1;
    }
    printf("macplugin - Compression level: ");
    switch (pAPEDecompress->GetInfo(APE_INFO_COMPRESSION_LEVEL))
	{
		case COMPRESSION_LEVEL_FAST: 
            printf("Fast\n"); 
#ifdef A320
            plugin.clockfreq = 360;
#else
            plugin.clockfreq = 150;
#endif
            break;
		case COMPRESSION_LEVEL_NORMAL: 
            printf("Normal\n"); 
#ifdef A320
            plugin.clockfreq = 396;
#else
            plugin.clockfreq = 200;
#endif
            break;
		case COMPRESSION_LEVEL_HIGH: 
            printf("High\n"); 
#ifdef A320
            plugin.clockfreq = 420;
#else
            plugin.clockfreq = 250;
#endif
            break;
		case COMPRESSION_LEVEL_EXTRA_HIGH: 
#ifdef A320
            printf("Extra High! The dingoo *WONT* play this.\n"); 
#else
            printf("Extra High! The gp2x *WONT* play this.\n"); 
#endif
            return -1;
        case COMPRESSION_LEVEL_INSANE:
#ifdef A320
            printf("Insane! The dingoo *WONT* play this.\n"); 
#else
            printf("Insane! The gp2x *WONT* play this.\n");
#endif
            return -1;
	}

    CAPETag * pAPETag = (CAPETag *) pAPEDecompress->GetInfo(APE_INFO_TAG);
	BOOL bHasID3Tag = pAPETag->GetHasID3Tag();
	BOOL bHasAPETag = pAPETag->GetHasAPETag();
    int x = 0;
    if (bHasAPETag || bHasID3Tag)
    {
        string tmpname   = "";
        string tmpartist = "";
        string tmpperf   = "";
        string tmpalbum  = "";
        string tmpgenre  = "";
        string tmptrack  = "";
        string tmpdesc   = "";
        string tmpcopy   = "";
        string tmpdate   = "";
        string tmpvers   = "";
        string tmploc    = "";
        CAPETagField *  pAPETagField;
        for (int i = 0; pAPETagField = pAPETag->GetTagField(i) ; i++)
        {
            const char *tmp_fieldname = GetANSIFromUTF16(pAPETagField->GetFieldName());
            const char *tmp_fielddata = pAPETagField->GetFieldValue();
            printf("cmp = %d\n",strcasecmp(tmp_fieldname,"artist"));
                 if (!strcasecmp(tmp_fieldname,"name"))
                tmpname   = tmp_fielddata;
            else if (!strcasecmp(tmp_fieldname,"title"))
                tmpname   = tmp_fielddata;
            else if (!strcasecmp(tmp_fieldname,"performer"))
                tmpperf   = tmp_fielddata;
            else if (!strcasecmp(tmp_fieldname,"artist"))
                tmpartist = tmp_fielddata;
            else if (!strcasecmp(tmp_fieldname,"album"))
                tmpalbum  = tmp_fielddata;
            else if (!strcasecmp(tmp_fieldname,"genre"))
                tmpgenre  = tmp_fielddata;
            else if (!strcasecmp(tmp_fieldname,"track"))
                tmptrack  = tmp_fielddata;
            else if (!strcasecmp(tmp_fieldname,"tracknumber"))
                tmptrack  = tmp_fielddata;
            else if (!strcasecmp(tmp_fieldname,"description"))
                tmpdesc   = tmp_fielddata;
            else if (!strcasecmp(tmp_fieldname,"tmp_fieldname"))
                tmpdesc   = tmp_fielddata;
            else if (!strcasecmp(tmp_fieldname,"copyright"))
                tmpcopy   = tmp_fielddata;
            else if (!strcasecmp(tmp_fieldname,"date"))
                tmpdate   = tmp_fielddata;
            else if (!strcasecmp(tmp_fieldname,"version"))
                tmpvers   = tmp_fielddata;
            else if (!strcasecmp(tmp_fieldname,"location"))
                tmploc    = tmp_fielddata;
        }
            // Add the comments
        if (tmpname.length())
        {
            fieldname[x] = "Title";
            fielddata[x] = tmpname;
            if (tmpvers.length())
                fielddata[x] += " (" + tmpvers + ")";
        }
        else
        {
            fieldname[x] = "File";
            fielddata[x] = strrchr(fname,'/')+1;
        }
        x++;
        if (tmpalbum.length()) 
        {
            fieldname[x] = "Album";
            fielddata[x] = tmpalbum;
            x++;
        } // max 2 so far
        if (tmpartist.length())
        {
            fieldname[x] = "Artist";
            fielddata[x] = tmpartist;
            x++;
        } // max 3
        if (tmpperf.length())
        {
            fieldname[x] = "Performer";
            fieldname[x] = tmpperf;
            x++;
        } // max 4 (have to start checking)
        if (x < 4 && tmpgenre.length())
        {
            fieldname[x] = "Genre";
            fielddata[x] = tmpgenre;
            x++;
        }
        if (x < 4 && tmptrack.length())
        {
            fieldname[x] = "Track";
            fielddata[x] = tmptrack;
            x++;
        }
        else if (tmptrack.length() && tmpname.length())
            fielddata[0] = string(tmptrack) + " - " + fielddata[0];
        if (x < 4 && tmpdesc.length())
        {
            fieldname[x] = "Description";
            fielddata[x] = tmpdesc;
        }
        if (x < 4 && tmpcopy.length())
        {
            fieldname[x] = "Copyright";
            fielddata[x] = tmpcopy;
            x++;
        }
        if (x < 4 && tmpdate.length())
        {
            fieldname[x] = "Date";
            fielddata[x] = tmpdate;
            x++;
        }
        if (x < 4 && tmploc.length())
        {
            fieldname[x] = "Location";
            fielddata[x] = tmploc;
            x++;
        }
    }
    fieldname[x] = "Format";
    fielddata[x] = "APE";
    x++;
    plugin.nfields = x;
    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;

    plugin.channels = pAPEDecompress->GetInfo(APE_INFO_CHANNELS);
    plugin.freq     = pAPEDecompress->GetInfo(APE_INFO_SAMPLE_RATE);
    plugin.length   = pAPEDecompress->GetInfo(APE_INFO_LENGTH_MS);
    charsperblock   = pAPEDecompress->GetInfo(APE_INFO_BLOCK_ALIGN);
    
    samples_done  = 0;
    playing = 1;
    return 0;
}

static int close()
{
    if (pAPEDecompress)
    {
        delete pAPEDecompress;
        pAPEDecompress = NULL;
    }
    for(int i = 0; i < 5; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }
    playing = plugin.length = plugin.nfields = 0;
    return 0;
}

static int fill_buffer(signed short *dest, int len)
{
    if (playing)
    {
        // 1 block = 1 sample from each channel.
        int writtenblocks;
        int error = pAPEDecompress->GetData((char*)dest, len/charsperblock, &writtenblocks);
        samples_done += writtenblocks << 1;
        return writtenblocks*charsperblock;
    }
    return 0;
}

static int set_position(int msecs, int subtune)
{
    if (playing && msecs)
    {
        int blocks_done = (int)(samples_done*2/charsperblock);
        int skip_blocks = (int)((double)plugin.freq*plugin.channels*msecs/1000)* \
                                2/charsperblock;
        if (skip_blocks < -blocks_done)
            skip_blocks = -blocks_done;
        blocks_done += skip_blocks;

        int error = pAPEDecompress->Seek(blocks_done);
        if (error)
        {
            playing = 0;
            printf("macplugin:set_position - Error %d\n",error);
        }
        int new_samples_done = blocks_done * charsperblock >> 1;
        int skipped_time = (int)(1000.0*(new_samples_done - samples_done)/ \
                            (plugin.channels*plugin.freq));
        samples_done = new_samples_done;
        return skipped_time;
    }
    return 0;
}
 
static int can_handle(const char *name)
{
    return (is_ext(name, ".ape") || is_ext(name, ".mac")); 
}

extern "C" {

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN mac_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
    memset(&plugin, 0, sizeof(plugin));
    plugin.plugname = "MAC";
    //plugin.init_data = init_data;
    plugin.init_file = init_file;
    //plugin.request_format = NULL;
    plugin.set_position = set_position;
    plugin.fill_buffer = fill_buffer;
    plugin.can_handle = can_handle;
    plugin.close = close;
    plugin.tune = 0;
    plugin.subtunes = 1;
    plugin.replaygain = 1;
    return &plugin;
}

}

