/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

extern "C" {
#include "hvl_replay.h"
}

#include "../plugin.h"
#include "../util.h"

static struct sound_plugin plugin;
struct hvl_tune *ht = NULL;

#define SAMPLERATE      44100
#define BPS             16
#define CHANNELS        2

char audiobuf[SAMPLERATE * (BPS/8 * CHANNELS * 2)];   // Samplerate * (bps/2) * channels * 2
int writeposbytes = 0;
int readposbytes  = 0;

int stereosep  = 2;

static int playing;
static string fieldname[1];
static string fielddata[1];
static string filename;

static int set_position(int msecs, int subtune);

static int init_file(char *fname)
{
    playing = 0;
    plugin.tune = 0;
    plugin.subtunes = 1;

	ht = hvl_LoadTune( fname, SAMPLERATE, stereosep );
	if( !ht ) return -1;
	
	// Big enough to play all song
    plugin.length = hvl_GetLen(ht, 0);
	hvl_InitSubsong(ht, 0);
	if( plugin.length <= 0 )
		plugin.length = 1800 * 1000;  // Just big enough for now (30 min) :0
		
    if( ht->ht_SongNum > 0 ) plugin.subtunes =  ht->ht_SongNum+1;
		
	writeposbytes = readposbytes = 0;

    plugin.nfields = 0;
    if( strlen(ht->ht_Name) ) {
        fieldname[0] = "Song";
        fielddata[0] = string(ht->ht_Name);   

        plugin.fieldname = fieldname;
        plugin.fielddata = fielddata;
        plugin.nfields  = 1;
    }
	
    playing = 1;
    return 0;
}

static int close()
{
   if( ht ) {
		hvl_FreeTune(ht);
		ht = NULL;
	}

    for(int i = 0; i < plugin.nfields; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }

    playing = plugin.tune = plugin.subtunes = plugin.length = plugin.nfields = 0;
    return 0;
}

static int fill_buffer(signed short *dest, int len)
{
    if (playing)
    {
        int writtenbytes = 0;
        for (;;)
        {
            int bytesleft = len - writtenbytes;
            int bufbytes  = writeposbytes - readposbytes;
           
            if (bytesleft <= bufbytes)
            {
                memcpy((char*)dest + writtenbytes, (char*)audiobuf + readposbytes, bytesleft);
                readposbytes += bytesleft;
                writtenbytes = len;
                break;
            }
            else
            {
                memcpy((char*)dest + writtenbytes, (char*)audiobuf + readposbytes, bufbytes);
                writtenbytes += bufbytes;
                readposbytes = writeposbytes = 0;

                hvl_DecodeFrame(ht, audiobuf, audiobuf+(BPS/8), (BPS/8)*CHANNELS);
				writeposbytes = (SAMPLERATE*(BPS/8)*CHANNELS)/50;
            }
        }

        return writtenbytes; 
    }
	
    return 0;
}

static int set_position(int msecs, int subtune)
{
    if (!playing)
        return 0;

    if( !msecs ) {
        // Big enough to play all song
        plugin.length = hvl_GetLen(ht, subtune);
	    hvl_InitSubsong(ht, subtune);
    	if( plugin.length <= 0 )
	    	plugin.length = 1800 * 1000;  // Just big enough for now (30 min :0)
	
    	memset(audiobuf, 0, sizeof(audiobuf));
	    writeposbytes = readposbytes = 0;
    }
    else {
        hvl_Seek(ht, msecs);
        return msecs;
    }

    return 0;
}


static int can_handle(const char *name)
{
    const char *tmp = strrchr(name,'/');
    tmp = tmp ? tmp+1 : name;

    return is_ext(name, ".ahx") || is_ext(name, ".hvl") || !strncasecmp(tmp, "ahx.", 4);
}

extern "C" {

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN hively_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
  	hvl_InitReplayer();

    memset(&plugin, 0, sizeof(plugin));
    plugin.plugname = "Hively Tracker v1.6";
    plugin.init_file        = init_file;
    //plugin.init_data      = init_data;
    //plugin.request_format = NULL;
    plugin.set_position     = set_position;
    plugin.fill_buffer      = fill_buffer;
    plugin.can_handle       = can_handle;
    plugin.close            = close;

    plugin.channels   = CHANNELS;
    plugin.freq       = SAMPLERATE;
    plugin.replaygain = 1;

#ifdef A320
    plugin.clockfreq = 264; 
#else
    plugin.clockfreq = 150; 
#endif

    return &plugin;
}

}

