/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "gme/Music_Emu.h"
#include "gme/Gme_File.h"

#include "../plugin.h"
#include "../util.h"

static struct sound_plugin plugin;
Music_Emu* emu = NULL;
gme_info_t* t_info = NULL;
static gme_err_t rc;
static int playing;
static int default_length;
static string fieldname[5];
static string fielddata[5];
static string filename;
static int set_position(int msecs, int subtune);

static int init_file(char *fname)
{
    playing = 0;
    plugin.tune = 0;
    plugin.subtunes = 1;
#ifdef A320
    plugin.clockfreq = 360; 
    if (is_ext(fname, ".spc"))
	    plugin.clockfreq = 396;
    else if (is_ext(fname, ".gym"))
    	plugin.clockfreq = 408;
    else if (is_ext(fname, ".vgz") || is_ext(fname, ".vgm"))
        plugin.clockfreq = 420;
#else
    plugin.clockfreq = 150; 
    if (is_ext(fname, ".gym"))
        plugin.clockfreq = 200;
    else if (is_ext(fname, ".vgz") || is_ext(fname, ".vgm"))
        plugin.clockfreq = 225;
#endif

    if (rc = gme_open_file( fname, &emu , plugin.freq))
    {
        printf("%d\n",rc);
        return -1;
    }
    
    gme_set_stereo_depth( emu, 0.8 );
    
    ////// M3u //////
    char* fnamem3u = (char*)malloc(sizeof(char)*(strlen(fname)+1));
    strcpy(fnamem3u,fname);
    char* end = strrchr(fnamem3u,'.');
    strcpy(end,".m3u");
    rc = gme_load_m3u( emu, (const char*)fnamem3u );
    free(fnamem3u);
    ////// M3u /////

    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;
    filename = strrchr(fname, '/')+1;

    default_length = (2*60+30)*1000;
    //if (is_ext(fname, ".sap")) // Im not so sure about this myself
    //    default_length = (5*60)*1000;

    playing = 1;
    set_position(0,plugin.tune);
    return 0;
}

static int close()
{
    gme_delete(emu);
    for(int i = 0; i < 5; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }
    playing = plugin.tune = plugin.subtunes = plugin.length = plugin.nfields = 0;
    return 0;
}

int fade = 0;
static int fill_buffer(signed short *dest, int len)
{
    if(playing)
    {
        if (fade && plugin.length > 0)
        {
            gme_set_fade(emu, plugin.length-3000);
            fade = 0;
        }
        gme_play(emu, len>>1, dest);

        if(gme_track_ended( emu ))
            plugin.length = 1;//gme_tell( emu );
        return len;
    }
    return 0;
}

//void hw_set_cpu(int speed); // For CPU-boost
static int set_position(int msecs, int subtune)
{
    if (!playing)
        return 0;
    if (!msecs)
    {
        gme_start_track(emu, subtune);
        plugin.tune = subtune;
        if (!(rc = gme_track_info( emu, &t_info, subtune ))) {
            if (t_info->length > 0)
                plugin.length = t_info->length;
            else if (t_info->intro_length >= 0 && t_info->loop_length > 0)
                plugin.length = t_info->intro_length + t_info->loop_length * 2;
            else
                plugin.length = -1; //default_length;
            int x = 0;
            if (strlen(t_info->game))
            {
                fieldname[x] = "Game";
                fielddata[x] = t_info->game;
            }
            else
            {
                fieldname[x] = "File";
                fielddata[x] = filename;
            }
            x++;
            if (strlen(t_info->song))
            {
                fieldname[x] = "Song";
                fielddata[x] = t_info->song;
                x++;
            }
            if (strlen(t_info->author))
            {
                fieldname[x] = "Author";
                fielddata[x] = t_info->author;
                x++;
            }
            if (strlen(t_info->copyright))
            {
                fieldname[x] = "Copyright";
                fielddata[x] = t_info->copyright;
                x++;
            }
            if (strlen(t_info->system))
            {
                fieldname[x] = "Format";
                fielddata[x] = t_info->system;
                x++;
            }
            plugin.nfields  = x;
    	    plugin.subtunes = gme_track_count( emu );
	        if( plugin.subtunes <= 0 ) plugin.subtunes = 1;
                fade = 1;
                return subtune;

            gme_free_info(t_info);
        }
    }
    else
    {
        if (rc = emu->seek(emu->tell()+(long)msecs))
            printf("%s\n",rc);
        return msecs;
    }
}

static int can_handle(const char *name)
{
    return (is_ext(name, ".nsf") || is_ext(name, ".nsfe") || 
            is_ext(name, ".spc") || is_ext(name, ".gym") || 
            is_ext(name, ".vgm") || is_ext(name, ".vgz") ||
            is_ext(name, ".sap") || is_ext(name, ".hes") ||
            is_ext(name, ".kss") || is_ext(name, ".gbs") ||
            is_ext(name, ".ay"));
}

extern "C" {

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN gme_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
    memset(&plugin, 0, sizeof(plugin));
    plugin.plugname = "GME";
    plugin.init_file        = init_file;
    //plugin.init_data      = init_data;
    //plugin.request_format = NULL;
    plugin.set_position     = set_position;
    plugin.fill_buffer      = fill_buffer;
    plugin.can_handle       = can_handle;
    plugin.close            = close;
    plugin.channels   = 2;
    plugin.freq       = 44100;
    plugin.replaygain = 1;
    return &plugin;
}

}

