/*  
    This file is part of OldPlay - a portable, multiformat musicplayer.
 
    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <FLAC/stream_decoder.h>
#include <FLAC/metadata.h>
#include <FLAC/callback.h>
#include <FLAC/format.h>

#include "../plugin.h"
#include "../util.h"

#define BUFFSIZE (32768*2)  // Default, should be enough (*2 for stereo)
// Increase to twice this? flac for linux only lets you go this high.

static struct sound_plugin plugin;
FLAC__StreamDecoder *mydecoder;
static int playing;
static unsigned int total_samples;
static unsigned int samples_done; // Samples done for each channel
static unsigned int buffsize;
static signed short *tmpbuf;
static unsigned int writeposbytes;
static unsigned int readposbytes;
static string fieldname[5];
static string fielddata[5];

//typedef FLAC__StreamDecoderWriteStatus (* FLAC__StreamDecoderWriteCallback)
//(const FLAC__StreamDecoder *decoder, const FLAC__Frame *frame, 
// const FLAC__int32 *const buffer[], void *client_data)
FLAC__StreamDecoderWriteStatus write_callback(
    const FLAC__StreamDecoder *decoder, const FLAC__Frame *frame,
    const FLAC__int32 *const buffer[], void *client_Data)
{
    int samples = frame->header.blocksize * plugin.channels;
    //printf("filling buffers with %d bytes\n",samples<<1);
    if (samples > buffsize)
    {
        fprintf(stderr,"flacplugin:write_callback - To big block!\n");
        writeposbytes = 0;
        return FLAC__STREAM_DECODER_WRITE_STATUS_ABORT;
    }
    if (plugin.channels == 2)
    {
        for(int i = 0, j = 0; j < samples; i++)
        {
            tmpbuf[j++] = buffer[0][i];
            tmpbuf[j++] = buffer[1][i];
        }
    }
    else // Assumes mono
        for (int i = 0; i < samples; i++)
            tmpbuf[i] = buffer[0][i];
    
    writeposbytes = samples<<1;

    /*if (samples_done < 100000) // This stream seems right
    {
        //printf("writing file, samples done = %d\n",samples_done);
        FILE *fp = fopen("/mnt/sd/oldplay/fil","ab");
        fwrite(tmpbuf,samples,2,fp);
        fclose(fp);
    }*/
    return FLAC__STREAM_DECODER_WRITE_STATUS_CONTINUE;
}

//typedef void (*FLAC__StreamDecoderMetadataCallback)
//(const FLAC__StreamDecoder *decoder, const FLAC__StreamMetadata *metadata, void *client_data);
void metadata_callback(const FLAC__StreamDecoder *decoder, 
                       const FLAC__StreamMetadata *metadata, void *client_data)
{
    //printf("metadata_callback!\n");
    // this function doesn't seem to recieve the proper metadata...
}

//typedef void (*FLAC__StreamDecoderErrorCallback)
//(const FLAC__StreamDecoder *decoder, FLAC__StreamDecoderErrorStatus status, void *client_data);
void error_callback(const FLAC__StreamDecoder *decoder, 
                    FLAC__StreamDecoderErrorStatus status, void *client_data)
{
    fprintf(stderr, "flacplugin:error_callback - libFLAC error\n");
} 

static int close(); 

static int init_file(char *fname)
{
    playing = 0;
    buffsize = 0;
#ifdef A320
    plugin.clockfreq = 420;
#else
    plugin.clockfreq = 150; 
#endif
    FLAC__StreamDecoderWriteCallback f1 = write_callback;
    mydecoder = FLAC__stream_decoder_new();
    FLAC__stream_decoder_init_file(
        mydecoder,
        fname,
        write_callback,
        metadata_callback,
        error_callback,
        NULL);


    FLAC__Metadata_Chain *metachain = FLAC__metadata_chain_new();
    FLAC__metadata_chain_read(metachain,fname);
    FLAC__Metadata_Iterator *metaiter = FLAC__metadata_iterator_new();
    FLAC__metadata_iterator_init(metaiter, metachain);
    bool not_last = true;
    string tmpname   = "";
    string tmpartist = "";
    string tmpperf   = "";
    string tmpalbum  = "";
    string tmpgenre  = "";
    string tmptrack  = "";
    string tmpdesc   = "";
    string tmpcopy   = "";
    string tmpdate   = "";
    string tmpvers   = "";
    string tmploc    = "";
    for(;;) 
    {
        FLAC__MetadataType metatype = FLAC__metadata_iterator_get_block_type(metaiter);
        FLAC__StreamMetadata *streammeta = 
                                    FLAC__metadata_iterator_get_block(metaiter);
        switch (metatype)
        {
        FLAC__StreamMetadata_VorbisComment vcomment;
        FLAC__StreamMetadata_StreamInfo streaminfo;
        case FLAC__METADATA_TYPE_STREAMINFO:
            //printf("streaminfo! i = %d, metadata[i].type = %d\n",i,metadata[i].type);
            streaminfo = streammeta->data.stream_info;
            plugin.channels = streaminfo.channels;
            plugin.freq     = streaminfo.sample_rate;
            total_samples   = streaminfo.total_samples;
            buffsize        = streaminfo.max_blocksize;
            break;
        case FLAC__METADATA_TYPE_VORBIS_COMMENT:
            vcomment = streammeta->data.vorbis_comment;
            for (int i = 0; i < vcomment.num_comments; i++)
            {
                char* comment = (char*)vcomment.comments[i].entry;
                if (!strncasecmp(comment,"name=",5) && strlen(comment+5))
                    tmpname   = comment+5;
                else if (!strncasecmp(comment,"title=",6) && strlen(comment+6))
                    tmpname   = comment+6;
                else if (!strncasecmp(comment,"performer=",10) && strlen(comment+10))
                    tmpperf   = comment+10;
                else if (!strncasecmp(comment,"artist=",7) && strlen(comment+7))
                    tmpartist = (comment+7);
                else if (!strncasecmp(comment,"album=",6) && strlen(comment+6))
                    tmpalbum  = comment+6;
                else if (!strncasecmp(comment,"genre=",6) && strlen(comment+6))
                    tmpgenre  = comment+6;
                else if (!strncasecmp(comment,"track=",6) && strlen(comment+6))
                    tmptrack  = comment+6;
                else if (!strncasecmp(comment,"tracknumber=",12) && strlen(comment+12))
                    tmptrack  = comment+12;
                else if (!strncasecmp(comment,"description=",12) && strlen(comment+12))
                    tmpdesc   = comment+12;
                else if (!strncasecmp(comment,"comment=",8) && strlen(comment+8))
                    tmpdesc   = comment+8;
                else if (!strncasecmp(comment,"copyright=",10) && strlen(comment+10))
                    tmpcopy   = comment+10;
                else if (!strncasecmp(comment,"date=",5) && strlen(comment+5))
                    tmpdate   = comment+5;
                else if (!strncasecmp(comment,"version=",8) && strlen(comment+8))
                    tmpvers   = comment+8;
                else if (!strncasecmp(comment,"location=",9) && strlen(comment+9))
                    tmploc    = comment+9;
            }
            break;
        default:
            break;
        }
        if (!not_last)
            break;
        not_last = FLAC__metadata_iterator_next(metaiter);        
    }
    FLAC__metadata_iterator_delete(metaiter);
    FLAC__metadata_chain_delete(metachain);

    // Add the comments
    int x = 0; // I can fit four of these + format
    if (tmpname.length())
    {
        fieldname[x] = "Title";
        fielddata[x] = tmpname;
        if (tmpvers.length())
            fielddata[x] += " (" + tmpvers + ")";
    }
    else
    {
        fieldname[x] = "File";
        fielddata[x] = strrchr(fname,'/')+1;
    }
    x++;
    if (tmpalbum.length()) 
    {
        fieldname[x] = "Album";
        fielddata[x] = tmpalbum;
        x++;
    } // max 2 so far
    if (tmpartist.length())
    {
        fieldname[x] = "Artist";
        fielddata[x] = tmpartist;
        x++;
    } // max 3
    if (tmpperf.length())
    {
        fieldname[x] = "Performer";
        fieldname[x] = tmpperf;
        x++;
    } // max 4 (have to start checking)
    if (x < 4 && tmpgenre.length())
    {
        fieldname[x] = "Genre";
        fielddata[x] = tmpgenre;
        x++;
    }
    if (x < 4 && tmptrack.length())
    {
        fieldname[x] = "Track";
        fielddata[x] = tmptrack;
        x++;
    }
    else if (tmptrack.length() && tmpname.length())
        fielddata[0] = string(tmptrack) + " - " + fielddata[0];
    if (x < 4 && tmpdesc.length())
    {
        fieldname[x] = "Description";
        fielddata[x] = tmpdesc;
    }
    if (x < 4 && tmpcopy.length())
    {
        fieldname[x] = "Copyright";
        fielddata[x] = tmpcopy;
        x++;
    }
    if (x < 4 && tmpdate.length())
    {
        fieldname[x] = "Date";
        fielddata[x] = tmpdate;
        x++;
    }
    if (x < 4 && tmploc.length())
    {
        fieldname[x] = "Location";
        fielddata[x] = tmploc;
        x++;
    }
    fieldname[x] = "Format";
    fielddata[x] = "FLAC";
    x++;
    plugin.nfields = x;
    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;

    //FLAC__stream_decoder_set_metadata_respond_all(mydecoder); // No good
    FLAC__stream_decoder_set_metadata_ignore_all(mydecoder); // Changes nothing?
    FLAC__stream_decoder_process_until_end_of_metadata(mydecoder);

    if (!total_samples)
        total_samples = FLAC__stream_decoder_get_total_samples(mydecoder);
    if (!plugin.channels)
        plugin.channels = FLAC__stream_decoder_get_channels(mydecoder);
    if (plugin.channels > 2)
        plugin.channels = 2;
    if (!plugin.freq)
        plugin.freq = FLAC__stream_decoder_get_sample_rate(mydecoder);
    
    // Worst case... nothing seems to know!
    if (!plugin.channels)
        plugin.channels = 2;
    if (!plugin.freq)
        plugin.freq = 44100;

    plugin.length    = (int)(1000.0*total_samples/plugin.freq);
    
    buffsize *= plugin.channels;
    if (!buffsize)
        buffsize = BUFFSIZE;
    if(!(tmpbuf = (short int*)malloc(buffsize*2)))
    {
        fprintf(stderr, "Cannot allocate memory\n");
        close();
        return -1;
    }
    playing = 1;
    writeposbytes = 0;
    readposbytes  = 0;
    samples_done  = 0;
    return 0;
}

static int close()
{
    if (mydecoder)
    {
        FLAC__stream_decoder_finish(mydecoder);
        FLAC__stream_decoder_delete(mydecoder);
        mydecoder = NULL;
    }
    if (tmpbuf)
    {
        free(tmpbuf);
        tmpbuf = NULL;
    }
    for(int i = 0; i < 5; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }
    playing = plugin.length = plugin.nfields = 0;
    return 0;
}

static int fill_buffer(signed short *dest, int len)
{
    if(playing)
    {        
        int writtenbytes = 0;
        for (;;)
        {
            int bytesleft = len - writtenbytes;
            int bufbytes  = writeposbytes - readposbytes;
            //printf("len = %04d - written = %04d = %04d, " \
                    "read = %04d - write = %04d = %04d\n",\
                len,writtenbytes,bytesleft,readposbytes,writeposbytes,bufbytes);
            if (bytesleft <= bufbytes)
            {
                memcpy((char*)dest + writtenbytes, (char*)tmpbuf + readposbytes, bytesleft);
                readposbytes += bytesleft;
                writtenbytes = len;
                break;
            }
            else
            {
                memcpy((char*)dest + writtenbytes, (char*)tmpbuf + readposbytes, bufbytes);
                writtenbytes += bufbytes;
                readposbytes = writeposbytes = 0;
                if (!FLAC__stream_decoder_process_single(mydecoder))
                {
                    playing = 0;
                    return 0;
                }
                if (!writeposbytes)
                    break;
            }
        }
        samples_done += writtenbytes/(2*plugin.channels);
        //printf("written = %d\n",writtenbytes);
        return writtenbytes;
    }
    return 0;
}

static int set_position(int msecs, int subtune)
{
    if (playing && msecs)
    {
        int skip_samples = (int)((double)plugin.freq*msecs/1000);
        if ((int)samples_done + skip_samples < 0)
            skip_samples = -samples_done;
        samples_done += skip_samples;
        FLAC__bool ok = FLAC__stream_decoder_seek_absolute(mydecoder, samples_done);
        readposbytes = writeposbytes = 0;

        return (int)(1000.0*skip_samples/plugin.freq);
    }
    return 0;
}

static int can_handle(const char *name)
{
    return (is_ext(name, ".flac")); 
}

extern "C" {

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN flac_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
    memset(&plugin, 0, sizeof(plugin));
    plugin.plugname = "FLAC";
    //plugin.init_data = init_data;
    plugin.init_file = init_file;
    //plugin.request_format = NULL;
    plugin.set_position = set_position;
    plugin.fill_buffer = fill_buffer;
    plugin.can_handle = can_handle;
    plugin.close = close;
    plugin.tune = 0;
    plugin.subtunes = 1;
    plugin.replaygain = 1;
    return &plugin;
}

}

