/* 
    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

extern "C" {
#include "faad/neaacdec.h"
}

#include "mp4ff/mp4ff.h"

#include "../plugin.h"
#include "../util.h"

#define SAFE_FREE(x)		if(x) free(x); x = NULL;
#define MAX_CHANNELS 2
typedef struct {
    long bytes_into_buffer;
    long bytes_consumed;
    long file_offset;
    unsigned char *buffer;
    int at_eof;
    FILE *infile;
} aac_buffer;
static int adts_sample_rates[] = 
{96000,88200,64000,48000,44100,32000,24000,22050,16000,12000,11025,8000,7350,0,0,0};

static struct sound_plugin plugin;
static NeAACDecHandle hDecoder;
static NeAACDecFrameInfo frameInfo;
static NeAACDecConfiguration *config;
static aac_buffer b;

/* MP4 Stuff */
static mp4ff_t 			  *infile = NULL;
static mp4ff_callback_t 	  *mp4cb  = NULL;
static mp4AudioSpecificConfig mp4ASC;
static long sampleId = 0, numSamples;
static int track;

static FILE 		 *mp4File = NULL;
static unsigned char *buffer = NULL;
static unsigned int   buffer_size;

static int mp4file = 0;

static void *tmpbuf = NULL;
static int writeposbytes;
static int readposbytes;
static int samples_done; // Counting total samples for all channels
static int bitrate;
static int playing;
static int tagsize;
static string fieldname[5];
static string fielddata[5];

static int close();
static int fill_b_buffer(aac_buffer *b);
static void advance_buffer(aac_buffer *b, int bytes);
static int fill_buffer_mp4(signed short *dest, int len);
static int fill_buffer(signed short *dest, int len);
int GetAACTrack(mp4ff_t *infile);

int adts_parse(aac_buffer *b, int *bitrate, float *length)
{
    int frames, frame_length;
    int t_framelength = 0;
    int samplerate;
    float frames_per_sec, bytes_per_frame;

    /* Read all frames to ensure correct time and bitrate */
    for (frames = 0; /* */; frames++)
    {
        fill_b_buffer(b);

        if (b->bytes_into_buffer > 7)
        {
            /* check syncword */
            if (!((b->buffer[0] == 0xFF)&&((b->buffer[1] & 0xF6) == 0xF0)))
                break;

            if (frames == 0)
                samplerate = adts_sample_rates[(b->buffer[2]&0x3c)>>2];

            frame_length = ((((unsigned int)b->buffer[3] & 0x3)) << 11)
                | (((unsigned int)b->buffer[4]) << 3) | (b->buffer[5] >> 5);

            t_framelength += frame_length;

            if (frame_length > b->bytes_into_buffer)
                break;

            advance_buffer(b, frame_length);
        } else {
            break;
        }
    }

    frames_per_sec = (float)samplerate/1024.0f;
    if (frames != 0)
        bytes_per_frame = (float)t_framelength/(float)(frames*1000);
    else
        bytes_per_frame = 0;
    *bitrate = (int)(8. * bytes_per_frame * frames_per_sec + 0.5);
    if (frames_per_sec != 0)
        *length = (float)frames/frames_per_sec;
    else
        *length = 1;

    return 1;
}

uint32_t read_callback(void *user_data, void *buffer, uint32_t length)
{
    return fread(buffer, 1, length, (FILE*)user_data);
}

uint32_t seek_callback(void *user_data, uint64_t position)
{
    return fseek((FILE*)user_data, position, SEEK_SET);
}


int init_file_mp4(char *fname)
{
    /* initialise the callback structure */
    mp4cb = (mp4ff_callback_t *)malloc(sizeof(mp4ff_callback_t));
	if( !mp4cb ) return -7;

    mp4File = fopen(fname, "rb");
	if( !mp4File ) return -8;
    mp4cb->read = read_callback;
    mp4cb->seek = seek_callback;
    mp4cb->user_data = mp4File;
	
	hDecoder = NeAACDecOpen();
	if( hDecoder == NULL ) return -9;

    /* Set configuration */
    config = NeAACDecGetCurrentConfiguration(hDecoder);
    config->outputFormat = FAAD_FMT_16BIT;
    config->downMatrix   = 1; // 5.1 to 2;
    //config->dontUpSampleImplicitSBR = 1;
    NeAACDecSetConfiguration(hDecoder, config);
	
	infile = mp4ff_open_read(mp4cb);
    if (!infile)
    {
        /* unable to open file */
        return -10;
    }

    if ((track = GetAACTrack(infile)) < 0)
    {
        NeAACDecClose(hDecoder); hDecoder = NULL;
        mp4ff_close(infile);     infile = NULL;
        SAFE_FREE(mp4cb);		
        fclose(mp4File);		 mp4File = NULL;
        return -11;
    }
	
	buffer = NULL;
    buffer_size = 0;
    mp4ff_get_decoder_config(infile, track, &buffer, &buffer_size);

	long unsigned int b_size = buffer_size, freq;
	unsigned char chn;
    if(NeAACDecInit2(hDecoder, buffer, b_size,
                    &freq, &chn) < 0)
    {
        /* If some error initializing occured, skip the file */
        NeAACDecClose(hDecoder); hDecoder = NULL;
        mp4ff_close(infile);     infile = NULL;
        SAFE_FREE(mp4cb);		
        fclose(mp4File);		 mp4File = NULL;
        return -12;
    }

    if (buffer)
    {
        if (NeAACDecAudioSpecificConfig(buffer, buffer_size, &mp4ASC) >= 0)
        {
            if (mp4ASC.frameLengthFlag == 1)  ;
            if (mp4ASC.sbr_present_flag == 1) ;
        }
		
        free(buffer); buffer = NULL;
    }

	/* GET FILE INFO */
    {
        char *tag = NULL, *item = NULL;
        int k, j;
        char *ot[6] = { "NULL", "MAIN AAC", "LC AAC", "SSR AAC", "LTP AAC", "HE AAC" };
/*
        long samples = mp4ff_num_samples(infile, track);
        double f = 1024.0;
        double seconds;
        if (mp4ASC.sbr_present_flag == 1)
        {
            f = f * 2.0;
        }
        seconds = (double)samples*(f-1.0)/(double)mp4ASC.samplingFrequency;
*/
        long track_length = mp4ff_get_track_duration(infile, track);
        long mseconds = (long)((double)track_length*1000.0 / (double)mp4ff_time_scale(infile, track) + 0.5);

		plugin.channels   = mp4ASC.channelsConfiguration;
		plugin.freq = mp4ASC.samplingFrequency;
        plugin.length     = (int)(mseconds);
		
        int x = 0;
		mp4ff_meta_get_title(infile, &tag);
		if( tag ) {
            fieldname[x] = "Title";
            fielddata[x++] = string(tag);
			free(tag); tag = NULL;
		}
		
		mp4ff_meta_get_artist(infile, &tag);
		if( tag ) {
            fieldname[x] = "Artist";
            fielddata[x++] = string(tag);
			free(tag); tag = NULL;
		}
		
		mp4ff_meta_get_album(infile, &tag);
		if( tag ) {
            fieldname[x] = "Album";
            fielddata[x++] = string(tag);
			free(tag); tag = NULL;
		}
		
		mp4ff_meta_get_date(infile, &tag);
		if( tag ) {
            fieldname[x] = "Date";
            fielddata[x++] = string(tag);
			free(tag); tag = NULL;
		}

        if( !x ) {
            fieldname[x] = "File";
            fielddata[x++] = strchr(fname, '/') ? strrchr(fname, '/')+1 : fname;
        }

        plugin.nfields = x;
        plugin.fieldname = fieldname;
        plugin.fielddata = fielddata;
    }
	
	sampleId = 0;
	numSamples = mp4ff_num_samples(infile, track);
	
    samples_done = 0;
	writeposbytes = 0;
    readposbytes  = 0;
	buffer = NULL;
	
	playing = 1;
	return 1;

}

static int init_file(char *fname)
{
    playing = 0;
#ifdef A320
    plugin.clockfreq = 336;
#else
    plugin.clockfreq = 200;
#endif

   	mp4file = 0;
	unsigned char header[8];
    FILE *in = fopen(fname, "rb");
    if (in == NULL) return -1;
    
    fread(header, 1, 8, in);
    fclose(in);
	
    if (header[4] == 'f' && header[5] == 't' && header[6] == 'y' && header[7] == 'p')
        mp4file = 1;
		
	if( mp4file ) {
        plugin.fill_buffer = fill_buffer_mp4;
		return init_file_mp4(fname);
	}

    plugin.fill_buffer = fill_buffer;
    memset(&b, 0, sizeof(aac_buffer));
    b.infile = fopen(fname, "rb");
    if (!b.infile)
    {
        fprintf(stderr, "faadplugin:init_file - Error opening file: %s\n", fname);
        return -1;
    }
    fseek(b.infile, 0, SEEK_END);
    int fileread = ftell(b.infile);
    fseek(b.infile, 0, SEEK_SET);

    if (!(b.buffer = (unsigned char*)malloc(FAAD_MIN_STREAMSIZE*MAX_CHANNELS)))
    {
        fprintf(stderr, "faadplugin:init_file - Memory allocation error\n");
        return -1;
    }

    int bread = fread(b.buffer, 1, FAAD_MIN_STREAMSIZE*MAX_CHANNELS, b.infile);
    b.bytes_into_buffer = bread;
    b.bytes_consumed = 0;
    b.file_offset = 0;
    if (bread != FAAD_MIN_STREAMSIZE*MAX_CHANNELS)
        b.at_eof = 1;

    tagsize = 0;
    if (!memcmp(b.buffer, "ID3", 3))
    {
        fprintf(stderr, "faadplugin:init_file - Skipping ID3-tags\n");
        tagsize = (b.buffer[6] << 21) | (b.buffer[7] << 14) |
            (b.buffer[8] <<  7) | (b.buffer[9] <<  0);
        tagsize += 10;
        for (int tagsizetmp = tagsize; tagsizetmp > 0; )
        {
            int tagsizetmp2 = tagsizetmp <= b.bytes_into_buffer ? tagsizetmp 
                : b.bytes_into_buffer;
            tagsizetmp -= tagsizetmp2;
            advance_buffer(&b, tagsizetmp2);
            fill_b_buffer(&b);
        }
    }
    int x = parse_id3v1(fname, fieldname, fielddata, 4);
    if (!x)
    {
        fieldname[x] = "File";
        fielddata[x++] = strchr(fname, '/') ? strrchr(fname,'/')+1 : fname;
    }
    fieldname[x] = "Format";
    fielddata[x++] = is_ext(fname, ".aac") ?  "AAC" : "MP4";
    plugin.nfields = x;
    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;

    hDecoder = NeAACDecOpen();

    config = NeAACDecGetCurrentConfiguration(hDecoder);
    //config->defSampleRate = 44100; // For RAW files. Which is.. ?
    config->defObjectType = LC;
    config->outputFormat = FAAD_FMT_16BIT; // double check this
    config->downMatrix = 1; // 5.1 to 2
    config->useOldADTSFormat = 0;
    //config->dontUpSampleImplicitSBR = 1;
    NeAACDecSetConfiguration(hDecoder, config);

    /* get AAC infos for printing */
    int header_type = 0;
    if ((b.buffer[0] == 0xFF) && ((b.buffer[1] & 0xF6) == 0xF0))
    {
        float length;
        adts_parse(&b, &bitrate, &length);
        fseek(b.infile, tagsize, SEEK_SET);
        plugin.length = (int)(length*1000);

        bread = fread(b.buffer, 1, FAAD_MIN_STREAMSIZE*MAX_CHANNELS, b.infile);
        b.at_eof = bread != FAAD_MIN_STREAMSIZE*MAX_CHANNELS ? 1 : 0;
        b.bytes_into_buffer = bread;
        b.bytes_consumed    = 0;
        b.file_offset       = tagsize;

        header_type = 1;
    } else if (memcmp(b.buffer, "ADIF", 4) == 0) {
        int skip_size = (b.buffer[4] & 0x80) ? 9 : 0;
        bitrate = ((unsigned int)(b.buffer[4 + skip_size] & 0x0F)<<19) |
            ((unsigned int)b.buffer[5 + skip_size]<<11) |
            ((unsigned int)b.buffer[6 + skip_size]<<3) |
            ((unsigned int)b.buffer[7 + skip_size] & 0xE0);

        float length = (float)fileread;
        if (fileread != 0)
        {
            length = length*8.f/bitrate + 0.5f;
        }
        plugin.length = (int)length; 
        bitrate = (int)((float)bitrate/1000.0f + 0.5f);

        header_type = 2;
    }
    printf("length = %d, bitrate = %d\n",plugin.length,bitrate);
    fill_b_buffer(&b);

    unsigned char chn;
    long unsigned int freq;
    if ((bread = NeAACDecInit(hDecoder, b.buffer,
        b.bytes_into_buffer, &freq, &chn)) < 0)
    {
        fprintf(stderr, "faadplugin:init_file - Error initializing decoder library.\n");
        close();
        return -1;
    }
    plugin.channels = chn;
    plugin.freq     = freq;

    advance_buffer(&b, bread);
    fill_b_buffer(&b);

    samples_done  = 0;
    writeposbytes = 0;
    readposbytes  = 0;
    playing = 1;
    return 0;
}

// FAAD file buffering routines 
static int foo = 1;
static int fill_b_buffer(aac_buffer *b)
{
    int bread;

    if (b->bytes_consumed > 0)
    {
        //printf("Hej 1!\n");
        if (b->bytes_into_buffer)
        {
            if(foo)
            {
            printf("bytes_consumed = %d, bytes_into_buffer = %d\n",
                b->bytes_consumed, b->bytes_into_buffer);
                foo = 0;
            }
            memmove((void*)b->buffer, (void*)(b->buffer + b->bytes_consumed),
                b->bytes_into_buffer);
            //printf("Hej 3!\n");
        }

        //printf("Hej 4!\n");
        if (!b->at_eof)
        {
            bread = fread((void*)(b->buffer + b->bytes_into_buffer), 1,
                b->bytes_consumed, b->infile);

            if (bread != b->bytes_consumed)
                b->at_eof = 1;

            b->bytes_into_buffer += bread;
        }

        b->bytes_consumed = 0;

        if (b->bytes_into_buffer > 3)
        {
            if (memcmp(b->buffer, "TAG", 3) == 0)
                b->bytes_into_buffer = 0;
        }
        if (b->bytes_into_buffer > 11)
        {
            if (memcmp(b->buffer, "LYRICSBEGIN", 11) == 0)
                b->bytes_into_buffer = 0;
        }
        if (b->bytes_into_buffer > 8)
        {
            if (memcmp(b->buffer, "APETAGEX", 8) == 0)
                b->bytes_into_buffer = 0;
        }
    }

    return 1;
}

static void advance_buffer(aac_buffer *b, int bytes)
{
    b->file_offset += bytes;
    b->bytes_consumed = bytes;
    b->bytes_into_buffer -= bytes;
}

int GetAACTrack(mp4ff_t *infile)
{
    /* find AAC track */
    int i, rc;
    int numTracks = mp4ff_total_tracks(infile);

    for (i = 0; i < numTracks; i++)
    {
        unsigned char *buff = NULL;
        unsigned int buff_size = 0;
        mp4AudioSpecificConfig mp4ASC;

        mp4ff_get_decoder_config(infile, i, &buff, &buff_size);

        if (buff)
        {
            rc = NeAACDecAudioSpecificConfig(buff, buff_size, &mp4ASC);
            free(buff);

            if (rc < 0)
                continue;
            return i;
        }
    }

    /* can't decode this */
    return -1;
}

static int close()
{
    if( hDecoder ) {
        NeAACDecClose(hDecoder);
        hDecoder = NULL;
    }

    if( mp4file ) {
        if( infile ) {
			mp4ff_close(infile);
			infile = NULL;
		}
        
		SAFE_FREE(mp4cb);
        if( mp4File ) {
			fclose(mp4File);
			mp4File = NULL;
		}
		
		sampleId = 0;
    } else {
        if (b.infile)
        {
            fclose(b.infile);
            b.infile = NULL;
        }
        if (b.buffer)
        {
            free(b.buffer);
            b.buffer = NULL;
        }
    }

    tmpbuf = NULL;

    for(int i = 0; i < 5; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }
    playing = plugin.length = plugin.nfields = 0;
    return 0;
}

static int fill_buffer(signed short *dest, int len)
{
    if (playing)
    {
        int writtenbytes = 0;
        for (;;)
        {
            int bytesleft = len - writtenbytes;
            int bufbytes  = writeposbytes - readposbytes;
            //printf("len = %04d - written = %04d = %04d, " \
                    "read = %04d - write = %04d = %04d\n",\
                len,writtenbytes,bytesleft,readposbytes,writeposbytes,bufbytes);
            if (bytesleft <= bufbytes)
            {
                memcpy((char*)dest + writtenbytes, (char*)tmpbuf + readposbytes, bytesleft);
                readposbytes += bytesleft;
                writtenbytes = len;
                break;
            }
            else
            {
                if( tmpbuf ) {
                    memcpy((char*)dest + writtenbytes, (char*)tmpbuf + readposbytes, bufbytes);
                    writtenbytes += bufbytes;
                }

                readposbytes = writeposbytes = 0;

                while (!writeposbytes)
                {
                    tmpbuf = NeAACDecDecode(hDecoder, &frameInfo, b.buffer, 
                        b.bytes_into_buffer);
                    if (frameInfo.error > 0)
                    {
                        fprintf(stderr, "faadplugin:fill_buffer - Error: %s\n",
                                NeAACDecGetErrorMessage(frameInfo.error));
                        playing = 0;
                        break;
                    }
                    writeposbytes = frameInfo.samples<<1;
                    advance_buffer(&b, frameInfo.bytesconsumed);
                    if (b.at_eof)
                        break;
                    fill_b_buffer(&b);
                    //printf("%d, %d, %d, %d\n",frameInfo.bytesconsumed,frameInfo.samples, \
                        frameInfo.channels,frameInfo.samplerate);
                }
                //printf("filling %d (len = %d)!\n",writeposbytes,len);
                if (!writeposbytes)
                    break;
            }
        }
        samples_done += writtenbytes/(2*plugin.channels);
        //printf("written = %d\n",writtenbytes);
        return writtenbytes;
    }
    return 0;
}

static int fill_buffer_mp4(signed short *dest, int len)
{
	// Finished decoding?
	if( sampleId >= numSamples ) {
		playing = 0;
		return 0;
	}
	
    if (playing)
    {
        int writtenbytes = 0;
        for (;;)
        {
            int bytesleft = len - writtenbytes;
            int bufbytes  = writeposbytes - readposbytes;
            //printf("len = %04d - written = %04d = %04d, " \
                    "read = %04d - write = %04d = %04d\n",\
                len,writtenbytes,bytesleft,readposbytes,writeposbytes,bufbytes);
            if (bytesleft <= bufbytes)
            {
                memcpy((char*)dest + writtenbytes, (char*)tmpbuf + readposbytes, bytesleft);
                readposbytes += bytesleft;
                writtenbytes = len;
                break;
            }
            else
            {
                if( tmpbuf ) {
					memcpy((char*)dest + writtenbytes, (char*)tmpbuf + readposbytes, bufbytes);
					writtenbytes += bufbytes;
				}
				
                readposbytes = writeposbytes = 0;

                while (!writeposbytes)
                {
				    /* get acces unit from MP4 file */
					buffer = NULL;
					buffer_size = 0;

					int dur = mp4ff_get_sample_duration(infile, track, sampleId);
					int rc = mp4ff_read_sample(infile, track, sampleId, &buffer,  &buffer_size);
					if (rc == 0) {
						fprintf(stderr, "Reading from MP4 file failed.\n");
						playing = 0;
						return 0;
					}

					sampleId ++;
					tmpbuf = NeAACDecDecode(hDecoder, &frameInfo, buffer, buffer_size);
		
                    if (frameInfo.error > 0)
                    {
                        fprintf(stderr, "faadplugin:fill_buffer - Error: %s\n",
                                NeAACDecGetErrorMessage(frameInfo.error));
                        playing = 0;
                        return 0;
                    }

                    if( buffer ) free(buffer);
					
                    writeposbytes = frameInfo.samples<<1;
                }
                
                if (!writeposbytes)
                    break;
            }
        }
        samples_done += writtenbytes/(2*plugin.channels);
        return writtenbytes;
    }
    return 0;
}


static int set_position(int msecs, int subtune)
{
    if (playing && msecs)
    {
        int seekbytes = (int)((double)bitrate*1024*msecs/(8*1000)); 
        printf("Seeking : %d : %d : %d\n",seekbytes,bitrate,msecs);
        int curposfile = ftell(b.infile);
        int curpos    = curposfile - FAAD_MIN_STREAMSIZE*MAX_CHANNELS + 
                                                b.bytes_into_buffer;
        if (seekbytes < 0 && -seekbytes > curposfile - tagsize)
        {
            fseek(b.infile,tagsize,SEEK_SET);
            b.file_offset = tagsize;
        }
        else
        {
            fseek(b.infile, seekbytes, SEEK_CUR);
            b.file_offset += seekbytes;
        }
        b.bytes_into_buffer = fread(b.buffer, 1, 
                FAAD_MIN_STREAMSIZE*MAX_CHANNELS, b.infile);
        b.bytes_consumed = 0;
        if (b.bytes_into_buffer != FAAD_MIN_STREAMSIZE*MAX_CHANNELS)
            b.at_eof = 1;

        for (int i = 0; i < b.bytes_into_buffer-1; i++)
        {
            if ((b.buffer[i] == 0xFF) && ((b.buffer[i+1] & 0xF6) == 0xF0))
            {
                b.bytes_consumed = i;
                break;
            }
        }
        int newpos = ftell(b.infile) - FAAD_MIN_STREAMSIZE*MAX_CHANNELS +
                                            b.bytes_into_buffer;

        readposbytes = writeposbytes = 0;
        printf("newpos = %d - oldpos = %d  => %d\n",newpos,curpos,newpos-curpos);
        printf(" ==> %d\n",(int)(8*1000.0*(newpos-curpos)/(1024*bitrate)));
        return (int)(8*1000.0*(newpos-curpos)/(1024*bitrate));
    }
    return 0;
}

static int can_handle(const char *name)
{
    return is_ext(name, ".aac") || is_ext(name, ".mp4") || is_ext(name, ".m4a"); 
}

extern "C" {

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN faad_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
    memset(&plugin, 0, sizeof(plugin));
    plugin.plugname = "FAAD";
    //plugin.init_data = init_data;
    plugin.init_file = init_file;
    //plugin.request_format = NULL;
    plugin.set_position = set_position;
    plugin.fill_buffer = fill_buffer;
    plugin.can_handle = can_handle;
    plugin.close = close;
    plugin.tune = 0;
    plugin.subtunes = 1;
    plugin.replaygain = 1;
    return &plugin;
}

}

