/*
    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <string>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "adplug/adplug.h"
#include "adplug/emuopl.h"
#include "../plugin.h"
#include "../util.h"

#define SAMPLERATE (44100>>1)

static int playing = 0, samples_left, timer;
static struct sound_plugin plugin;

//         (FREQ, BIT16, STEREO)
CEmuopl opl(44100,true,false);
CPlayer *p;

static int dbloaded = 0;
CAdPlugDatabase	*mydb;

using namespace std;
static string fieldname[4];
static string fielddata[4];
static string filename;

static int init_file(char *fname)
{
    if( !dbloaded ) {
        mydb = new CAdPlugDatabase;
        mydb->load("adplug.db");
	CAdPlug::set_database(mydb);
        dbloaded = 1;
    }

    playing = 0;
    plugin.tune = 0;
    plugin.nfields = 0;
#ifdef A320
    plugin.clockfreq = 372;
#else
    plugin.clockfreq = 200; 
#endif
    p = CAdPlug::factory(fname, &opl);
    if (!p)
    {
        fprintf(stderr, "aplugin::init_file - Couldn't load file!\n");
        return -1; 
    }
    filename = strrchr(fname, '/')+1;
    int x = 0;
    if (p->gettitle().length())
    {
        fieldname[x] = "Title";
        fielddata[x] = p->gettitle();
    }
    else
    {
        fieldname[x] = "File";
        fielddata[x] = filename;
    }
    x++;
    if (p->getauthor().length())
    {
        fieldname[x] = "Artist";
        fielddata[x] = p->getauthor();
        x++;
    }
    if (p->getdesc().length())
    {
        fieldname[x] = "Description";
        fielddata[x] = p->getdesc();
        x++;
    }
    if (p->gettype().length())
    {
        fieldname[x] = "Format";
        fielddata[x] = p->gettype();
        x++;
    }
    plugin.fieldname = fieldname;
    plugin.fielddata = fielddata;
    plugin.nfields   = x;
    plugin.length    = p->songlength();
    plugin.subtunes  = p->getsubsongs();
    playing = 1;
    p->update(); // First step
    samples_left = (int)(2 * plugin.freq / p->getrefresh());
    timer = 0;
    return 0;
}

static int close()
{
    if (p) delete(p);
    p = NULL;
    for(int i = 0; i < plugin.nfields; i++)
    {
        fieldname[i].clear();
        fielddata[i].clear();
    }
    playing = plugin.tune = plugin.subtunes = plugin.length = plugin.nfields = 0;
    return 0;
}

static int fill_buffer(signed short *dest, int len)
{
    if (playing)
    {
        int real_len = len>>1;
        int write_pos = 0;
        for(;;)
        {
            if (samples_left < real_len)
            {
                opl.update(&dest[write_pos],samples_left);
                write_pos += samples_left;
                real_len -= samples_left;
                if (!p->update())
                {// I probably *should* stop here? But the music often
                 // continues and they are almost all cut off to soon i think
                //    playing = 0;
                //    return write_pos;
                }
                samples_left = (int)(2 * plugin.freq / p->getrefresh());
            }
            else
            {
                opl.update(&dest[write_pos],real_len);
                write_pos += real_len;
                samples_left -= real_len;
                timer += real_len;
                return len;
            }
        }
    }
    return 0;
}

static int set_position(int seconds, int subtune)
{
    if (!playing)
        return 0;
    if (!seconds)
    {
        p->rewind(subtune);
        int x = 0;
        if (p->gettitle().length())
        {
            fieldname[x] = "Title";
            fielddata[x] = p->gettitle();
        }
        else
        {
            fieldname[x] = "File";
            fielddata[x] = filename;
        }
        x++;
        if (p->getauthor().length())
        {
            fieldname[x] = "Author";
            fielddata[x] = p->getauthor();
        x++;
        }
        if (p->getdesc().length())
        {
            fieldname[x] = "Description";
            fielddata[x] = p->getdesc();
        x++;
        }
        if (p->gettype().length())
        {
            fieldname[x] = "Format";
            fielddata[x] = p->gettype();
            x++;
        }
        plugin.tune   = subtune;
        plugin.length = p->songlength();
        return subtune;
    }
    else
    {
        // It seems to always seek to the start in some songs.. ? Looks fine here
        p->seek((int)((double)timer/plugin.freq*1000) + seconds*1000);
        timer += seconds*plugin.freq;
        return seconds*10;
    }
}

static int can_handle(const char *name)
{
    return (is_ext(name, ".MID") || is_ext(name, ".MIDI") || is_ext(name, ".A2M") || 
            is_ext(name, ".ADL") || is_ext(name, ".AMD") || is_ext(name, ".BAM") || 
            is_ext(name, ".CFF") || is_ext(name, ".CMF") || is_ext(name, ".D00") || 
            is_ext(name, ".DFM") || is_ext(name, ".DMO") || is_ext(name, ".DRO") || 
            is_ext(name, ".DTM") || is_ext(name, ".HSC") || is_ext(name, ".HSP") || 
            is_ext(name, ".JBM") ||
            is_ext(name, ".IMF") || is_ext(name, ".KSM") || is_ext(name, ".LAA") || 
            is_ext(name, ".LDS") || is_ext(name, ".M")   || is_ext(name, ".MAD") || 
            is_ext(name, ".MKJ") || is_ext(name, ".MSC") || is_ext(name, ".MTK") || 
            is_ext(name, ".RAD") || is_ext(name, ".RAW") || is_ext(name, ".RIX") || 
            is_ext(name, ".ROL") || is_ext(name, ".S3M") || is_ext(name, ".SA2") || 
            is_ext(name, ".SAT") || is_ext(name, ".SCI") || is_ext(name, ".SNG") || 
            is_ext(name, ".WLF") ||
            is_ext(name, ".XAD") || is_ext(name, ".XMS") || is_ext(name, ".XSM"));
}

extern "C" {

#ifndef INIT_SOUND_PLUGIN
#define INIT_SOUND_PLUGIN adplug_init_sound_plugin
#endif

struct sound_plugin *INIT_SOUND_PLUGIN()
{
    memset(&plugin, 0, sizeof(plugin));
    plugin.plugname = "adplug";
    //plugin.init_data = init_data;
    plugin.init_file = init_file;
    //plugin.request_format = NULL;
    plugin.set_position = set_position;
    plugin.fill_buffer = fill_buffer;
    plugin.can_handle = can_handle;
    plugin.close = close;
    plugin.channels = 2;
    plugin.freq = SAMPLERATE;
    plugin.replaygain = 1;
    return &plugin;
}

}

