/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "ZipFileList.h"

#include <vector>

#include "zip/zipint.h"


extern "C" { void busy(int); }

static ZipDir *root;

// Return positive if dirpart of d and f are same
int cmpdir(const char *d, const char *f)
{
	while(*f == *d)
	{
		f++;
		d++;
	}
	// If whole dirname matched were ok
	if(!*d)
		return 1;
	return 0;
}

// Return positive if dirpart of d and f are same
int cmpdir2(const char *d, const char *f)
{
	// Compare until end of one string or inequality
	while((*f == *d) && *d && *f)
	{
		f++;
		d++;
	}

	// If file is longer, check for slashes
	while(*f && *f != '/') f++;

	// If whole dir matched and no slashes in rest of file, its all good
	if(!(*d) && *f != '/')
		return 1;

	return 0;
}

static ZipDir *find_dir(ZipDir *parent, char *name)
{
	ZipDir *rc = NULL;
	for(int i=0; i<parent->dirs.size() && !rc; i++)
		if(cmpdir2(parent->dirs[i]->filename, name))
			return parent->dirs[i];
		else
			rc = find_dir(parent->dirs[i], name);
	return rc;
}

ZipFileList::ZipFileList(const char *zipfile, int (*fcb)(void *, const char *), void *data) :
    filter_cb(fcb),
    cb_data(data)
{
	char *lastpardir = "";
	ZipDir *lastdir = NULL;
	busy(1);
	strcpy(zipname, zipfile);
	zipf = zip_open(zipfile, 0, NULL);
	busy(1);
	count = zip_get_num_files(zipf);
	marked = count ? 0 : -1;
    root = new ZipDir("");
	char pardir[128];
	for(int i=0; i<count; i++)
	{
		struct zip_dirent *de = &zipf->cdir->entry[i];
		if((i & 0x1FF) == 0x1FF)
			busy(1);
		if(de->ext_attrib & 0x10)
		{
			strcpy(pardir, de->filename);
			pardir[strlen(pardir)-1] = 0;
			char *end = strrchr(pardir, '/');
			if(end)
				end[1] = 0;

			ZipDir *zdir = new ZipDir(de->filename);
			ZipDir *parent = find_dir(root, pardir);
			if(!parent)
				parent = root;
			parent->dirs.push_back(zdir);
			lastdir = zdir;
			lastpardir = de->filename;
		}
		else
		{
			ZipDir *parent = NULL;
			if(lastdir)
			{
				if(cmpdir(lastpardir, de->filename))
					parent = lastdir;
				else
					parent = find_dir(root, de->filename);
			}

			if(!parent)
				parent = root;

			parent->files.push_back(i);
		}

	}
	//strcpy(curdir, "");
	Enter("");
	dirty = true;
	curdir = root;
	busy(0);
}

char *ZipFileList::GetCurDir() { return curdir ? curdir->filename : NULL; }

void ZipFileList::Enter(char *dirname)
{
	ZipDir *zd;
	if(!strlen(dirname) || strcmp(dirname, "/") == 0)
		curdir = root;
	else
	if(zd = find_dir(root, dirname))
		curdir = zd;
#ifdef ZIPFILELISTENTERDEBUG
    fprintf(stderr, "ZipFileList::Enter - dirname = %s\n",dirname);
#endif
	filerefs.clear();
	int i;
	for(i=0; i<curdir->dirs.size(); i++)
	{
		FileData fd;
		char *ptr = curdir->dirs[i]->filename;
		char *end = &ptr[strlen(ptr)-2];
		while(end >= ptr && *end != '/')
			end--;
		end++;

		fd.name = end;//strrchr(curdir->dirs[i]->filename, '/');
		fd.size = -1;
        fd.time = -1;
        fd.track = -1;
		fd.index = -1; // Not meant for unzipping
		filerefs.push_back(fd);
	}

	struct zip_stat zs;
	dircount = i;

	for(i=0; i<curdir->files.size(); i++)
	{
		zip_stat_index(zipf, curdir->files[i], 0, &zs);

		char *s = (char *)strrchr(zs.name, '/');
		if(!s)
			s = (char *)zs.name;
		else
			s++;
#ifdef ZIPFILELISTENTERDEBUG
        fprintf(stderr, "ZipFileList::Enter - Zipped file: %s\n", s);
#endif
        if (!filter_cb(cb_data,s))
            continue;
		FileData fd;
		fd.name = string(s);
		fd.path = string(zipname);
		fd.size = zs.size;
        fd.time = -1;
        fd.track = -1;
        fd.index = zs.index;
		filerefs.push_back(fd);
	}
}

int ZipFileList::Enter(int index)
{
	char *old = NULL;
	if(index == -1)
	{
		char tmp[128];
		if(!strlen(curdir->filename))
			return -1;

		old = curdir->filename;
		strcpy(tmp, curdir->filename);
		tmp[strlen(tmp)-1] = 0;
		char *end = strrchr(tmp, '/');
		if(end)
			end[1] = 0;
		else
			*tmp = 0;
		Enter(tmp);

		for(int i=0; i<curdir->dirs.size(); i++)
		{
			if(strcmp(curdir->dirs[i]->filename, old) == 0)
				return i;
		}
		return 0;
	}
	else
	if(index < dircount)
	{
		Enter(curdir->dirs[index]->filename);
		return 0;
	}

	return index;
}


void *ZipFileList::GetZipReference(int index)
{
	struct zip *z = (struct zip *)malloc(sizeof(struct zip));
	memcpy(z, zipf, sizeof(struct zip));
	int idx = curdir->files[index-dircount];

	z->cdir = (struct zip_cdir *)malloc(sizeof(struct zip_cdir));
	z->cdir->nentry = 1;
	z->cdir->entry = (struct zip_dirent *)malloc(sizeof(struct zip_dirent));
	memcpy(z->cdir->entry, &zipf->cdir->entry[idx], sizeof(struct zip_dirent));
	z->cdir->entry->filename = strdup(zipf->cdir->entry[idx].filename);	
	z->cdir->entry->comment = NULL;
	z->cdir->entry->ext_attrib = 0; //NULL;
	z->zn = strdup(zipf->zn);
	z->nentry = 1;

	return (void *)z;
}

