/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "ZipFileList.h"

#include <vector>

extern "C" { void busy(int); }

static ZipDir *root = NULL;

static void error( fex_err_t err )
{
	if ( err != NULL )
	{
		const char* str = fex_err_str( err );
		fprintf( stderr, "Error: %s\n", str );
		exit( EXIT_FAILURE );
	}
}

// Return positive if dirpart of d and f are same
int cmpdir(const char *d, const char *f)
{
	while(*f == *d)
	{
		f++;
		d++;
	}
	// If whole dirname matched were ok
	if(!*d)
		return 1;
	return 0;
}

// Return positive if dirpart of d and f are same
int cmpdir2(const char *d, const char *f)
{
	// Compare until end of one string or inequality
	while((*f == *d) && *d && *f)
	{
		f++;
		d++;
	}

	// If file is longer, check for slashes
	while(*f && *f != '/') f++;

	// If whole dir matched and no slashes in rest of file, its all good
	if(!(*d) && *f != '/')
		return 1;

	return 0;
}

static ZipDir *find_dir(ZipDir *dir, const char *name)
{
	ZipDir *rc = NULL;
	for(int i=0; i<dir->dirs.size() && !rc; i++) {
		if(cmpdir2(dir->dirs[i]->filename.c_str(), name))
			return dir->dirs[i];
		else
			rc = find_dir(dir->dirs[i], name);
    }
	return rc;
}

int strcount(const char *s, char c)
{
    int i = 0;
    const char *p = s;
    while( *p ) {
       if( *p++ == c ) i++;
    }
    return i;
}

static ZipDir *get_fix_parent(char *dir)
{
    if( strcount(dir, '/') <= 1 ) return root;
 
    char pardir[128];   
    strcpy(pardir, dir);
	pardir[strlen(pardir)-1] = 0;
	char *end = strrchr(pardir, '/');
	if(end)
		end[1] = 0;

    ZipDir *parent = find_dir(root, pardir);
    if( !parent ) {
        ZipDir *pdir = new ZipDir(pardir);
        parent = get_fix_parent(pardir);
        parent->dirs.push_back(pdir);
        return pdir;
    }

    return parent;
}

ZipFileList::ZipFileList(const char *zipfile, int (*fcb)(void *, const char *), void *data) :
    filter_cb(fcb),
    cb_data(data)
{
	char *lastpardir = "";
	ZipDir *lastdir = NULL;

  	busy(1);
	strcpy(zipname, zipfile);
    error( fex_open( &fex, zipfile ) );

   	busy(1);
    // TODO - Should marked always be 0?
	//marked = count ? 0 : -1;
    marked = 0;

    root = new ZipDir("");

	char dir[128];
    int i = 0;
    while ( !fex_done( fex ) ) {	
   		if((i & 0x1FF) == 0x1FF)
			busy(1);

        if( !strcount(fex_name( fex ), '/') ) {
            /*fprintf(stderr, "ZipFileList - Adding file %d [%s]\n", i, root->filename.c_str());*/
            root->files.push_back(i);
            error( fex_next( fex ) );
            i++;
            continue;            
        }

        strncpy(dir, fex_name( fex ), 128);
        char *end = strrchr(dir, '/');
        if(end) end[1] = 0;

        ZipDir *zdir = find_dir(root, dir);
        if( zdir != NULL ) {
            /*fprintf(stderr, "ZipFileList - Adding file %d [%s]\n", i, zdir->filename.c_str());*/
            zdir->files.push_back(i);
            error( fex_next( fex ) );
            i++;
            continue;
        }

        // Must create and/or add missing entries in tree
        zdir = new ZipDir(dir);
        /*fprintf(stderr, "ZipFileList - Adding file %d [%s]\n", i, zdir->filename.c_str());*/ 
        zdir->files.push_back(i);

        ZipDir *parent = get_fix_parent(dir);
        /*fprintf(stderr, "ZipFileList - Adding dir %s [%s]\n", dir, parent->filename.c_str());*/ 
	    parent->dirs.push_back(zdir);

        error( fex_next( fex ) );
        i++;
	} 

    // Close fex, will reopen when necessary ...
    fex_close( fex );

	//strcpy(curdir, "");
	Enter("");
	dirty = true;
	curdir = root;
  	busy(0);
}

void free_dir_tree(ZipDir *dir)
{
    for(int i = 0; i < dir->dirs.size(); i++) {
        free_dir_tree(dir->dirs[i]);
        delete dir->dirs[i];
    }

    dir->dirs.clear();
    dir->files.clear();
}

ZipFileList::~ZipFileList()
{
    if( root == NULL ) return;
    free_dir_tree(root);
}

const char *ZipFileList::GetCurDir() { return curdir ? curdir->filename.c_str() : NULL; }

void ZipFileList::Enter(const char *dirname)
{
	ZipDir *zd;
	if(!strlen(dirname) || strcmp(dirname, "/") == 0)
		curdir = root;
	else
	if(zd = find_dir(root, dirname))
		curdir = zd;
#ifdef ZIPFILELISTENTERDEBUG
    fprintf(stderr, "ZipFileList::Enter - dirname = %s\n",dirname);
#endif
	filerefs.clear();
	int i;
	for(i=0; i<curdir->dirs.size(); i++)
	{
		FileData fd;
		const char *ptr = curdir->dirs[i]->filename.c_str();
		const char *end = &ptr[strlen(ptr)-2];
		while(end >= ptr && *end != '/')
			end--;
		end++;

		fd.name = end;//strrchr(curdir->dirs[i]->filename, '/');
		fd.size = -1;
        fd.time = -1;
        fd.track = -1;
		fd.index = -1; // Not meant for unzipping
		filerefs.push_back(fd);
	}

	dircount = i;

    // If curdir is empty just return ;)
    if( !curdir->files.size() ) return;

    // Reopen fex file
    error( fex_open( &fex, zipname ) );

    i = 0;
    while( !fex_done( fex ) ) {
        if( i == curdir->files[0] ) break;
        error( fex_next( fex ) );
        i++;
    }
    
	for(i=0; i<curdir->files.size(); i++)
	{
        error( fex_stat( fex ) );

        char name[128];
        strncpy(name, fex_name( fex ), 128);
    
		char *s = (char *)strrchr(name, '/');
		if(!s)
			s = (char *)name;
		else
			s++;
#ifdef ZIPFILELISTENTERDEBUG
        fprintf(stderr, "ZipFileList::Enter - Zipped file: %s\n", s);
#endif
        if (!filter_cb(cb_data,s)) {
            error( fex_next( fex ) );
            continue;
        }

		FileData fd;
		fd.name = string(s);
		fd.path = string(zipname);
		fd.size = fex_size( fex );
        fd.time = -1;
        fd.track = -1;
        fd.index = curdir->files[i];
		filerefs.push_back(fd);

        error( fex_next( fex ) );
	}

    // Close fex file
    fex_close( fex );
}

int ZipFileList::Enter(int index)
{
	const char *old = NULL;
	if(index == -1)
	{
		char tmp[128];
		if(!strlen(curdir->filename.c_str()))
			return -1;

		old = curdir->filename.c_str();
		strcpy(tmp, curdir->filename.c_str());
		tmp[strlen(tmp)-1] = 0;
		char *end = strrchr(tmp, '/');
		if(end)
			end[1] = 0;
		else
			*tmp = 0;
		Enter(tmp);

		for(int i=0; i<curdir->dirs.size(); i++)
		{
			if(strcmp(curdir->dirs[i]->filename.c_str(), old) == 0)
				return i;
		}
		return 0;
	}
	else
	if(index < dircount)
	{
		Enter(curdir->dirs[index]->filename.c_str());
		return 0;
	}

	return index;
}


void *ZipFileList::GetZipReference(int index)
{
#if 0
	struct zip *z = (struct zip *)malloc(sizeof(struct zip));
	memcpy(z, zipf, sizeof(struct zip));
	int idx = curdir->files[index-dircount];

	z->cdir = (struct zip_cdir *)malloc(sizeof(struct zip_cdir));
	z->cdir->nentry = 1;
	z->cdir->entry = (struct zip_dirent *)malloc(sizeof(struct zip_dirent));
	memcpy(z->cdir->entry, &zipf->cdir->entry[idx], sizeof(struct zip_dirent));
	z->cdir->entry->filename = strdup(zipf->cdir->entry[idx].filename);	
	z->cdir->entry->comment = NULL;
	z->cdir->entry->ext_attrib = 0; //NULL;
	z->zn = strdup(zipf->zn);
	z->nentry = 1;

	return (void *)z;
#endif
    return NULL;
}

