/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
    */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <vector>

#include "SaveList.h"
#include "util.h"

using namespace std;

//#ifdef _WIN32
const char *BaseName(const char *fname);
//#endif

#ifdef _WIN32
#define STRDUP _strdup
#include <io.h>
struct dirent
{
    char *d_name;
};

typedef struct
{
    struct dirent de;
    struct _finddata_t fd;
    intptr_t handle;
} DIR;

DIR *opendir(char *name)
{
    char tmp[256];
    DIR *dir = (DIR*)malloc(sizeof(DIR));
    dir->de.d_name = NULL;
    sprintf(tmp, "%s\\*", name);
    dir->handle = _findfirst(tmp, &dir->fd);
    return dir;
}

struct dirent *readdir(DIR *dir)
{
    intptr_t rc = dir->handle;
    if(dir->de.d_name)
        rc = _findnext(dir->handle, &dir->fd);
    if(rc == -1)
        return NULL;
    dir->de.d_name = dir->fd.name;
    return &dir->de;
}

void closedir(DIR *dir)
{
    _findclose(dir->handle);
    free(dir);
}
#else
#define STRDUP strdup
#include <dirent.h>
#endif

SaveList::SaveList(char *dirname, int (*fcb)(void *, const char *), void *data) : 
    filter_cb(fcb),
    cb_data(data)
{
    if(dirname)
    {
        strcpy(curdir, dirname);
        Enter(dirname);
    }
    else
    {
        *curdir = 0;
    }

}

void SaveList::AddDirectory(char *dirname, bool recursive)
{
    DIR *dir = opendir(dirname);
    if(dir)
    {
        fprintf(stderr, "SaveList::AddDirectory - Opening %s\n",dirname);
        char tmp[128];
        int count = 0;
        int folders = 0, files = 0;
        for (struct dirent *de; de = readdir(dir);) {
            struct stat ss;
            ss.st_mode = 0;
            sprintf(tmp, "%s%c%s", dirname, SEPARATOR, de->d_name);
            if (stat(tmp, &ss) == 0 && de->d_name[0] != '.') {
                if (ss.st_mode & S_IFDIR)
                    folders++;
                else
                    files++;
            }
        }
        rewinddir(dir);
        char sorted[folders+files][128];
        int k = 0, m = 0;
        for (struct dirent *de; de = readdir(dir);) {
            struct stat ss;
            ss.st_mode = 0;
            sprintf(tmp, "%s%c%s", dirname, SEPARATOR, de->d_name);
            if (stat(tmp, &ss) == 0 && de->d_name[0] != '.') {
                if (ss.st_mode & S_IFDIR)
                    strcpy(sorted[k++],de->d_name);
                else
                    strcpy(sorted[folders+m++],de->d_name);
            }
        }
        qsort(sorted, folders, sizeof(char)*128, &compare);
        qsort(&sorted[folders], files, sizeof(char)*128, &compare);
        int oplcount = 1;
        for(k = 0, m=folders+files; k < m ; k++)
        {
            char *tmp2 = sorted[k];
            struct stat ss;
            ss.st_mode = 0;
            sprintf(tmp, "%s%c%s", dirname, SEPARATOR, tmp2);
            if(stat(tmp, &ss) == 0)
            {
                if(!filter_cb || (ss.st_mode & S_IFDIR) || filter_cb(cb_data, tmp2))
                {
                    if(recursive && (ss.st_mode & S_IFDIR))
                        AddDirectory(tmp, recursive);
                    else
                    {
                        FileData fd;
                        fd.path = string(tmp);
                        fd.name = string(BaseName(tmp));
                        fd.size = (ss.st_mode & S_IFDIR) ? -1 : ss.st_size;
                        fd.index = -1;
                        fd.time  = -1;
                        fd.track = -1;
                        filerefs.push_back(fd);
                    }
                }
            }
        }
        for (int oplc = 1; oplc < 100; oplc++) // Ridicously large
        {
            sprintf(tmp, "%s%cplaylist%d.opl", dirname, SEPARATOR, oplc);
            FILE *fptmp = fopen(tmp,"rb");
            if (fptmp)
                fclose(fptmp);
            else
            {
                FileData fd;
                fd.path = string(tmp);
                sprintf(tmp, "NEW: playlist%d.opl", oplc);
                fd.name  = string(tmp);
                fd.size  =  0;
                fd.index = -1;
                fd.time  = -1;
                fd.track = -1;
                filerefs.push_back(fd);
                break;
            }
        }
        if (strlen(dirname) > 2) // Isn't root
        {
            char *dirend = strrchr(dirname,'/')+1;
            if (strlen(dirend))
            {
                sprintf(tmp, "%s%c%s.opl", dirname, SEPARATOR, dirend);
                FILE *fptmp = fopen(tmp,"rb");
                if (fptmp)
                    fclose(fptmp);
                else
                {
                    FileData fd;
                    fd.path  = string(tmp);
                    sprintf(tmp, "NEW: %s.opl",dirend);
                    fd.name  = string(tmp);
                    fd.size  = 0;
                    fd.index = -1;
                    fd.time  = -1;
                    fd.track = -1;
                    filerefs.push_back(fd);
                }
            }
        }

        closedir(dir);
        marked = start = 0;
    }
}

void SaveList::Enter(char *dirname)
{
    filerefs.clear();
    AddDirectory(dirname);
}

int SaveList::Enter(int index)
{
    char old[128] = "";
    if(index == -1)
    {
        char *end = strrchr(curdir, SEPARATOR);
        if(end)
        {
            strcpy(old, curdir);
            *end = 0;
            if(!strchr(curdir, SEPARATOR))
            {
                *end = SEPARATOR;
                end[1] = 0;
            }
        }
    }
    else
        strcpy(curdir, filerefs[index].path.c_str());

    Enter(curdir);
    if(strlen(old))
    {
        for(unsigned int i=0; i<filerefs.size(); i++)
        {
            if(strcmp(filerefs[i].path.c_str(), old) == 0)
            {
                marked = i;
                return i;
            }
        }
    }
    marked = 0;
    return 0;
}

void SaveList::ReloadCurDir()
{
    this->Enter(curdir);
    marked = filerefs.size()-1;
}
