/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
    */
#ifndef _SCREEN_H_
#define _SCREEN_H_

#include <string>
#include <stdarg.h>
#include "FileList.h"
#include "SaveList.h"
#include "PlayList.h"
#include "ZipFileList.h"
#include "pogo/font.h"

#define TO_RGB16(r,g,b) ( ((r << 8) & 0xF800) | ((g << 3) & 0x07E0) | (b >> 3) )
#define UINT2RGB16(x) (((x>>19)&0x1F)<<11) | (((x>>10)&0x3F)<<5) | ((x>>3)&0x1F)

#define LINEDISTINFO 2
#define LINEDISTBROWSE 1

using namespace std;

class FileList;
struct SDL_Surface;

class Screen
{
    public:
        Screen();

        enum {
            COL_BEVEL_DARK, COL_BEVEL_MID, COL_BEVEL_LIGHT,
            COL_BG, COL_MARK,
            COL_DIR, COL_TEXT, COL_FLAG_TEXT, COL_MARK_TEXT, COL_INFO_TEXT, COL_M3U_TEXT,
            COL_SONG0, COL_SONG1,
            COL_LCD_BACK, COL_LCD_DARK, COL_LCD_LIGHT,
            COL_EQ_LOW, COL_EQ_MID, COL_EQ_HIGH, COL_EQ_BG,
            COL_BROWSER_TITLE0, COL_BROWSER_TITLE1, 
            COL_PLAYLIST_TITLE0, COL_PLAYLIST_TITLE1,
            COL_SAVELIST_TITLE0, COL_SAVELIST_TITLE1,
            MAX_COL};

        static const char *ColorNames[];

        struct DisplaySettings
        {
            int width;
            int height;
            char *font0;
            char *font1;
            char *background;
            uint32 colors[MAX_COL];
        };

        int Width() { return settings.width; }
        int Height() { return settings.height; }
        int PrintV(char *fmt, va_list vl);
        int Print(char *fmt, ...);
        void Init(DisplaySettings *s);
        void ScreenShot(const char *fname);
        void CreateConsole(int x, int y, int w, int h);
        void *Lock() { SDL_LockSurface(screen); return screen->pixels; }
        void Unlock() { SDL_UnlockSurface(screen); }
        void Update() { SDL_UpdateRect(screen, 0, 0, 0, 0); }
        void Resize(int w, int h);
        Font *GetPropFont() { return font1; }

    protected:
        void Setup();
        friend class Pane;
        DisplaySettings settings;
        SDL_Surface *screen;
        Font *font0;
        Font *font1;
        int con_fd;
        int mainwin;
};


class Pane
{
    public:
        Pane(int x, int y, int w, int h, Screen *scr = NULL);

        enum { REDRAW = 1, RESIZE };

        virtual int Text(const char *t, int x, int y);
        virtual void Render();
        virtual void Move(int x, int y);
        virtual void Resize(int w, int h);
        virtual void Clear(bool bevel);
        virtual bool Update();
        virtual int Width() { return m_w; }
        virtual int Height() { return m_h; }

    protected:
        Font *font;
        SDL_Surface **screen;
        Screen::DisplaySettings settings;
        Font *fonts[4];
        int dirty;
        int  m_x;
        int  m_y;
        int  m_w;
        int  m_h;
        int con_fd;
};


class TextDisplay : public Pane
{
    public:
        TextDisplay(int x, int y, int w, int h, Screen *scr = NULL);

        void PutText(int x, int y, const char *str);
        void Clear(bool bevel);

    protected:
        int conwin;
};

class ListPane : public Pane
{
    public:
        ListPane(int x, int y, int w, int h, Screen *scr = NULL);

        enum {NOTHING, UP, DOWN, PGUP, PGDN, JUMPTO};

        void SetList(IFileList *fl)
        { 
            dirty |= REDRAW;
            file_list = fl;
        }
        IFileList *GetList() { return file_list; }
        void Render();
        bool Update();
        void DoCommand(int cmd, int cmd2 = 0);
        void Resize(int w, int h);
        void SetTitle(char *t) { strcpy(title, t); }
        void SetTitleColor(int col0, int col1)
        { 
            titlecol0 = col0; titlecol1 = col1;
        }

    protected:
        IFileList *file_list;
        char title[128];
        int titlecol0;
        int titlecol1;
        int con_fd;
        int conwin;
        int lines;
        int last_cmd;
        int delay;
        int last_change;
        //int marked;
        //int oldmarked;
        //int start;
};

class LedControl : public Pane
{
    public:
        LedControl(char *fname, int x, int y, Screen *scr = NULL);

        void SetLed(int led, int val);
        int GetLed(int led);
        bool Update();

        enum {
            BLACK, SPEAKER, VOLUME,
            HOLD    = 12,
            PLAYING = 13, SONG, SOLID, REPEAT, SHUFFLE, 
            NUMBERS = 18, LENGTH = 18,
            SONGNO  = NUMBERS + 15*5,
            MAXSONG = NUMBERS + 15*7,
            BAR0 = 153,
            BAR1 = 165,
            EQ0  = 177,
            EQ1  = 182,
            EQ2  = 187,
            EQ3  = 192,
            BUSY = 197, // 6 long
            VOLT = 203,
        };

    protected:
        void SetNumbers(int led, int val, int digs);
        SDL_Surface *display;
        unsigned short clut[256];
        unsigned short dark;
        unsigned short light;
};

class SongInfoPane : public Pane
{
    public:
        SongInfoPane(int x, int y, int w, int h, Screen *scr = NULL);

        bool Update();
        void SetNFields(int n);
        void SetFieldName(string fieldname, int i);
        void SetFieldData(string fielddata, int i);
        void SetLength(int l);

    protected:
        int length;
        int nfields;
        string fieldname[5];
        string fielddata[5];
};

int cprintf(char *fmt, ...);

extern Screen g_screen;

#endif
