/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
//#include <sdl_syswm.h>
#include <SDL/SDL.h>
#include <stdio.h>
#include <fcntl.h>
#include <vector>

#ifdef _WIN32
#include <io.h>
#endif

#include "SDLScreen.h"
#include "FileList.h"
#include "PlayerApp.h"
#include "util.h"
#include "pogo/screen.h"
#include "pogo/font.h"
#include "pogo/device.h"
#include "pogo/console.h"

int LINEDIST = LINEDISTBROWSE;

extern PlayerApp *app;

const char *Screen::ColorNames[] = { 
        "COL0", "COL1", "COL2", 
        "BGCOL", "MARKCOL", 
        "DIRCOL", "TEXTCOL", "FLAGTEXTCOL", "MARKTEXTCOL", "INFOTEXTCOL", "M3UTEXTCOL",  
        "SONG0COL", "SONG1COL", 
        "LCD0COL", "LCD1COL", "LCD2COL",
		"EQ0COL", "EQ1COL", "EQ2COL", "EQBGCOL", 
        "BTITLECOL0", "BTITLECOL1", 
        "PTITLECOL0", "PTITLECOL1", 
        "STITLECOL0", "STITLECOL1"
        };

bool reinit   = false;
int bevelmid;
int beveldark;
int bevellight;
int bgcol       = 0x000000;
int markcol     = 0x000080;
int dircol      = 0xFFFF80;
int playlistcol = 0xFFFF80;
int textcol     = 0xFFFFFF;
int flagtextcol = 0x00FF00;
int marktextcol = 0x000FF0;
int infotextcol = 0x0000FF;
int songrast0   = 0x438AB3;
int songrast1   = 0x63AAD3;
int lcd0;
int lcd1;
int lcd2;
int bevsize = 2;

void sub_hline(unsigned short *ptr, int len, int thick, int width, int s)
{
	while(thick--)
	{
		int l = len;
		while(l--)
		{
			unsigned short c = *ptr;

			int b =  c      & 0x1f;
			int g = (c>>5)  & 0x3f;
			int r = (c>>11) & 0x1f;

			r += s>>1;
			if(r < 0) r = 0;
			if(r > 0x1f) r = 0x1f;
			g += s;
			if(g < 0) g = 0;
			if(g > 0x3f) g = 0x3f;
			b += s>>1;
			if(b < 0) b = 0;
			if(b > 0x1f) b = 0x1f;

			*ptr++ = (r<<11) | (g<<5) | b;
		}
		ptr += (width-len);
	}
}

void sub_vline(unsigned short *ptr, int len, int thick, int width, int s)
{

	while(thick--)
	{
		int l = len;
		while(l--)
		{
			unsigned short c = *ptr;

			int b =  c      & 0x1f;
			int g = (c>>5)  & 0x3f;
			int r = (c>>11) & 0x1f;

			r += s>>1;
			if(r < 0) r = 0;
			if(r > 0x1f) r = 0x1f;
			g += s;
			if(g < 0) g = 0;
			if(g > 0x3f) g = 0x3f;
			b += s>>1;
			if(b < 0) b = 0;
			if(b > 0x1f) b = 0x1f;

			*ptr = (r<<11) | (g<<5) | b;

			ptr += width;
		}
		ptr = ptr - (width*len) + 1;
	}
}

void bevel_box(SDL_Surface *surf, int x, int y, int w, int h)
{
	SDL_LockSurface(surf);
	
	unsigned short *pixels = (unsigned short *)surf->pixels;

	x-=bevsize;
	y-=bevsize;

	sub_hline(&pixels[x+y*surf->w], w+2, bevsize, surf->w, -5);
	sub_vline(&pixels[x+(y+bevsize)*surf->w], h, bevsize, surf->w, -5);

	x+=bevsize;
	y+=bevsize;

	sub_hline(&pixels[x+(y+h)*surf->w], w+bevsize, bevsize, surf->w, 5);
	sub_vline(&pixels[x+w+y*surf->w], h, bevsize, surf->w, 5);

	SDL_UnlockSurface(surf);
	SDL_UpdateRect(surf, x-2, y-2, w+4, h+4);
}

void raster(SDL_Surface *surf, int x, int y, int width, int height, int col0, int col1)
{
	int r =  col0>>16;
	int g = (col0>>8) & 0xff;
	int b =  col0     & 0xff;
	int dr = ( (col1>>16)        - r) * 256 / width;
	int dg = (((col1>>8) & 0xff) - g) * 256 / width;
	int db = ( (col1     & 0xff) - b) * 256 / width;

	r <<= 8;
	g <<= 8;
	b <<= 8;

	int w = width;
	unsigned short *pix = (unsigned short *)surf->pixels + x + y*surf->w;
	while(w--)
	{
		unsigned short col = TO_RGB16((r>>8), (g>>8), (b>>8));
		int h = height;
		while(h--)
		{
			*pix = col;
			pix += surf->w;
		}
		pix -= (surf->w*height-1);
		r += dr;
		g += dg;
		b += db;
	}
}

Pane::Pane(int x, int y, int w, int h, Screen *scr)
{
	if(!scr)
		scr = &g_screen;

	font = scr->font1;
	screen = &scr->screen;

	settings = scr->settings;

	con_fd = scr->con_fd;
	for(int i=0; i<4; i++)
		fonts[i] = scr->font0;

	dirty = REDRAW | RESIZE;

	m_x = x;
	m_y = y;
	m_w = w;
	m_h = h;
}

int Pane::Text(const char *t, int x, int y)
{
	int len;

	if(y >= 0)
	{
		int l = m_w - (x < 0 ? 0 : x*font->charwidth);
		char tmp[128];
		strncpy(tmp, t, 127);
        tmp[127] = 0;
		char *endp = &tmp[strlen(tmp)-1];
		while((endp > tmp) && (Text(tmp, -1, -1) > l))
			*endp-- = 0;

		len = font_text(font, (char *)tmp, -1, -1);

		if(x < 0)
			font_text(font, (char *)tmp, m_x + m_w - len + x*font->charwidth, 
                            m_y + y*(font->height+LINEDIST) + LINEDIST/2);
		else
			font_text(font, (char *)tmp, m_x + x*font->charwidth, m_y + 
                            y*(font->height+LINEDIST) + LINEDIST/2);
	}
	else
		len = font_text(font, (char *)t, -1, -1);
	return len;
}

void Pane::Clear(bool bevel)
{
	if(m_w && m_h)
	{
		if(bevel)
			bevel_box(*screen, m_x, m_y, m_w, m_h);

		SDL_Rect rect;
		rect.x = m_x;
		rect.y = m_y;
		rect.w = m_w;
		rect.h = m_h;
		SDL_FillRect(*screen, &rect, UINT2RGB16(settings.colors[Screen::COL_BG]));
		dirty |= REDRAW;
	}
}

void Pane::Render()
{
	if(reinit)
	{
		console_set_target((*screen)->pixels, (*screen)->w);
		font_settarget((*screen)->pixels, (*screen)->w, (*screen)->h, 0);
	}

	if(dirty && m_w && m_h)
	{
		if(dirty & RESIZE)
			Clear(true);

		dirty = 0;

		SDL_LockSurface(*screen);

		font_setcolor(textcol, bgcol);
		font_settarget((*screen)->pixels, (*screen)->w, (*screen)->h, 0);

		
        if(Update())
			SDL_UpdateRect(*screen, m_x, m_y, m_w, m_h);
        
		SDL_UnlockSurface(*screen);
        
	}
}

bool Pane::Update()
{
	return true;
}

void Pane::Move(int x, int y)
{
	m_x = x;
	m_y = y;
	dirty |= REDRAW;
}

void Pane::Resize(int w, int h)
{
	m_w = w;
	m_h = h;
	dirty |= (REDRAW | RESIZE);
}

using namespace std;

const unsigned char digits[] =
{
	1,1,1,1,0,1,1,0,1,1,0,1,1,1,1,
	0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,
	1,1,1,0,0,1,1,1,1,1,0,0,1,1,1,
	1,1,1,0,0,1,1,1,1,0,0,1,1,1,1,
	1,0,1,1,0,1,1,1,1,0,0,1,0,0,1,
	1,1,1,1,0,0,1,1,1,0,0,1,1,1,1,
	1,1,1,1,0,0,1,1,1,1,0,1,1,1,1,
	1,1,1,0,0,1,0,0,1,0,0,1,0,0,1,
	1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,
	1,1,1,1,0,1,1,1,1,0,0,1,0,0,1,
};

bool LedControl::Update()
{
	int width = (*screen)->w;

	unsigned short *pix = (unsigned short *)(*screen)->pixels + m_x + width * m_y;

	unsigned char *data = (unsigned char *)display->pixels;
	int mod = width - display->w;
	int height = display->h;

	while(height--)
	{
		int w = display->w;
		while(w--)
			*pix++ = clut[*data++];
		pix += mod;
	}

	return true;

}

LedControl::LedControl(char *fname, int x, int y, Screen *scr) :
	Pane(x, y, 0, 0, scr)
{
	dark = UINT2RGB16(settings.colors[Screen::COL_LCD_DARK]);
	light =  UINT2RGB16(settings.colors[Screen::COL_LCD_LIGHT]);

	for(int i = 1; i<256; i++)
		clut[i] = dark;
	clut[0] = UINT2RGB16(settings.colors[Screen::COL_LCD_BACK]);

	clut[SPEAKER] = light;
	clut[SOLID] = light;
	clut[SONG] = light;

	if(fname)
		display = SDL_LoadBMP(fname);

	if(!display)
		display = SDL_CreateRGBSurface(0, 104, 56, 8, 0,0,0,0);

	Resize(display->w, display->h);
}

void LedControl::SetNumbers(int led, int val, int digs)
{
	for(int n=digs-1; n>=0; n--)
	{
		int d = val % 10;
		val = val/10;
		const unsigned char *ptr = &digits[d * 15];
		for(int i=0; i<15; i++)
			clut[led + n*15 + i] = (ptr[i] ? light : dark);
	}
}

void LedControl::SetLed(int led, int val)
{
	int i;

	dirty = REDRAW;

	switch(led)
	{
	case SPEAKER:
	case SOLID:
	case SONG:
	case REPEAT:
	case SHUFFLE:
	case PLAYING:
	case HOLD:
		clut[led] = (val ? light : dark);
		break;
	case VOLUME:
		for(i=0; i<10; i++)
			clut[VOLUME + i] = (val > i ? light : dark);
		break;
	case SONGNO:
	case MAXSONG:
		SetNumbers(led, val, 2);
		break;
	case LENGTH:
		SetNumbers(led + 3*15, val % 60, 2);
		SetNumbers(led, val / 60, 3);
		break;
	case BUSY:
		for(i=0; i<6; i++)
			clut[BUSY + i] = (val > i ? light : dark);
		break;
	case BAR0:
	case BAR1:
	case VOLT:
		for(i=0; i<12; i++)
			clut[led + i] = (val > i ? light : dark);
		break;
	case EQ0:
	case EQ1:
	case EQ2:
	case EQ3:
        // I'd rather want them all combined
        for(i=0; i<20; i++)
            clut[led + i] = (val > i ? light : dark);
		//for(i=0; i<5; i++) // Split
		//	clut[led + i] = (val > i ? light : dark);
		break;
	//default:
		//dirty = false;
	}
}

Screen::Screen() :
con_fd(-1),
screen(NULL)
{
}

PogoDevice native;

int native_open(const char *name, int flags)
{
	return (int)open(name, 0);
}

int native_read(int fd, void *dest, int size)
{
	return read(fd, dest, size);
}

int native_write(int fd, const void *dest, int size)
{
	return -1;
}

void native_close(int fd)
{
	close(fd);
}

int native_lseek(int fd, int offset, int origin)
{
	return lseek(fd, offset, origin);
}

int native_tell(int fd)
{
	return -1;
}

int native_ioctl(int fd, int request, ...)
{
	return -1;
}

int native_remove(const char *name)
{
	return -1;
}

int native_stat(const char *name, struct pogo_stat *buffer)
{
	return -1;
}

SDL_Surface *backing = NULL;

void draw_backing(SDL_Surface *screen, SDL_Surface *bak)
{
	if(bak)
	{
		SDL_Rect dr,sr;
        SDL_SetColorKey(bak,SDL_SRCCOLORKEY,SDL_MapRGB(bak->format,0xFF,0x00,0xFF));
		dr.x = 0;		
		dr.y = 0;
		SDL_BlitSurface(bak, NULL, screen , &dr);
	}
}

void Screen::Setup()
{
	reinit = true;
	if(screen)
		SDL_FreeSurface(screen);

	screen = SDL_SetVideoMode(settings.width, settings.height, 16, SDL_RESIZABLE);
	if( screen == NULL ) {
		fprintf(stderr, "Screen::Setup - Cannot set video mode!\n");
		exit(-1);
	}


#if defined(GP2X) || defined(A320)
	SDL_ShowCursor(0);
#endif

	console_set_target(screen->pixels, settings.width);
	font_settarget(screen->pixels, settings.width, settings.height, 0);


	SDL_FillRect(screen, NULL, UINT2RGB16(bevelmid));
	if(backing)
		draw_backing(screen, backing);
}

void Screen::Resize(int w, int h)
{
	w = (w+3) & 0xFFFC;

	settings.width = w;
	settings.height = h;

	Setup();
	app->Layout();
	SDL_UpdateRect(screen, 0, 0, 0, 0);
}

void Screen::Init(DisplaySettings *ss)
{
	settings = *ss;
    
    bevelmid    = settings.colors[COL_BEVEL_MID];
	beveldark   = settings.colors[COL_BEVEL_DARK]; //Not used
	bevellight  = settings.colors[COL_BEVEL_LIGHT]; //Not used
	bgcol       = settings.colors[COL_BG];
	markcol     = settings.colors[COL_MARK];
	dircol      = settings.colors[COL_DIR];
	playlistcol = settings.colors[COL_M3U_TEXT];
	textcol     = settings.colors[COL_TEXT];
	flagtextcol = settings.colors[COL_FLAG_TEXT];
	marktextcol = settings.colors[COL_MARK_TEXT];
	infotextcol = settings.colors[COL_INFO_TEXT];
	songrast0   = settings.colors[COL_SONG0];
	songrast1   = settings.colors[COL_SONG1];

	bevsize = (ss->width <= 320) ? 1 : 2;

	SDL_Init(SDL_INIT_VIDEO | SDL_INIT_AUDIO | 
             SDL_INIT_JOYSTICK | SDL_INIT_NOPARACHUTE);

#if !defined(GP2X) && !defined(A320)
	SDL_WM_SetCaption("OldPlay", NULL);
#endif

	if(settings.background && strlen(settings.background))
		backing = SDL_LoadBMP(settings.background);

	screen = NULL;
	Setup();

	SDL_JoystickOpen(0);

	pogo_device_init();
	memset(&native, 0, sizeof(PogoDevice));

	native.open = native_open;
	native.read = native_read;
	native.close = native_close;
	native.lseek = native_lseek;

	pogo_device_register(&native, "native", NULL, -1);

	font0 = font1 = NULL;

	if(settings.font0 && strlen(settings.font0))
		font0 = font_load(settings.font0);
	if(settings.font1 && strlen(settings.font1))
		font1 = font_load(settings.font1);

	if(!font0)
		 font0 = font_getdefault();
	if(!font1)
		 font1 = font_getdefault();

	console_init();
	con_fd = pogo_open("/dev/console", 0);

	int mainwin = -1;
}

void Screen::ScreenShot(const char *fname)
{
	SDL_SaveBMP(screen, fname);
}

void Screen::CreateConsole(int x, int y, int w, int h)
{
	Font *fonts[4];

	for(int i=0; i<4; i++)
		fonts[i] = font1;

	mainwin = pogo_ioctl(con_fd, CC_OPENWIN, x, y, w, h, fonts, CONWIN_STREAM);
	bevel_box(screen, x, y, w, h);
	pogo_ioctl(con_fd, CC_SETCOLOR, CCOL_GREEN, CCOL_BLACK);
	pogo_ioctl(con_fd, CC_CLEAR);
}

int Screen::PrintV(char *fmt, va_list vl)
{
	char tmp[256];
	int rc;

	if(con_fd < 0 || mainwin < 0)
		return -1;

	rc = vsprintf(tmp, fmt, vl);
	pogo_ioctl(con_fd, CC_SETWIN, mainwin);
	pogo_write(con_fd, tmp, rc);
	pogo_ioctl(con_fd, CC_FLUSH);
	if(screen)
		SDL_UpdateRect(screen, 0, 0, 0, 0);

	return rc;
}

int Screen::Print(char *fmt, ...)
{
	va_list vl;
	va_start(vl, fmt);
	int rc = PrintV(fmt, vl);
	va_end(vl);
	return rc;
}

Screen g_screen;

int cprintf(char *fmt, ...)
{
	va_list vl;
	va_start(vl, fmt);
	int rc = g_screen.PrintV(fmt, vl);
	va_end(vl);
	return rc;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////// ListPane ////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////

ListPane::ListPane(int x, int y, int w, int h, Screen *scr) :
	Pane(x, y, w, h, scr)
{
	//start = marked = 0;
	lines = h / (font->height+LINEDIST) - 2;
	dirty = REDRAW;
	delay = 0;
	last_change = -1;
	*title = 0;
	titlecol0 = UINT2RGB16(settings.colors[Screen::COL_BROWSER_TITLE0]);
	titlecol1 = UINT2RGB16(settings.colors[Screen::COL_BROWSER_TITLE1]);
}

void ListPane::Resize(int w, int h)
{
	m_w = w;
	m_h = h;
	dirty |= (REDRAW | RESIZE);
}

void ListPane::Render()
{
	if(file_list->Changed() > last_change)
		dirty |= REDRAW;
	last_change = file_list->Changed();
	Pane::Render();
}

bool ListPane::Update()
{

	char buf[256];

	Clear(false);
	dirty = 0;

	int count = (int)file_list->Size();

	lines = m_h / (font->height+LINEDIST) - 2;

	int top = 0;

	if(strlen(title))
	{
		raster(*screen, m_x, m_y, m_w, font->height, titlecol1, titlecol1);
		int f = font->flags;
		font->flags |= FFLG_TRANSP;
		font_setcolor(titlecol0, titlecol1);
		Text(title, 0, 0);
		font->flags = f;
		lines--;
		top = 1;
	}

	int start = file_list->Start();
	font_setcolor(textcol, bgcol);
	if(start !=0)
	{
		Text("...", 1, top);
	}
	if(count > start+lines)
	{
		Text("...", 1, top+lines+1);
	}

	int l = m_w - Text("9999M  ", -1, -1);

	for(int i = start; i<count && i<(start+lines); i++)
	{
		int bg = bgcol;
		int fg = textcol;

		if(i == file_list->Marked())
		{
			bg = markcol;
			SDL_Rect r;
			r.x = m_x;
			r.y = m_y + (top+1+i-start)*(font->height+LINEDIST);
			r.w = m_w;
			r.h = font->height+LINEDIST;
			SDL_FillRect(*screen, &r, UINT2RGB16(bg));
		}

		char tmp[128];

		strcpy(tmp, file_list->GetName(i));

		char *endp = &tmp[strlen(tmp)-1];

		while((endp > tmp) && (Text(tmp, -1, -1) > l))
			*endp-- = 0;

		int   fsize  = file_list->GetSize(i);
        int   ftime  = file_list->GetTime(i)/1000;
        int   ftrack = file_list->GetTrack(i);
        const char *fname = file_list->GetName(i);

		if (file_list->Flagged(i))
			fg = flagtextcol;
        else if (file_list->Marked() == i)
            fg = marktextcol;
        else if (fsize == -1 || is_fex(fname))
            fg = dircol;
        else if (is_playlist(fname))
            fg = playlistcol;

		if(fsize == -1 || is_fex(file_list->GetName(i)))
		{
			font_setcolor(fg, bg);
			sprintf(buf, "[%s]", tmp);
			Text(buf, 1, top+1+i-start);
		}
        else if (ftime>0)
        {
            font_setcolor(fg, bg);
            Text(tmp, 1, top+1+i-start);
            if (ftrack >= 0)
                sprintf(buf, "(%d) %02d:%02d",ftrack+1,ftime/60,ftime%60);
            else
                sprintf(buf, "%02d:%02d",ftime/60,ftime%60);
            Text(buf, -1, top+1+i-start);
        }
		else
		{
			font_setcolor(fg, bg);
			Text(tmp, 1, top+1+i-start);
			if(fsize < 1024)
                if (ftrack >= 0)
				    sprintf(buf, "(%d) %4dB", ftrack+1, fsize);
                else
                    sprintf(buf, "%4dB", fsize);
			else
			if(fsize < 1024*1024)
                if (ftrack >= 0)
				    sprintf(buf, "(%d) %4dK", ftrack+1, fsize >> 10);
                else
				    sprintf(buf, "%4dK", fsize >> 10);
			else
                if (ftrack >= 0)
				    sprintf(buf, "(%d) %4dM", ftrack+1, fsize >> 20);
                else
				    sprintf(buf, "%4dM", fsize >> 20);
			Text(buf, -1, top+1+i-start);
		}
	}
	return true;
}

void ListPane::DoCommand(int cmd, int cmd2)
{
	int marked = file_list->Marked();

	switch(cmd)
	{
	case UP:
		marked--;
		break;
	case DOWN:
		marked++;
		break;
	case PGUP:
		marked -= lines;
		break;
	case PGDN:
		marked += lines;
		break;
    case JUMPTO:
        marked = cmd2;
        break;
	}

	int start = file_list->Start();
	int count = (int)file_list->Size();

	if(marked < 0)
		marked = 0;
	if(marked >= count)
		marked = count-1;

	while(marked >= start+lines)
		start += 3;
	while(marked < start)
		start -= 3;

	if(start > count-lines)
		start = count-lines;

	if(start < 0)
		start = 0;

	//if((start != file_list->Start()) || (marked != file_list->Marked()))
		dirty = REDRAW;

	file_list->SetMarked(marked);
	file_list->SetStart(start);
}

TextDisplay::TextDisplay(int x, int y, int w, int h, Screen *scr) :
	Pane(x, y, w, h, scr)
{
	//conwin = pogo_ioctl(con_fd, CC_OPENWIN, x, y, w, h, fonts, CONWIN_FIXED);
	pogo_ioctl(con_fd, CC_SETCOLOR, CCOL_YELLOW, CCOL_BLACK);
}

void TextDisplay::PutText(int x, int y, const char *str)
{
	dirty |= REDRAW;
}

void TextDisplay::Clear(bool bevel)
{
	dirty |= REDRAW;
}

SongInfoPane::SongInfoPane(int x, int y, int w, int h, Screen *scr) :
	Pane(x, y, w, h, scr)
{
    nfields = 0;
	length = -1;
}

bool SongInfoPane::Update()
{
    LINEDIST = LINEDISTINFO;
	char tmp[128];
	Clear(false);
	dirty = 0;
	int y = 0;
	int fh = font->height + LINEDIST;
	int flgs = font->flags;
	font->flags |= FFLG_TRANSP;
    for (int i = 0; i < nfields; i++)
    {
        font_setcolor(infotextcol, songrast0);
        raster(*screen, m_x, m_y+fh*y, m_w, font->height+LINEDIST, 
                                                songrast0, songrast1);
        m_x+=2; // Dont write it to the edge of the bevel!
        Text(fieldname[i].c_str(), 0, y); y++;
        font_setcolor(textcol, bgcol);
        Text(fielddata[i].c_str(), 0, y); y++;
        m_x-=2;
    }
	if(length > 0)
	{
		font_setcolor(infotextcol, songrast0);
		raster(*screen, m_x, m_y+fh*10, m_w, font->height+LINEDIST, 
                                                songrast0, songrast1);
        m_x+=2;
		sprintf(tmp, "Length: %02d:%02d", length/60, length % 60);
		Text(tmp, 0, 10); //y++;
        m_x-=2;
	}
	font->flags = flgs;
    LINEDIST = LINEDISTBROWSE;
	return true;
}

void SongInfoPane::SetNFields(int n)
{
    nfields = n;
    dirty |=REDRAW;
}

void SongInfoPane::SetFieldName(string fname, int i)
{
    fieldname[i] = fname;
    dirty |=REDRAW;
}

void SongInfoPane::SetFieldData(string fdata, int i)
{
    fielddata[i] = fdata;
    dirty |=REDRAW;
}

void SongInfoPane::SetLength(int l)
{
	length = l;
	dirty |= REDRAW;
}

