/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <SDL/SDL.h>
#include <string>
#ifdef _WIN32
#include <direct.h>
#endif

#include "FileList.h"
#include "PlayList.h"
#include "SaveList.h"
#include "SDLScreen.h"
#include "MusicPlayer.h"
#include "Equalizer.h"
#include "PlayerApp.h"
#include "Input.h"
#include "util.h"

#define ABS(x) ((x) < 0 ? (-x) : (x))
using namespace std; 

PlayerApp::~PlayerApp()
{
    char dst[8];
    config["REPEAT"]  = repeat  ? (char *)"1" : (char *)"0";
    config["SHUFFLE"] = shuffle ? (char *)"1" : (char *)"0";
    sprintf(dst, "%d", music_player->GetVolume());
    config["VOLUME"] = dst;
    config.Save();
}

PlayerApp::PlayerApp(char *fname)
{
    InitSettings(fname);

    pane         = new ListPane(0, 0, 0, 0);
    songInfo     = new SongInfoPane(0, 0, 0, 0);
    lcd          = new LedControl(screen_cfg["LEDBMP"], 0, 0);
    eq           = new Equalizer(0, 0, lcd->Width(), lcd->Width()/3);

    music_player = new MusicPlayer(cpu_scaling);
    music_player->SetVolume(atoi(config["VOLUME"]));
    music_player->SetDefaultLength(atoi(screen_cfg["DEFLENGTH"]));

    lcd->SetLed(LedControl::SONGNO, 1);
    lcd->SetLed(LedControl::MAXSONG, 1);
    lcd->SetLed(LedControl::VOLUME, music_player->GetVolume()*10/VOLSTEPS);
    lcd->SetLed(LedControl::REPEAT, repeat);
    lcd->SetLed(LedControl::SHUFFLE, shuffle);

    char curdir[256];
    char * usedir = curdir;
    if (screen_cfg["DIR"])
        usedir = screen_cfg["DIR"];
    else
#ifdef A320
	strcpy(curdir,"/usr/local");
#else
        getcwd(curdir,sizeof(curdir));
#endif
    
    file_list = new FileList(usedir, &plugin_filter_cb, this);
    save_list = new SaveList(usedir, &filter_opl_cb, this);
    play_list = new PlayList(&plugin_filter_cb, this);
    //zip_list  = NULL;

    fprintf(stderr, "PlayerApp::PlayerApp - Constructed\n");
}

void PlayerApp::Layout()
{
    int fh = g_screen.GetPropFont()->height;
    int w  = g_screen.Width() - hmargin*2;
    int h  = g_screen.Height() - vmargin*2;

    lcd->Move(w - lcd->Width() + hmargin, fh+sep + vmargin);
    lcd->Clear(true);

    pane->Move(hmargin, fh+sep + vmargin);
    pane->Resize(w - lcd->Width() - sep, h - (fh+sep));
    pane->Clear(true);

    songInfo->Move(w - lcd->Width() + hmargin, lcd->Height() + fh + sep + vmargin + sep);
    songInfo->Resize(lcd->Width(), (fh+LINEDISTINFO) * 11);
    songInfo->Clear(true);
    songInfo->Render();

    int y = lcd->Height() + (fh+sep) + sep + sep + songInfo->Height()  + vmargin;
    eq->Move(w - lcd->Width() + hmargin, y);
    eq->Clear(true);

    Font *font = g_screen.GetPropFont();

    int len = font_text(font, OLDPLAYVERSION, -1, -1);
    int f = font->flags;
    font->flags |= FFLG_TRANSP;
    font_setcolor(0x000000, settings.colors[Screen::COL_BEVEL_MID]);
    font_text(font, OLDPLAYVERSION, w-len+1+hmargin, h-font->height+vmargin+1);
    font_setcolor(0xFFFFFF, settings.colors[Screen::COL_BEVEL_MID]);
    font_text(font, OLDPLAYVERSION, w-len+hmargin, h-font->height+vmargin);

#ifdef A320
    font->flags |= FFLG_TRANSP;
    font_setcolor(0x000000, settings.colors[Screen::COL_BEVEL_MID]);
    font_text(font, "for Dingux", 74, vmargin+1);
    font_setcolor(0xFFFFFF, settings.colors[Screen::COL_BEVEL_MID]);
    font_text(font, "for Dingux", 74, vmargin);
#endif

    font->flags = f;
}

static int  last_sec   = -1;
static int  paused     = 0;
static bool oldplaying = false;

void PlayerApp::Render()
{
    if(paused)
    {
        if(((paused-1) % 15) == 0)
            lcd->SetLed(LedControl::PLAYING, (paused-1) % 30 );
        paused++;
    }
    if(playing != oldplaying)
    {
        lcd->SetLed(LedControl::PLAYING, playing ); 
        oldplaying = playing;
    }
    int sec = music_player->GetSeconds();
    if(last_sec != sec)
    {
        lcd->SetLed(LedControl::LENGTH, sec);
        last_sec = sec;
    }
    pane->Render();
    songInfo->Render();
    lcd->Render();
#if defined(GP2X) || defined(A320)
    if(!music_player->GetHeavy())
#endif
    {
        if(eq)
            eq->Render();
        RenderEq();
    }

}

static int info_update_counter = 1;
static int new_song            = 0;
bool PlayerApp::Update()
{

    if (music_player->Playing() && !info_update_counter--)
    {
        info_update_counter = 10;
        lcd->SetLed(LedControl::SONGNO, music_player->GetTune()+1);
        lcd->SetLed(LedControl::MAXSONG, music_player->GetMaxTunes());
        UpdateInfo();
    }
    if(!music_player->Playing() && play_list->Size())
    {
        play_list->Next(repeat);
        if(play_list->Current() != -1)
            PlaySong(play_list, play_list->Current() );
    }
    return true;
}

static char name_len, name_pos, subname_len, subname_pos, format_len, format_pos,
     artist_len, artist_pos, copyright_len, copyright_pos;
     
static string field[5];
static int    nfields;
static int    fieldlen[5];
static int    fieldpos[5];

void PlayerApp::UpdateInfo()
{
#if defined(GP2X) || defined(A320)
    if (new_song)
    {
        new_song--;
        nfields = music_player->GetNFields();
        for (int i = 0; i < nfields; i++)
        {
            field[i] = music_player->GetFieldData(i);
            songInfo->SetFieldName(music_player->GetFieldName(i),i);
            fieldlen[i] = field[i].length();
            fieldpos[i] = 0;
        }
        songInfo->SetNFields(nfields);
        songInfo->SetLength(music_player->GetLength());
    }
    else
    {
        for (int i = 0; i < nfields; i++)
            fieldpos[i] = fieldlen[i] <= 16 || fieldpos[i] + 16 >= fieldlen[i] 
                            ? 0 : fieldpos[i] + 1;
    }
#endif
    for (int i = 0; i < nfields; i++)
        songInfo->SetFieldData(field[i].substr(fieldpos[i]),i);
}

bool PlayerApp::HandleKey(int key)
{
    switch(key)
    {
    case Input::NEXT:
        if(play_list->Size())
        {
            play_list->Next(true);
            if(play_list->Current() != -1)
                PlaySong(play_list, play_list->Current() );
        }
        break;
    case Input::PREV:
        if(play_list->Size())
        {
            play_list->Prev(true);
            if(play_list->Current() != -1)
                PlaySong(play_list, play_list->Current() );
        }
        break;
    case Input::FF:
        music_player->SetTune(music_player->GetTune() + 1);
        lcd->SetLed(LedControl::SONGNO, music_player->GetTune()+1);
        new_song = 5;
        UpdateInfo();
        break;
    case Input::REW:
        music_player->SetTune(music_player->GetTune() - 1);
        lcd->SetLed(LedControl::SONGNO, music_player->GetTune()+1);
        new_song = 5;
        UpdateInfo();
        break;
    case Input::VOLUP:
        music_player->SetVolume(music_player->GetVolume() + 1);
        lcd->SetLed(LedControl::VOLUME, music_player->GetVolume()*10/VOLSTEPS);
        break;
    case Input::VOLDOWN:
        music_player->SetVolume(music_player->GetVolume() - 1);
        lcd->SetLed(LedControl::VOLUME, music_player->GetVolume()*10/VOLSTEPS);
        break;
    case Input::SHUFFLE:
        shuffle = !shuffle;
        lcd->SetLed(LedControl::SHUFFLE, shuffle);
        play_list->SetShuffle((bool)shuffle);
        break;
    case Input::REPEAT:
        repeat = !repeat;
        lcd->SetLed(LedControl::REPEAT, repeat);
        break;
    case Input::PAUSE:
        paused = music_player->Pause();
        break;
    case Input::BLENDUP:
        music_player->SetBlend(music_player->GetBlend() + 20);
        //lcd->SetLed(LedControl::EQ0, music_player->GetBlend() / 50);
        break;
    case Input::BLENDDOWN:
        music_player->SetBlend(music_player->GetBlend() - 20);
        //lcd->SetLed(LedControl::EQ0, music_player->GetBlend() / 50);
        break;
    case Input::TOGGLEBLEND:
        music_player->SetBlend(music_player->GetBlend() ? 0 : 128);
        //lcd->SetLed(LedControl::EQ0, music_player->GetBlend() / 50);
        break; 
    case Input::SCREENSHOT:
        g_screen.ScreenShot("screenshot.bmp");
        break;
    }
    return true;
}

void PlayerApp::InitSettings(char *fname)
{
    fprintf(stderr, "PlayerApp::InitSettings\n");

    config.Init(".settings");
    config["REPEAT"]   = "0";
    config["SHUFFLE"]  = "0";
#ifdef A320
    config["CPUSCALE"] = "200";
#else
    config["CPUSCALE"] = "336";
#endif
    config["VOLUME"]   = "8";
    config.Load();
    repeat      = atoi(config["REPEAT"]);
    shuffle     = atoi(config["SHUFFLE"]);
    //cpu_scaling = atoi(config["CPUSCALE"]);
    config.Save();

    if(fname)
        screen_cfg.Init(fname);
    else
        screen_cfg.Init("default.cfg");

    fprintf(stderr, "PlayerApp::InitSettings - Setting defaults\n");

    screen_cfg["COL0"]        = "0xF8FCF8";
    screen_cfg["BGCOL"]       = "0xC8C8A8";
    screen_cfg["MARKCOL"]     = "0x88A8B8";
    screen_cfg["DIRCOL"]      = "0x000070";
    screen_cfg["TEXTCOL"]     = "0x000000";
    screen_cfg["FLAGTEXTCOL"] = "0x688078";
    screen_cfg["MARKTEXTCOL"] = "0x688078";
    screen_cfg["INFOTEXTCOL"] = "0xFFFFFF";
    screen_cfg["M3UTEXTCOL"]  = "0xFFFFFF";
    screen_cfg["BTITLECOL0"]  = "0xFFFFFF";
    screen_cfg["BTITLECOL1"]  = "0x286488";
    screen_cfg["PTITLECOL0"]  = "0xFFFFFF";
    screen_cfg["PTITLECOL1"]  = "0xa40206";
    screen_cfg["STITLECOL0"]  = "0xFFFFFF";
    screen_cfg["STITLECOL1"]  = "0x000000";
    screen_cfg["COL1"]        = "0x707068";
    screen_cfg["COL2"]        = "0x00A0A0";
    screen_cfg["SONG0COL"]    = "0x596168";
    screen_cfg["SONG1COL"]    = "0x8C99A3";
    screen_cfg["LCD0COL"]     = "0xC8C8A8";
    screen_cfg["LCD1COL"]     = "0xB8B898";
    screen_cfg["LCD2COL"]     = "0x202020";
    screen_cfg["EQ0COL"]      = "0x6C0000";
    screen_cfg["EQ1COL"]      = "0xE01700";
    screen_cfg["EQ2COL"]      = "0xFDF953";
    screen_cfg["EQBGCOL"]     = "0x000000";
    screen_cfg["BACKGROUND"]  = "";
    screen_cfg["HMARGIN"]     = "4";
    screen_cfg["VMARGIN"]     = "4";
    screen_cfg["SPACING"]     = "5";
    // Hack.. don't want another file
    screen_cfg["DIR"]         = NULL;
    screen_cfg["DEFLENGTH"]   = "150";

#if defined(GP2X) || defined(A320)
    screen_cfg["CPUSCALING"] = 0;
    settings.width  = 320;
    settings.height = 240;
    screen_cfg["FIXEDFONT"] = "theme/nokia.font";
    screen_cfg["PROPFONT"]  = "";
    screen_cfg["LEDBMP"]    = "theme/nuled.bmp";
#else
    screen_cfg["FIXEDFONT"] = "theme/Fixed.font";
    screen_cfg["PROPFONT"]  = "theme/Proportional.font";
    screen_cfg["WIDTH"]     = "640";
    screen_cfg["HEIGHT"]    = "570";
    screen_cfg["LEDBMP"]    = "theme/bigled.bmp";
#endif

    fprintf(stderr, "PlayerApp::InitSettings - Loading\n");
    screen_cfg.Load();

    for(int i=0; i<Screen::MAX_COL; i++)
        if(screen_cfg[Screen::ColorNames[i]])
            settings.colors[i] = strtol(screen_cfg[Screen::ColorNames[i]], NULL, 16);
    fprintf(stderr, "PlayerApp::InitSettings - Loading\n");

    settings.font0      = screen_cfg["FIXEDFONT"];
    settings.font1      = screen_cfg["PROPFONT"];
    settings.background = screen_cfg["BACKGROUND"];
    hmargin      = strtol(screen_cfg["HMARGIN"], NULL, 10);
    vmargin      = strtol(screen_cfg["VMARGIN"], NULL, 10);
    sep          = strtol(screen_cfg["SPACING"], NULL, 10);
#if defined(GP2X) || defined(A320)
    cpu_scaling     = atoi(screen_cfg["CPUSCALING"]);
#else
    settings.width  = strtol(screen_cfg["WIDTH"],  NULL, 10);
    settings.height = strtol(screen_cfg["HEIGHT"], NULL, 10);
#endif
    
    fprintf(stderr, "PlayerApp::InitSettings - Applying settings\n");
    g_screen.Init(&settings);
    fprintf(stderr, "PlayerApp::InitSettings - OK\n");
}

void PlayerApp::RenderEq()
{
    short buff[2048*2];
    short buff2[1024*2];

    int l = music_player->GetNextSamples(buff, sizeof(buff)) / 2;

    int avg0 = 0;
    int avg1 = 0;

    if(l)
    {
        int i;
        for(i=0; i<l; i+=2)
        {
            avg0 += ABS(buff[i]);
            avg1 += ABS(buff[i+1]);
        }
        lcd->SetLed(LedControl::BAR0, avg0 / (l*256));
        lcd->SetLed(LedControl::BAR1, avg1 / (l*256));

        for(i=0; i<l/4; i++)
            buff2[i] = (buff[i*2] + buff[i*2+1]) / 2;

        if(eq)
            eq->WriteSamples(buff2, l/4);
    }
}

bool PlayerApp::PlaySong(IFileList *list, int index)
{
    bool ok = false;
    const char *name = list->GetName(index);
    const char *path = list->GetPath(index);
    int fsize        = list->GetSize(index);
    int time         = list->GetTime(index);
    int zipindex     = list->GetIndex(index);
    int subtune      = list->GetTrack(index);

    eq->ClearBars();
    if (zipindex >= 0) // Zip file!
    {
        char tmpm3ufile[128];
        fprintf(stderr, "PlayerApp::PlaySong - Loading (zipped) %s (%d) tune %d\n", 
                        path, zipindex, subtune);
        char *tmp = (char *)malloc(fsize);
        read_zip(tmp, path, fsize, zipindex);
        // This is a *really* ugly hack.
        int m3u = 0;
        if (is_ext(name,".kss") || is_ext(name,".ay") || is_ext(name,".hes"))
        {
            int safelen = strlen(name)+1;
            char tmpm3uname[safelen];
            strncpy(tmpm3uname,name,safelen);
            char *end = strrchr(tmpm3uname,'.');
            strcpy(end+1,"m3u");

            fex_t *fex;
            error( fex_open( &fex, path ) );
            
            // Search for file
            int zipindex2 = 0;
            while( !fex_done( fex ) ) {  
                char tmp[128];
                if( !strcmp(tmpm3uname, fex_name( fex )) ) {
           			error( fex_stat( fex ) );
                    int fsize2 = fex_size( fex );
                    char *tmp2 = (char *)malloc(fsize2);
                    fex_close( fex );
                    fex = NULL;        
                
                    read_zip(tmp2,path,fsize2,zipindex2);
                    // Then save it
                    sprintf(tmpm3ufile, ".tmpsongs/tmp_%s", tmpm3uname);
                    for (int i = 14; i < strlen(tmpm3ufile); i++) // Due to FAT
                        tmpm3ufile[i] = (char)tolower(tmpm3ufile[i]);
                    FILE *fp = fopen(tmpm3ufile, "wb");
                    if(fp)
                    {
                        fwrite(tmp2, 1, fsize2, fp);
                        fclose(fp);
                        m3u = 1;
                    }
                    free(tmp2);
                    
                    break;
                }

                error( fex_next( fex ) );
                zipindex2++;
            }

            fex_close( fex );
        }

        ok = music_player->PlaySong(name, tmp, fsize, time, subtune);
        free(tmp);

        if (m3u)
            remove(tmpm3ufile); // Last, remove it.
    }
    else if(list->GetPath(index))
    {
        fprintf(stderr, "PlayerApp::PlaySong - Loading (through init_file) %s\n", 
                path);
        ok = music_player->PlaySong(path, NULL, -1, time, subtune);
    }
    else
    {
        fprintf(stderr, "PlayerApp::PlaySong - Loading (through init_data) %s\n", 
                path);
        char *tmp = (char *)malloc(fsize);
        read_file(tmp,list->GetPath(index),fsize);
        ok = music_player->PlaySong(name, tmp, fsize, time, subtune);
        free(tmp);
    }
    playing = ok;
    if(playing)
    {
        lcd->SetLed(LedControl::MAXSONG, music_player->GetMaxTunes());
        lcd->SetLed(LedControl::SONGNO, music_player->GetTune()+1);
        UpdateInfo();
    }
    else
    {
        fprintf(stderr, "PlayerApp::PlaySong - Could not play file\n");
        lcd->SetLed(LedControl::MAXSONG, 0);
        lcd->SetLed(LedControl::SONGNO, 0);
        songInfo->SetFieldName(string("Error"),0);
        songInfo->SetFieldData(string("Unplayable file"),0);
        songInfo->SetLength(0);
        songInfo->SetNFields(1);
    }
    music_player->SetSongEnd(true);
    new_song = 5;
    fprintf(stderr, "PlayerApp::PlaySong - Done loading file\n");
    return ok;
}

