/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
    */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <vector>
#include <algorithm>

#include "PlayList.h"
#include "FileList.h"

#ifdef _WIN32
#define STRDUP _strdup
#else
#define STRDUP strdup
#endif

using namespace std;

PlayList::PlayList(int (*fcb)(void *, const char *), void *data) : 
	filter_cb(fcb),
	cb_data(data)
{
    shuffle_current = marked = current = -1;
    changed = 0;
    shuffled = false;
    shuffle_on = false;
}

void PlayList::Shuffle()
{
    int i;
    if(filerefs.size())
    {
        shuffle_list.resize(filerefs.size());
        for(i=0; i<filerefs.size(); i++)
            shuffle_list[i] = i;
        std::random_shuffle(&shuffle_list[0], &shuffle_list[i-1]);
        shuffled = true;
    }
}

void PlayList::AddFile(const char *name, int size, const char *path, 
                       int index, int time, int subtune)
{
    FileData data;
    data.name  = string(name);
    data.path  = string(path);
    data.size  = size;
    data.time  = time;
    data.index = index;
    data.track = subtune;
    filerefs.push_back(data);
    changed++;
}

void PlayList::Prev(bool wrap)
{
    changed++; 
    current--;
    if (wrap && current < 0)
        current = (int)filerefs.size()-1;

    if(shuffle_on)
        shuffle_current = (current < 0) ? -1 : shuffle_list[current];
}

void PlayList::Next(bool wrap)
{
    if (wrap || current >= 0)
    {
        current++;
        changed++; 
    }
    if(current >= filerefs.size())
    {
        if(wrap)
            current = 0;
        else
            current = -1;
    }

    if(shuffle_on)
        shuffle_current = (current < 0) ? -1 : shuffle_list[current];
}

int PlayList::Current()
{ 
    return shuffle_on ? shuffle_current : current;
}

void PlayList::SetCurrent(int n)
{
    changed++; 
    shuffle_current = current = (n < 0 || n  >= (int)filerefs.size()) ? -1 : n;
}

void PlayList::RemoveFile(int n)
{
    if(n >= 0 && n < filerefs.size())
    {
        if(n == current)
            current = -1;
        else
            if(n < current)
                current--;

        filerefs.erase(filerefs.begin() + n);
        if(marked >= filerefs.size())
            marked--;
        changed++;
    }
}

void PlayList::Clear()
{
    filerefs.clear();
    current = -1;
    marked = -1;
    changed++;
}

void PlayList::MoveFile(int from, int to)
{
    if(to == current)
        current++;

    filerefs.insert(filerefs.begin() + to, *(filerefs.begin() + from));

    if(to < current)
        current--;

    if(from > to)
        from++;
    RemoveFile(from);

    changed++;
}

void PlayList::Save(const char *name)
{
    FILE *fp = fopen(name, "wb");
    if(fp)
    {
        int size = this->Size();
        for(int i=0; i < size; i++)
            fprintf(fp, "Title  =%s\nFile   =%s\nIndex  =%d\nLength =%d\nSubtune=%d\n", 
                        this->GetName(i),this->GetPath(i),this->GetIndex(i),
                        this->GetTime(i)/1000,this->GetTrack(i));
        fclose(fp);
    }
}

