/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
    */
#ifndef MUSIC_PLAYER_H
#define MUSIC_PLAYER_H

#include <vector>
#include <string>
#include "plugin.h"

#define SOFTVOL 20
#define VOLSTEPS 50
#define MAXVOL 100
class BlockFifo;

extern struct sound_plugin *current_plugin;

class MusicPlayer
{
    public:
        MusicPlayer(bool);

        bool PlaySong(const char *name, char *data = NULL, int size = -1, 
                      int time_ms = -1, int subtune = -1);
        int  CanHandle(const char *name);
        bool Playing() { return playing; }
        bool Pause()   { return paused = !paused; }
        void Stop()    { songend_samples = -1; }

        int GetMaxTunes() { return (current_plugin ? current_plugin->subtunes : -1); }
        int GetTune()     { 
            if( current_plugin && (current_plugin->tune > subtune) ) 
                return current_plugin->tune;
            else return subtune; 
        }
        int GetVolume()   { return vol; }
        int GetSeconds()  { return current_plugin ? total_samples / 
                                (current_plugin->freq*current_plugin->channels) : 0; }
        int GetLength()   { return current_plugin ? current_plugin->length / 1000 : -1; }
        int GetBlend()    { return blend; }
        int GetHeavy()    { return heavy; }
        int GetNextSamples(short *dest, int len);
        
        int    GetNFields() { return (current_plugin ? current_plugin->nfields : 0); }
        string GetFieldName(int i)  { return (current_plugin && 
            current_plugin->fieldname ? current_plugin->fieldname[i] : "");}
        string GetFieldData(int i)  { return (current_plugin && 
            current_plugin->fielddata ? current_plugin->fielddata[i] : "");}

        void SetTune(int n);
        void SetVolume(int v);
        void SetDefaultLength(int sec) { deflength = sec; }
        void SetSongEnd(bool b) { songend = b; }

        void SetBlend(int b)
        { 
            if (b < 0) b = 0;
            else if (b > 128) b = 128;
	    fprintf(stderr, "MusicPlayer::SetBlend - Changing blend factor => %d\n", b);
            blend = b;
        }

    protected:
        static void audio_cb(void *userdata, Uint8 *stream, int len);
        void fill_audio(BlockFifo *af, int max);
        BlockFifo *audioFifo;
        std::vector<sound_plugin*> sound_plugins;
        bool playing;
        bool paused;
        int subtune;
        int vol;
        int total_samples;
        int songend_samples;
        int blend;
        int heavy;
        bool cpu_scaling;
        int songend; 
        int deflength;
        int InitSound(int freq, int chn);
        char temp_name[64];
};

#endif

