/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <SDL/SDL.h>
#include <vector>
#include <sys/stat.h>

#ifdef USE_DEV_MIXER
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/soundcard.h>
#endif

#include "MusicPlayer.h"
#include "FileList.h"
#include "Fifo.h"
#include "util.h"
#include "plugin.h"

#ifdef _WIN32
#include <windows.h>
#define strcasecmp stricmp 
#define strncasecmp strnicmp 
int cprintf(char *fmt, ...);
void *dlopen(const char *filename, int flag)
{ return (void *)LoadLibraryA(filename); }
void *dlsym(void *handle, const char *symbol)
{ HINSTANCE h = (HINSTANCE)handle; return (void *)GetProcAddress(h, symbol); }
#else
#include <dlfcn.h>
#endif

#if defined(GP2X)
#define BLOCK_SIZE 2048
#define BLOCK_COUNT 8
#else
#define BLOCK_SIZE 8192
#define BLOCK_COUNT 4
#endif

void hw_set_cpu(int speed);

struct sound_plugin *current_plugin = NULL;
static short audiobuf[BLOCK_SIZE * BLOCK_COUNT];
static char go_on = 1;
static char *module = NULL;
static char tmpmodfile[128] = "";
static int mixer_fd = -1;
static int current_freq;
static int current_channels;
extern int current_clockfreq;
extern bool aopen;

void MusicPlayer::fill_audio(BlockFifo *af, int maxbytes)
{
	if(current_plugin && current_plugin->fill_buffer)
	{
		int blockofbytes = af->LeftToWrite();

		if(blockofbytes)
		{
			if(maxbytes && blockofbytes > maxbytes) blockofbytes = maxbytes;
			int rc = current_plugin->fill_buffer(audiobuf, BLOCK_SIZE * blockofbytes);

            //printf("rc = %d\n",rc);
			if(!rc)
			{
				if(current_plugin->close)
					current_plugin->close();
				current_plugin = NULL;
				af->Reset();
				fprintf(stderr, "MusicPlayer::fill_audio - Done, closing song\n");
			}
			else
            {

                int b = blend;
                if (b) 
                    for (int i = 0; i < rc/2; i+=2)
                    {
                        short chn1 = (short)((((int)audiobuf[i])*(256-b) + ((int)audiobuf[i+1])*b)/256);
                        short chn2 = (short)((((int)audiobuf[i+1])*(256-b) + ((int)audiobuf[i])*b)/256);
                        audiobuf[i] = chn1;
                        audiobuf[i+1] = chn2;
                    }

                if (vol <= SOFTVOL)
                {
                    int volS = vol ? vol : 1;
                    for (int i = 0; i < rc/2; i++)
                        audiobuf[i] = (audiobuf[i]*volS)/(SOFTVOL+1);
                }
                    
				af->Write(audiobuf, rc  / BLOCK_SIZE);
            }
		}
	}
}

int MusicPlayer::GetNextSamples(short *dest, int len)
{
	int blocks = (len+BLOCK_SIZE-1)/BLOCK_SIZE;

	if(blocks > audioFifo->LeftToRead())
		blocks = audioFifo->LeftToRead();

	if(current_plugin && current_plugin->freq < 30000)
		audioFifo->Read(dest, blocks, 1);
	else
		audioFifo->Read(dest, blocks+1, 2);
	return blocks * BLOCK_SIZE; 
}

void MusicPlayer::audio_cb(void *userdata, Uint8 *stream, int len)
{
	MusicPlayer *mp = (MusicPlayer *)userdata;
	if(mp->paused == true)
	{
		memset(stream, 0, len);
		return;
	}

	//fprintf(stderr, "%d bytes, %d blocks available\n", len, mp->audioFifo->LeftToRead());
	if(mp->audioFifo->LeftToRead())
	{
#ifdef USE_DEV_MIXER
		mp->audioFifo->Read(stream, 1);
#else
		mp->audioFifo->Read16(stream, 1, mp->vol * 0x10000 / 10);
#endif
		mp->total_samples += len>>1;
        if (current_plugin->length > 0)
            mp->songend_samples = current_plugin->length * current_plugin->channels
                                    / 1000 * current_plugin->freq;
        //printf("foo %d, %d , s = %d\n",mp->total_samples,mp->songend_samples, \
            mp->GetSeconds());
		if(mp->songend_samples < 0 || (mp->total_samples > mp->songend_samples))
		{
            if (current_plugin->subtunes > (1+current_plugin->tune))
            {
                fprintf(stderr, "MusicPlayer::audio_cb - Going to next subtune %d to %d!\n", \
                                current_plugin->tune,current_plugin->tune+1);
                mp->SetTune(current_plugin->tune+1);
            }
            else
            {
                fprintf(stderr, "MusicPlayer::audio_cb: Closing song\n");
                if(current_plugin && current_plugin->close)
                    current_plugin->close();
                mp->audioFifo->Reset();
                current_plugin = NULL;
            }
		}
	}
	else
		memset(stream, 0, len);

	if(!go_on)
		return;

	mp->fill_audio(mp->audioFifo, 2);
	if(!current_plugin)
		mp->playing = false;
}

int MusicPlayer::InitSound(int freq, int chn)
{
	SDL_AudioSpec desired, obtained;
	memset(&desired, 0, sizeof(desired));
	desired.format   = SDL_BYTEORDER == SDL_BIG_ENDIAN ? AUDIO_S16MSB : AUDIO_S16LSB;
	desired.freq     = freq;
	desired.channels = chn;
	desired.samples  = BLOCK_SIZE/(2*chn);
	desired.callback = audio_cb;
	desired.userdata = this;

	fprintf(stderr, "MusicPlayer::InitSound - Open Audio\n");
	if(aopen)
    {
        fprintf(stderr, "MusicPlayer::InitSound - Closing old first\n");
		SDL_CloseAudio();
        fprintf(stderr, "MusicPlayer::InitSound - Audio closed\n");
    }

	audioFifo->Reset();
	aopen = true;
	int tries = 5;

	while(SDL_OpenAudio(&desired, &obtained) < 0)
	{
      	//fprintf(stderr, "Couldn't open SDL audio: %s\n", SDL_GetError());
      	fprintf(stderr, "MusicPlayer::InitSound - Couldn't open SDL audio: %s\n", \
                        SDL_GetError());
		SDL_Delay(500);

		if(tries == 0)
			exit(-1);
		tries--;
	}

	SDL_PauseAudio(0);
	fprintf(stderr, "MusicPlayer::InitSound - Audio Opened\n");
    current_channels = obtained.channels;
    current_freq     = obtained.freq;

#ifdef USE_DEV_MIXER
	if(mixer_fd < 0)
		mixer_fd = open("/dev/mixer", O_RDWR);
    int tmpvol = 1;//5*0x50/10;
	int mvol=(tmpvol<<8)|tmpvol;
	ioctl(mixer_fd, SOUND_MIXER_WRITE_VOLUME, &mvol); 
    SetVolume(vol);
#endif
	return (current_channels == chn && current_freq == freq);
}

#ifdef STATIC
extern "C" {
// Lossy
sound_plugin *mpg123_init_sound_plugin();
sound_plugin *vorbis_init_sound_plugin();
sound_plugin *mpc_init_sound_plugin();
sound_plugin *faad_init_sound_plugin();

// Lossless
sound_plugin *flac_init_sound_plugin();
sound_plugin *tta_init_sound_plugin();
sound_plugin *mac_init_sound_plugin();
sound_plugin *wavpack_init_sound_plugin();

// Modules
sound_plugin *sidplay_init_sound_plugin();
sound_plugin *modplay_init_sound_plugin();
sound_plugin *adplug_init_sound_plugin();
sound_plugin *timidity_init_sound_plugin();

sound_plugin *stsound_init_sound_plugin();
sound_plugin *hively_init_sound_plugin();
sound_plugin *sc68_init_sound_plugin();

sound_plugin *uade_init_sound_plugin();

// Game music
sound_plugin *gme_init_sound_plugin();
sound_plugin *pokecubed_init_sound_plugin();
sound_plugin *vgmstream_init_sound_plugin();
//sound_plugin *sexypsf_init_sound_plugin();
}
#endif

int MusicPlayer::CanHandle(const char *name)
{
	for(unsigned int i=0; i<sound_plugins.size(); i++)
		if(sound_plugins[i]->can_handle(name))
			return 1;
	return 0;
}

MusicPlayer::MusicPlayer(bool use_cpu_scaling)
{
#ifdef STATIC
	struct sound_plugin *plugin;
    if(plugin = mpg123_init_sound_plugin())    sound_plugins.push_back(plugin);
	if(plugin = gme_init_sound_plugin())       sound_plugins.push_back(plugin);
	if(plugin = sidplay_init_sound_plugin())   sound_plugins.push_back(plugin);
    // OBSOLET - Replaced by mp123plugin * if(plugin = mad_init_sound_plugin())       sound_plugins.push_back(plugin);    
	if(plugin = vorbis_init_sound_plugin())    sound_plugins.push_back(plugin);
	if(plugin = flac_init_sound_plugin())      sound_plugins.push_back(plugin);
	if(plugin = wavpack_init_sound_plugin())   sound_plugins.push_back(plugin);
	if(plugin = tta_init_sound_plugin())       sound_plugins.push_back(plugin);
	// TODO - if(plugin = mac_init_sound_plugin())       sound_plugins.push_back(plugin);
	// TODO - if(plugin = mpc_init_sound_plugin())       sound_plugins.push_back(plugin);
	if(plugin = faad_init_sound_plugin())      sound_plugins.push_back(plugin); 
	if(plugin = timidity_init_sound_plugin())  sound_plugins.push_back(plugin); 

    /* NEW ST-Sound, SC68 and HivelyTracker plugins :) */
    if(plugin = stsound_init_sound_plugin()) sound_plugins.push_back(plugin);
    if(plugin = hively_init_sound_plugin()) sound_plugins.push_back(plugin);
    if(plugin = sc68_init_sound_plugin()) sound_plugins.push_back(plugin);

    // Overlaps with modplug (best first)
    if(plugin = adplug_init_sound_plugin())    sound_plugins.push_back(plugin);
	// Doesn't handle error well (best last)
	if(plugin = modplay_init_sound_plugin())   sound_plugins.push_back(plugin);
    // OBSOLETE - Replaced by vgmstream - if(plugin = pokecubed_init_sound_plugin()) sound_plugins.push_back(plugin);

    if(plugin = vgmstream_init_sound_plugin()) sound_plugins.push_back(plugin);
    // TOO HEAVY - if(plugin = sexypsf_init_sound_plugin())   sound_plugins.push_back(plugin);
#if !defined(_MSC_VER)
	if(plugin = uade_init_sound_plugin())      sound_plugins.push_back(plugin);
#endif
 
#else
	FileList *flist = new FileList("plugins");
	fprintf(stdout, "Musicplaayer::MusicPlayer - Dynamic loading " 
        "%p %d\n", flist, flist ? flist->Size() : -1);
	for(int i=0; i<flist->Size(); i++)
	{
		const char *fn = flist->GetPath(i);
		void *ptr = dlopen(fn, 0);

		fprintf(stderr, "MusicPlayer::MusicPlayer - Plugin %s:\n", fn);
		if(ptr)
		{
			struct sound_plugin *(*init)() = (struct sound_plugin *(*)())dlsym(ptr, 
                                                "init_sound_plugin");
			if(init)
			{
				struct sound_plugin *plugin = init();
				sound_plugins.push_back(plugin);
				fprintf(stderr, "MusicPlayer::MusicPlayer - Loaded\n");
			}
			else
				fprintf(stderr, "MusicPlayer::MusicPlayer - Init failed!\n");
		}
		else
			fprintf(stderr, "MusicPlayer::MusicPlayer - Not valid sound plugin [%s]\n", dlerror());
	}
#endif

	audioFifo = new BlockFifo(BLOCK_SIZE, BLOCK_COUNT);
	memset(audiobuf, 0, BLOCK_SIZE*BLOCK_COUNT);

	songend = 1;
	current_plugin = NULL;
	vol = 10;
	subtune = 0;
	total_samples = 0;
	heavy = 0;
	// Check: Must initialize paused and blend variables, caused error on Dingux
	playing = paused = false;
	blend = 0;
	current_freq = 0;//44100;
	current_channels = 0;//2;
	songend_samples = 0;//99999999;
    cpu_scaling = use_cpu_scaling;
	//InitSound(current_freq, current_channels);
#ifndef _WIN32
	mkdir(".tmpsongs", 0x1FF);
#endif
} 

const char *BaseName(const char *fname)
{ 
	const char *ptr = &fname[strlen(fname)-1];
	while(ptr > fname && *ptr != ':' && *ptr != '\\' && *ptr != '/')
		ptr--;
	if(ptr != fname)
		ptr++;
	return ptr;
}

bool MusicPlayer::PlaySong(const char *name, char *data, int size, int time, int track)
{
    for (int i=0; i<sound_plugins.size(); i++)
	{
		if (sound_plugins[i]->can_handle(name))
		{
			int rc = -1;

			playing = false;

			fprintf(stderr, "MusicPlayer::PlaySong - plugin %d, %s\n",i,sound_plugins[i]->plugname);
			go_on = 0;
			SDL_LockAudio();
			if(current_plugin && current_plugin->close)
				current_plugin->close();
			audioFifo->Reset();

			if(module)
				free(module);
			module = NULL;

			if(strlen(tmpmodfile))
				remove(tmpmodfile);
			*tmpmodfile = 0;

			current_plugin = sound_plugins[i];
			if(data && current_plugin->init_data)
			{
				module = (char *)malloc(size);
				memcpy(module, data, size);
				rc = current_plugin->init_data((char *)name, module, size);
			}
			else if(!data && current_plugin->init_file)
			{
				rc = current_plugin->init_file((char *)name);
			}
			else if(!data && current_plugin->init_data)
			{
				struct stat ss;
				memset(&ss, 0, sizeof(struct stat));
				if(stat(name, &ss) == 0)
				{
					char *tmp = (char *)malloc(ss.st_size);
					FILE *fp = fopen(name, "rb");
					if(fp)
					{
						fread(tmp, 1, ss.st_size, fp);
						fclose(fp);
						rc = current_plugin->init_data((char *)name, tmp, ss.st_size);
						free(tmp);
					}
				}
			}
			else if(data && current_plugin->init_file)
			{
#ifndef _WIN32
				sprintf(tmpmodfile, ".tmpsongs/tmp_%s", BaseName(name));
#else
				sprintf(tmpmodfile, ".oldplay_song_%s", BaseName(name));
#endif
                for (int i = 14; i < strlen(tmpmodfile); i++) // Due to FAT
                    tmpmodfile[i] = (char)tolower(tmpmodfile[i]);
				FILE *fp = fopen(tmpmodfile, "wb");
				if(fp)
				{
					fwrite(data, 1, size, fp);
					fclose(fp);
					rc = current_plugin->init_file(tmpmodfile);
				}
                else
                    fprintf(stderr, "MusicPlayer::PlaySong - Couldn't save data to %s. %p\n",tmpmodfile,fp);
			}

			fprintf(stderr, "MusicPlayer::PlaySong - rc = %d, %d msec, %d chn, %d Hz @ %d MHz\n", \
                    rc, current_plugin->length, current_plugin->channels, current_plugin->freq, \
                    current_plugin->clockfreq);

			if(rc >= 0)
			{
                fprintf(stderr, "MusicPlayer::PlaySong - %d:%d\n", track, current_plugin->subtunes);
                if (track >= 0 && track < current_plugin->subtunes && current_plugin->set_position)
                {
                    fprintf(stderr, "MusicPlayer::PlaySong - Enforcing subtune %d\n",track);
                    current_plugin->set_position(0, track);
                    subtune = 0;
                    current_plugin->tune = 0;
                    current_plugin->subtunes = 1;
                }
                if (time > 0)
                {
                    fprintf(stderr, "MusicPlayer::PlaySong - Enforcing length %d ms\n",time);
                    current_plugin->length = time;
                } 
				total_samples = 0;    
				if(current_plugin->length > 0)
					songend_samples = (current_plugin->length * current_plugin->channels )  
                                        / 1000 * current_plugin->freq;
				else if(current_plugin->length == -1) // -1 == Plugin doesn't want to decide
                {
                    current_plugin->length = deflength*1000;
					songend_samples = deflength * current_plugin->channels * 
                                        current_plugin->freq;
                }
				else // 0 (or < -2) == Infinite ... should change this.
					songend_samples = 0x7FFFFFFE; // Approximately infinite
				subtune = current_plugin->tune;
				audioFifo->Reset();

#if defined(GP2X) || defined(A320)
                if(current_plugin->freq != current_freq || \
                   current_plugin->channels != current_channels || \
                   current_plugin->clockfreq != current_clockfreq)
                {
                    fprintf(stderr, "MusicPlayer::PlaySong - Changing, unlocking audio\n");
                    SDL_UnlockAudio();
                    //fprintf(stderr, "MusicPlayer::PlaySong - Unlocked, closing audio\n");
                    //SDL_CloseAudio();
                    //aopen = false;
                    //fprintf(stderr, "MusicPlayer::PlaySong - Closed\n");

                    if(current_plugin->clockfreq && cpu_scaling 
                        && (current_plugin->clockfreq != current_clockfreq))
                    {
                        current_clockfreq = current_plugin->clockfreq;
                        fprintf(stderr, "MusicPlayer::PlaySong - clockfreq to %d\n",current_clockfreq);
                        hw_set_cpu(current_clockfreq);
                    }
                    if (current_plugin->freq != current_freq || current_plugin->channels != current_channels)
                    {
                        int wanted = InitSound(current_plugin->freq, current_plugin->channels);
                        fprintf(stderr, "MusicPlayer::PlaySong - Plugin wanted %d Hz %d chn, gets %d Hz %d chn\n", \
                                current_plugin->freq, current_plugin->channels, current_freq, current_channels);
                        if (!wanted)
                        {
                            current_plugin->freq     = current_freq;
                            current_plugin->channels = current_channels;
                        }
                    }
                    SDL_LockAudio();
                }
#else
				if(current_plugin->freq != current_freq || current_plugin->channels != current_channels)
				{
					SDL_UnlockAudio();
					int wanted = InitSound(current_plugin->freq, current_plugin->channels);
                    fprintf(stderr, "MusicPlayer::PlaySong - Plugin wanted %d Hz %d chn, gets %d Hz %d chn\n", \
                            current_plugin->freq, current_plugin->channels, current_freq, current_channels);
                    if (!wanted)
                    {
                        current_plugin->freq     = current_freq;
                        current_plugin->channels = current_channels;
                    }
					SDL_LockAudio();
				}
#endif			
                SetVolume(vol);

				fill_audio(audioFifo, 0);
				if(!current_plugin)
					rc = -1; 
				else
				{
#ifdef A320
					heavy = current_clockfreq > 384;
#else
					heavy = current_clockfreq > 200;// ? 0 : (strcmp(current_plugin->plugname, "uade") == 0);
#endif
					playing = true;
				}
			}
            else
                songend_samples = -1;

			go_on = 1;
            if (rc >= 0)
            	fprintf(stderr, "MusicPlayer::PlaySong - Ready to play, unlocking audio\n");
            else
                fprintf(stderr, "MusicPlayer::PlaySong - Unlocking audio\n");
            SDL_UnlockAudio();
			if (rc >= 0)
                return true;
		}
	}
	return false;
}

int lastvol = -1;
void MusicPlayer::SetVolume(int v)
{
	if(v < 0)
		v = 0;
	else if(v > VOLSTEPS)
		v = VOLSTEPS;
	vol = v;
#ifdef USE_DEV_MIXER
    int volH;
    if (vol == 0) volH = 0;
    else if (vol <= SOFTVOL) volH = 1;
    else volH = vol - SOFTVOL;
    int tmpvol = vol == 0 ? 0 : 1+(MAXVOL-1)*(volH-1)/(VOLSTEPS-1-SOFTVOL);
    if (tmpvol > MAXVOL) tmpvol = MAXVOL; // Shouldn't happen, but just in case.
    //if (lastvol == tmpvol) return;
    lastvol = tmpvol;
    
    int mvol=(tmpvol<<8)|tmpvol;
    printf("MusicPlayer::SetVolume() - Hardware volume %d => %d\n",vol,tmpvol);
    // TODO - why was micket calling SOUND_MIXER_WRITE_PCM?
    //ioctl(mixer_fd, SOUND_MIXER_WRITE_PCM, &mvol); 
    ioctl(mixer_fd, SOUND_MIXER_WRITE_VOLUME, &mvol); 
#endif
}

void MusicPlayer::SetTune(int n)
{
	if(!current_plugin || !current_plugin->set_position)
		return;
    if (current_plugin->subtunes <= 1)
    {
        if (!strcasecmp(current_plugin->plugname,"uade"))
            return; // Hack'ish, but uade is a bitch (crashing if locking audio)
        // Seek n*msecs [ms] (n = +1,-1)
        SDL_LockAudio();
        int x = current_plugin->length/(1000*60);
        double msecs = 363.4*x+8181.82; // 5 min -> 10 s, 1 h -> 30 s,
        if (msecs < 10000)
            msecs = 10000;
        int actual_msecs = current_plugin->set_position((int)(n*msecs),0);
        SDL_UnlockAudio();
        int samp = total_samples + (int)((double)actual_msecs / 1000 * 
                    current_plugin->freq * current_plugin->channels);
        total_samples = samp > 0 ? samp : 0;
    }
    else
    {
        fprintf(stderr, "MusicPlayer::SetTune - Changing subtune %d\n",n);
        if(n < 0)
            n = 0;
        if(n >= current_plugin->subtunes)
            n = current_plugin->subtunes-1;
    
        if(subtune != n)
        {
            total_samples = 0;
            subtune = n;
            go_on = 0;
            SDL_LockAudio();
            current_plugin->set_position(0, n);
            if(current_plugin->length > 0)
                songend_samples = (int)((double)current_plugin->length * 
                    current_plugin->channels / 1000 * current_plugin->freq);
            else if(current_plugin->length == -1)
            {
                songend_samples = deflength * 
                    current_plugin->channels * current_plugin->freq;
                current_plugin->length = deflength*1000;
            }
            else
                songend_samples = 0x7FFFFFFE;
    
            audioFifo->Reset();
            fill_audio(audioFifo, 0);
            go_on = 1;
            SDL_UnlockAudio();
        }
    }
}

