/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <stdlib.h>
#include <SDL/SDL.h>
#include "Input.h"
#include "SDLScreen.h"

#ifdef GP2X
#define GP2X_BUTTON_UP        (0)
#define GP2X_BUTTON_UPLEFT    (1)
#define GP2X_BUTTON_LEFT      (2)
#define GP2X_BUTTON_DOWNLEFT  (3)
#define GP2X_BUTTON_DOWN      (4)
#define GP2X_BUTTON_DOWNRIGHT (5)
#define GP2X_BUTTON_RIGHT     (6)
#define GP2X_BUTTON_UPRIGHT   (7)
#define GP2X_BUTTON_START     (8)
#define GP2X_BUTTON_SELECT    (9)
#define GP2X_BUTTON_R         (10)
#define GP2X_BUTTON_L         (11)
#define GP2X_BUTTON_A         (12)
#define GP2X_BUTTON_B         (13)
#define GP2X_BUTTON_Y         (14)
#define GP2X_BUTTON_X         (15)
#define GP2X_BUTTON_VOLUP     (16)
#define GP2X_BUTTON_VOLDOWN   (17)
#define GP2X_BUTTON_CLICK     (18)
#define MOD_L                 (0x20)
#define MOD_R                 (0x40)
#define KEYREP                (0x80)
unsigned char joy_actions[0x80];
#elif defined A320
#define A320_BUTTON_UP        (273)
#define A320_BUTTON_LEFT      (276)
#define A320_BUTTON_DOWN      (274)
#define A320_BUTTON_RIGHT     (275)
#define A320_BUTTON_START     (13)
#define A320_BUTTON_SELECT    (27)
#define A320_BUTTON_R         (8)
#define A320_BUTTON_L         (9)
#define A320_BUTTON_A         (306)
#define A320_BUTTON_B         (308)
#define A320_BUTTON_Y         (304)
#define A320_BUTTON_X         (32)
#define MOD_L                 (0x200)
#define MOD_R                 (0x400)
#define KEYREP                (0x80)
unsigned int joy_actions[0x800];

unsigned int power_action = Input::QUIT;
unsigned int hold_action = Input::HOLD_BLANK;
unsigned int hw_read_power_switch_click();
unsigned int hw_read_hold_switch_click();
#else
#define  KEYREP  (0x80)
unsigned char key_actions[SDLK_LAST];
#endif

int Input::parse_button(char* x)
{
    int mod = 0;
#if defined(A320)
    if (!strncmp(x,"MOD_R_",6))
    {
        x += 6;
        mod = MOD_R;
    }

    if (!strncmp(x,"MOD_L_",6))
    {
        x += 6;
        mod = MOD_L;
    }

    if (!strncmp(x,"MOD_L_R_",8) || !strncmp(x,"MOD_R_L_",8))
    {
        x += 9;
	mod = MOD_L | MOD_R;
    }
#endif
#ifdef A320
    if (!strncmp(x,"POWER",5)) return -2;
    if (!strncmp(x,"HOLD",4)) return -3;
#endif

#ifdef GP2X
         if (!strncmp(x,"UPLEFT",   6)) return mod | GP2X_BUTTON_UPLEFT;
    else if (!strncmp(x,"UPRIGHT",  7)) return mod | GP2X_BUTTON_UPRIGHT;
    else if (!strncmp(x,"UP",       2)) return mod | GP2X_BUTTON_UP;
    else if (!strncmp(x,"DOWNLEFT", 8)) return mod | GP2X_BUTTON_DOWNLEFT;
    else if (!strncmp(x,"DOWNRIGHT",9)) return mod | GP2X_BUTTON_DOWNRIGHT;
    else if (!strncmp(x,"DOWN",     4)) return mod | GP2X_BUTTON_DOWN;
    else if (!strncmp(x,"LEFT",     4)) return mod | GP2X_BUTTON_LEFT;
    else if (!strncmp(x,"RIGHT",    5)) return mod | GP2X_BUTTON_RIGHT;
    else if (!strncmp(x,"A",        1)) return mod | GP2X_BUTTON_A;
    else if (!strncmp(x,"B",        1)) return mod | GP2X_BUTTON_B;
    else if (!strncmp(x,"X",        1)) return mod | GP2X_BUTTON_X;
    else if (!strncmp(x,"Y",        1)) return mod | GP2X_BUTTON_Y;
    else if (!strncmp(x,"START",    5)) return mod | GP2X_BUTTON_START;
    else if (!strncmp(x,"SELECT",   6)) return mod | GP2X_BUTTON_SELECT;
    else if (!strncmp(x,"VOLUP",    5)) return mod | GP2X_BUTTON_VOLUP;
    else if (!strncmp(x,"VOLDOWN",  7)) return mod | GP2X_BUTTON_VOLDOWN;
    else if (!strncmp(x,"L",        1)) return mod | GP2X_BUTTON_L;
    else if (!strncmp(x,"CLICK",    5)) return mod | GP2X_BUTTON_CLICK;
    else return -1;
#elif defined A320
         if (!strncmp(x,"UP",       2)) return mod | A320_BUTTON_UP;
    else if (!strncmp(x,"DOWN",     4)) return mod | A320_BUTTON_DOWN;
    else if (!strncmp(x,"LEFT",     4)) return mod | A320_BUTTON_LEFT;
    else if (!strncmp(x,"RIGHT",    5)) return mod | A320_BUTTON_RIGHT;
    else if (!strncmp(x,"A",        1)) return mod | A320_BUTTON_A;
    else if (!strncmp(x,"B",        1)) return mod | A320_BUTTON_B;
    else if (!strncmp(x,"X",        1)) return mod | A320_BUTTON_X;
    else if (!strncmp(x,"Y",        1)) return mod | A320_BUTTON_Y;
    else if (!strncmp(x,"START",    5)) return mod | A320_BUTTON_START;
    else if (!strncmp(x,"SELECT",   6)) return mod | A320_BUTTON_SELECT;
    else return -1;
#endif
    /*{
        fprintf(stderr, "Input::parse_button - Incorrect keymapping; %s\n",x);
        exit(-1);
    }*/
}

int Input::parse_action(char* x)
{
    int mod = 0;
    if (!strncmp(x,"+",1))
    {
        x += 1;
        mod = KEYREP;
    }
         if (!strncmp(x,"UP",           2)) return mod | UP;
    else if (!strncmp(x,"DOWN",         4)) return mod | DOWN;
    else if (!strncmp(x,"PGUP",         4)) return mod | PGUP;
    else if (!strncmp(x,"PGDOWN",       6)) return mod | PGDOWN;
    else if (!strncmp(x,"ENTER",        5)) return mod | ENTER;
    else if (!strncmp(x,"BACK",         4)) return mod | BACK;
    else if (!strncmp(x,"QUIT",         4)) return mod | QUIT;
    else if (!strncmp(x,"VOLUP",        5)) return mod | VOLUP;
    else if (!strncmp(x,"VOLDOWN",      7)) return mod | VOLDOWN;
    else if (!strncmp(x,"REPEAT",       6)) return mod | REPEAT;
    else if (!strncmp(x,"SHUFFLE",      7)) return mod | SHUFFLE;
    else if (!strncmp(x,"FF",           2)) return mod | FF;
    else if (!strncmp(x,"REW",          3)) return mod | REW;
    else if (!strncmp(x,"PAUSE",        5)) return mod | PAUSE;
    else if (!strncmp(x,"LENGTHUP",     8)) return mod | LENGTHUP;
    else if (!strncmp(x,"LENGTHDOWN",  10)) return mod | LENGTHDOWN;
    else if (!strncmp(x,"PLISTADD",     8)) return mod | PLISTADD;
    else if (!strncmp(x,"PLAYLIST",     8)) return mod | PLAYLIST;
    else if (!strncmp(x,"NEXT",         4)) return mod | NEXT;
    else if (!strncmp(x,"PREV",         4)) return mod | PREV;
    else if (!strncmp(x,"BLANK",        5)) return mod | BLANK;
    else if (!strncmp(x,"HOLD",         4)) return mod | HOLD;
    else if (!strncmp(x,"SPEED",        5)) return mod | SPEED;
    else if (!strncmp(x,"DELETE",       6)) return mod | DELETE;
    else if (!strncmp(x,"SCREENSHOT",  10)) return mod | SCREENSHOT;
    else if (!strncmp(x,"SAVE",         4)) return mod | SAVE;
    else if (!strncmp(x,"INC_SUBTUNE", 11)) return mod | INC_SUBTUNE;
    else if (!strncmp(x,"DEC_SUBTUNE", 11)) return mod | DEC_SUBTUNE;
    else if (!strncmp(x,"BLENDUP",      7)) return mod | BLENDUP;
    else if (!strncmp(x,"BLENDDOWN",    9)) return mod | BLENDDOWN;
    else if (!strncmp(x,"TOGGLEBLEND", 11)) return mod | TOGGLEBLEND;
    else if (!strncmp(x,"HOLD_BLANK",  10)) return mod | HOLD_BLANK;
    else return -1;
    /*{
        fprintf(stderr, "Input::parse_action - Incorrect keymapping; %s\n",x);
        exit(-1);
    }*/
}

Input::Input()
{

#if defined(GP2X)
    memset(joy_actions, NONE, sizeof(joy_actions));
    joy_actions[GP2X_BUTTON_UPLEFT]    = UP   | KEYREP;
    joy_actions[GP2X_BUTTON_UP]        = UP   | KEYREP;
    joy_actions[GP2X_BUTTON_UPRIGHT]   = UP   | KEYREP;
    joy_actions[GP2X_BUTTON_DOWNLEFT]  = DOWN | KEYREP;
    joy_actions[GP2X_BUTTON_DOWN]      = DOWN | KEYREP;
    joy_actions[GP2X_BUTTON_DOWNRIGHT] = DOWN | KEYREP;
    joy_actions[GP2X_BUTTON_LEFT]      = REW  | KEYREP;
    joy_actions[GP2X_BUTTON_RIGHT]     = FF   | KEYREP;

    joy_actions[GP2X_BUTTON_A]       = NEXT;
    joy_actions[GP2X_BUTTON_B]       = ENTER;
    joy_actions[GP2X_BUTTON_X]       = BACK;
    joy_actions[GP2X_BUTTON_Y]       = PLISTADD | KEYREP;
    joy_actions[GP2X_BUTTON_START]   = PAUSE;
    joy_actions[GP2X_BUTTON_SELECT]  = PLAYLIST;
    joy_actions[GP2X_BUTTON_VOLUP]   = VOLUP   | KEYREP;
    joy_actions[GP2X_BUTTON_VOLDOWN] = VOLDOWN | KEYREP;
    joy_actions[GP2X_BUTTON_L]       = SAVE;
    joy_actions[GP2X_BUTTON_CLICK]   = SCREENSHOT;

    joy_actions[MOD_R | GP2X_BUTTON_UPLEFT]    = PGUP   | KEYREP;
    joy_actions[MOD_R | GP2X_BUTTON_UP]        = PGUP   | KEYREP;
    joy_actions[MOD_R | GP2X_BUTTON_UPRIGHT]   = PGUP   | KEYREP;
    joy_actions[MOD_R | GP2X_BUTTON_DOWNLEFT]  = PGDOWN | KEYREP;
    joy_actions[MOD_R | GP2X_BUTTON_DOWN]      = PGDOWN | KEYREP;
    joy_actions[MOD_R | GP2X_BUTTON_DOWNRIGHT] = PGDOWN | KEYREP;
    joy_actions[MOD_R | GP2X_BUTTON_RIGHT]     = INC_SUBTUNE | KEYREP;
    joy_actions[MOD_R | GP2X_BUTTON_LEFT]      = DEC_SUBTUNE | KEYREP;

    joy_actions[MOD_R | GP2X_BUTTON_A]       = PREV;
    joy_actions[MOD_R | GP2X_BUTTON_B]       = SHUFFLE;
    joy_actions[MOD_R | GP2X_BUTTON_X]       = DELETE | KEYREP;
    joy_actions[MOD_R | GP2X_BUTTON_Y]       = REPEAT;
    joy_actions[MOD_R | GP2X_BUTTON_START]   = QUIT;
    joy_actions[MOD_R | GP2X_BUTTON_SELECT]  = BLANK;
    joy_actions[MOD_R | GP2X_BUTTON_VOLUP]   = LENGTHUP | KEYREP;
    joy_actions[MOD_R | GP2X_BUTTON_VOLDOWN] = LENGTHDOWN | KEYREP;
    joy_actions[MOD_R | GP2X_BUTTON_L]       = HOLD;
    joy_actions[MOD_R | GP2X_BUTTON_CLICK]   = SPEED;

    fprintf(stderr, "Input::Input - Looking for keymappings...\n");
    FILE *fp = fopen("keymappings.cfg","rb");
    if (fp)
    {   
        fprintf(stderr, "Input::Input - Loading custom keymappings\n");
        char line[64];// Maximum length, i'm lazy. Should be enough anyway
        line[63] = 0;
        for(;fgets(line,63,fp);)
        {
            char* actionstr = strrchr(line,'=');
            if (actionstr)
            {
                actionstr++;
                for (; *actionstr == ' '; actionstr++);
                int button = parse_button(line);
                int action = parse_action(actionstr);
                if (button >= 0 && action >= 0)
                    joy_actions[button] = action;
            }

        }
        fclose(fp);
    }
#elif defined(A320)
    joy_actions[A320_BUTTON_UP]        = UP   | KEYREP;
    joy_actions[A320_BUTTON_DOWN]      = DOWN | KEYREP;
    joy_actions[A320_BUTTON_LEFT]      = REW  | KEYREP;
    joy_actions[A320_BUTTON_RIGHT]     = FF   | KEYREP;

    joy_actions[A320_BUTTON_A]       = NEXT;
    joy_actions[A320_BUTTON_B]       = ENTER;
    joy_actions[A320_BUTTON_X]       = BACK;
    joy_actions[A320_BUTTON_Y]       = PLISTADD | KEYREP;
    joy_actions[A320_BUTTON_START]   = PAUSE;
    joy_actions[A320_BUTTON_SELECT]  = PLAYLIST;

    joy_actions[MOD_L | A320_BUTTON_A]       = SAVE;
    joy_actions[MOD_L | A320_BUTTON_SELECT]  = HOLD;
    joy_actions[MOD_L | A320_BUTTON_UP]      = VOLUP   | KEYREP;
    joy_actions[MOD_L | A320_BUTTON_DOWN]    = VOLDOWN | KEYREP;
    joy_actions[MOD_L | A320_BUTTON_RIGHT]   = LENGTHUP | KEYREP;
    joy_actions[MOD_L | A320_BUTTON_LEFT]    = LENGTHDOWN | KEYREP;

    joy_actions[MOD_R | A320_BUTTON_UP]      = PGUP   | KEYREP;
    joy_actions[MOD_R | A320_BUTTON_DOWN]    = PGDOWN | KEYREP;
    joy_actions[MOD_R | A320_BUTTON_RIGHT]   = INC_SUBTUNE | KEYREP;
    joy_actions[MOD_R | A320_BUTTON_LEFT]    = DEC_SUBTUNE | KEYREP;
    joy_actions[MOD_R | A320_BUTTON_A]       = PREV;
    joy_actions[MOD_R | A320_BUTTON_B]       = SHUFFLE;
    joy_actions[MOD_R | A320_BUTTON_X]       = DELETE | KEYREP;
    joy_actions[MOD_R | A320_BUTTON_Y]       = REPEAT;
    joy_actions[MOD_R | A320_BUTTON_START]   = QUIT;

    joy_actions[MOD_L | MOD_R | A320_BUTTON_A]      = SCREENSHOT;
    joy_actions[MOD_L | MOD_R | A320_BUTTON_START]  = BLANK;
    joy_actions[MOD_L | MOD_R | A320_BUTTON_SELECT] = SPEED;
    joy_actions[MOD_L | MOD_R | A320_BUTTON_UP]     = BLENDUP | KEYREP;
    joy_actions[MOD_L | MOD_R | A320_BUTTON_DOWN]   = BLENDDOWN | KEYREP;
    joy_actions[MOD_L | MOD_R | A320_BUTTON_Y]      = TOGGLEBLEND;

    fprintf(stderr, "Input::Input - Looking for keymappings...\n");
    FILE *fp = fopen("keymappings.cfg","rb");
    if (fp)
    {   
        fprintf(stderr, "Input::Input - Loading custom keymappings\n");
        char line[64];// Maximum length, i'm lazy. Should be enough anyway
        line[63] = 0;
        for(;fgets(line,63,fp);)
        {
            char* actionstr = strrchr(line,'=');
            if (actionstr)
            {
                actionstr++;
                for (; *actionstr == ' '; actionstr++);
                int button = parse_button(line);
        		/* Special case for power and hold buttons */
	        	if( button == -2 ) {
	        	    power_action = parse_action(actionstr);
	        	}
                else if( button == -3 ) {
                    hold_action = parse_action(actionstr);
                }
	        	else {
                        int action = parse_action(actionstr);
                        if (button >= 0 && action >= 0)
                            joy_actions[button] = action;
	        	}
            }
        }
        fclose(fp);
    }
#else
    key_actions[SDLK_LEFT]      = REW;
    key_actions[SDLK_RIGHT]     = FF;
    key_actions[SDLK_UP]        = UP      | KEYREP;
    key_actions[SDLK_DOWN]      = DOWN    | KEYREP;
    key_actions[SDLK_F2]        = VOLUP   | KEYREP;
    key_actions[SDLK_F1]        = VOLDOWN | KEYREP;
    key_actions[SDLK_PAGEUP]    = PGUP    | KEYREP;
    key_actions[SDLK_PAGEDOWN]  = PGDOWN  | KEYREP;
    key_actions[SDLK_RETURN]    = ENTER;
    key_actions[SDLK_BACKSPACE] = BACK;
    key_actions[SDLK_ESCAPE]    = QUIT;
    key_actions['r']            = REPEAT;
    key_actions['s']            = SHUFFLE;
    key_actions[' ']            = PAUSE;
    key_actions['p']            = PREV;
    key_actions['n']            = NEXT;
    key_actions[SDLK_F9]        = BLANK;
    key_actions[SDLK_F10]       = HOLD;
    key_actions['a']            = PLISTADD  | KEYREP;
    key_actions[SDLK_F4]        = LENGTHUP   | KEYREP;
    key_actions[SDLK_F3]        = LENGTHDOWN | KEYREP;
    key_actions[SDLK_TAB]       = PLAYLIST;
    key_actions[SDLK_F11]       = SCREENSHOT;
#endif
}

static unsigned char last_button = 0;
static int repeat = 0;
static int repeat_first = 160;
static int repeat_rate = 40;
static int mods = 0;

static int updown = 0, downdown = 0;
int Input::GetKey(int msec)
{
    SDL_Event evt;
    int ie = SDL_PollEvent(&evt);

#ifdef A320
    /* May change it's use in the future ... */
    if( hw_read_hold_switch_click() )  return hold_action;
    if( hw_read_power_switch_click() ) return power_action;
#endif

    if(ie)
    {
        switch(evt.type)
        {
#ifdef GP2X
        case SDL_JOYBUTTONUP:
            //if(evt.jbutton.button == GP2X_BUTTON_L)
            //    mods &= ~MOD_L;
            //else
            if(evt.jbutton.button == GP2X_BUTTON_R)
                mods &= ~MOD_R;
            // Hack for F200 with no stickclick
            if (evt.jbutton.button == GP2X_BUTTON_VOLUP )     updown = 0;
            if (evt.jbutton.button == GP2X_BUTTON_VOLDOWN ) downdown = 0;
            last_button = 0;
            break;
        case SDL_JOYBUTTONDOWN:
            //if(evt.jbutton.button == GP2X_BUTTON_L)
            //    mods |= MOD_L;
            //else
            if(evt.jbutton.button == GP2X_BUTTON_R)
            {
                mods |= MOD_R;
                return NONE;
            }
            if (evt.jbutton.button == GP2X_BUTTON_VOLUP ) updown = 1;
            if (evt.jbutton.button == GP2X_BUTTON_VOLDOWN ) downdown = 1;
            if (updown && downdown)
                last_button = joy_actions[GP2X_BUTTON_CLICK | mods];
            else 
                last_button = joy_actions[evt.jbutton.button | mods];

            repeat = repeat_first;
            return last_button & (~KEYREP);
            break;
#elif defined A320
        case SDL_KEYUP:
            if(evt.key.keysym.sym == A320_BUTTON_L)
                mods &= ~MOD_L;
            else
            if(evt.key.keysym.sym == A320_BUTTON_R)
                mods &= ~MOD_R;
            last_button = 0;
            break;
        case SDL_KEYDOWN:
            if(evt.key.keysym.sym == A320_BUTTON_L)
            {
                mods |= MOD_L;
                return NONE;
            }
            
            if(evt.key.keysym.sym == A320_BUTTON_R)
            {
                mods |= MOD_R;
                return NONE;
            }


            last_button = joy_actions[evt.key.keysym.sym | mods];
            repeat = repeat_first;
            return last_button & (~KEYREP);
            break;
#else
        case SDL_KEYDOWN:
            last_button = key_actions[evt.key.keysym.sym];
            repeat = repeat_first;
            return last_button & (~KEYREP);
            break;
        case SDL_VIDEORESIZE:
            g_screen.Resize(evt.resize.w, evt.resize.h);
            return RESIZE;
            break;
        case SDL_QUIT:
            exit(0);
#endif
        }
    }

    if(last_button & KEYREP)
    {
        repeat -= msec;
        if(repeat <= 0)
        {
            repeat += repeat_rate;
            return last_button & (~KEYREP);
        }
    }
    return NONE;
}
