/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
    */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <vector>
#include "FileList.h"
#include "util.h"

#define MAX_FILE_NAME 128

using namespace std;

const char *BaseName(const char *fname);

#ifdef _WIN32
#define STRDUP _strdup
#include <io.h>
struct dirent
{
    char *d_name;
};

typedef struct
{
    struct dirent de;
    struct _finddata_t fd;
    intptr_t handle;
} DIR;

DIR *opendir(char *name)
{
    char tmp[256];
    DIR *dir = (DIR*)malloc(sizeof(DIR));
    dir->de.d_name = NULL;
    sprintf(tmp, "%s\\*", name);
    //fprintf(stderr, "opendir - Looking for %s\n", tmp);
    dir->handle = _findfirst(tmp, &dir->fd);
    return dir;
}

struct dirent *readdir(DIR *dir)
{
    intptr_t rc = dir->handle;
    if(dir->de.d_name)
        rc = _findnext(dir->handle, &dir->fd);
    if(rc == -1)
        return NULL;
    dir->de.d_name = dir->fd.name;
    return &dir->de;
}

void closedir(DIR *dir)
{
    _findclose(dir->handle);
    free(dir);
}
#else
#define STRDUP strdup
#include <dirent.h>
#endif

FileList::FileList(char *dirname, int (*fcb)(void *, const char *), void *data) : 
    filter_cb(fcb),
    cb_data(data)
{
    if(dirname)
    {
        strcpy(curdir, dirname);
        Enter(dirname);
    }
    else
        *curdir = 0;
}

void FileList::AddDirectory(char *dirname, bool recursive)
{
    fprintf(stderr, "FileList::AddDirectory - Searching %s\n",dirname);
    DIR *dir = opendir(dirname);
    if(dir)
    {
        fprintf(stderr,"FileList::AddDirectory - Opening %s\n",dirname);
        char tmp[MAX_FILE_NAME];
        int count = 0;
        int folders = 0, files = 0, playlists = 0;
        for (struct dirent *de; de = readdir(dir);)
        {
            struct stat ss;
            ss.st_mode = 0;
            sprintf(tmp, "%s%c%s", dirname, SEPARATOR, de->d_name);
            if (stat(tmp, &ss) == 0 && de->d_name[0] != '.')
            {
                if (ss.st_mode & S_IFDIR || is_fex(de->d_name))
                    folders++;
                else if (is_playlist(de->d_name))
                    playlists++;
                else
                    files++;
            }
        }
        rewinddir(dir);
        char sorted[folders+playlists+files][MAX_FILE_NAME];
        int k = 0, m = 0, n = 0;
        for (struct dirent *de; de = readdir(dir);)
        {
            struct stat ss;
            ss.st_mode = 0;
            sprintf(tmp, "%s%c%s", dirname, SEPARATOR, de->d_name);
            if (stat(tmp, &ss) == 0 && de->d_name[0] != '.') {
                if (ss.st_mode & S_IFDIR || is_fex(de->d_name))
                    strcpy(sorted[k++], de->d_name);
                else if (is_playlist(de->d_name))
                    strcpy(sorted[folders+m++], de->d_name);
                else
                    strcpy(sorted[folders+playlists+n++], de->d_name);
            }
        }

        qsort(sorted,                   folders, MAX_FILE_NAME, &compare);
        qsort(&sorted[folders],       playlists, MAX_FILE_NAME, &compare);
        qsort(&sorted[folders+playlists], files, MAX_FILE_NAME, &compare);

        for(k = 0, m=folders+files+playlists; k < m ; k++)
        {
            char *tmp2 = sorted[k];
            struct stat ss;
            ss.st_mode = 0;
            sprintf(tmp, "%s%c%s", dirname, SEPARATOR, tmp2);
            if(stat(tmp, &ss) == 0)
            {
                if(!filter_cb || (ss.st_mode & S_IFDIR) || filter_cb(cb_data, tmp2))
                {
                    if(recursive && (ss.st_mode & S_IFDIR))
                        AddDirectory(tmp, recursive);
                    else
                    {
                        FileData fd;
                        fd.path = string(tmp);
                        fd.name = string(BaseName(tmp));
                        if(ss.st_mode & S_IFDIR)
                            fd.size = -1;
                        else
                            fd.size = ss.st_size;
                        fd.index = -1;
                        fd.time = -1;
                        fd.track = -1;
                        filerefs.push_back(fd);
                    }
                }
            }
        }
        closedir(dir);
        marked = start = 0;
    }
}


void FileList::Enter(char *dirname)
{
    filerefs.clear();
    AddDirectory(dirname);
}

int FileList::Enter(int index)
{
    char old[MAX_FILE_NAME] = "";
    if(index == -1)
    {
        char *end = strrchr(curdir, SEPARATOR);
        if(end)
        {
            strcpy(old, curdir);
            *end = 0;
            if(!strchr(curdir, SEPARATOR))
            {
                *end = SEPARATOR;
                end[1] = 0;
            }
        }
    }
    else
        strcpy(curdir, filerefs[index].path.c_str());

    Enter(curdir);

    if(strlen(old))
    {
        //fprintf(stderr,"FileList::Enter - Old was '%s'\n", old);
        for(unsigned int i=0; i<filerefs.size(); i++)
        {
            //fprintf(stderr,"FileList::Enter - '%s'\n", filerefs[i].path.c_str());
            if(strcmp(filerefs[i].path.c_str(), old) == 0)
            {
                marked = i;
                return i;
            }
        }
    }

    marked = 0;
    return 0;

}

