/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <string.h>
#include <stdio.h>
#include "Fifo.h"

Fifo::Fifo(int n)
{
	size = n;
	ptr = buffer = new unsigned char[size];
}

void Fifo::Reset()
{
	ptr = buffer;
}

Fifo::~Fifo()
{
	delete [] buffer;
}

int Fifo::Write(void *src, int size)
{
	if(src)
		memcpy(ptr, src, size);
	ptr += size;
	return size;
}

int Fifo::Read(void *dst, int size)
{
	int filled = ptr - buffer;
	memcpy(dst, buffer, size);
	memmove(buffer, buffer+size, filled-size);
	ptr -= size;
	return size;
}

int Fifo::Read16(void *dst, int size, int scale)
{
	int filled = ptr - buffer;
	int len = size >> 1;
	short *d = (short *)dst;
	short *s = (short *)buffer;
	while(len--)
		*d++ = ((*s++) * scale) >> 16;
	//memcpy(dst, buffer, size);
	memmove(buffer, buffer+size, filled-size);
	ptr -= size;

	return size;
}

BlockFifo::BlockFifo(int bsize, int bcount)
{
	buffer = readptr = writeptr = new unsigned char [bsize * bcount];
	blocksize = bsize;
	blockcount = bcount;
	wrap = 0;
}

BlockFifo::~BlockFifo()
{
	delete [] buffer;
}

int BlockFifo::Write(void *src, int blocks)
{
	unsigned char *s = (unsigned char*)src;
	for(int i = 0; i<blocks; i++)
	{
		memcpy(writeptr, s, blocksize);
		s += blocksize;
		writeptr += blocksize;
		if(writeptr == (buffer + blocksize * blockcount))
		{
			writeptr = buffer;
			wrap++;
		}
	}
	return blocks;
}

int BlockFifo::Read(void *dst, int blocks, int peek)
{
	unsigned char *d = (unsigned char*)dst;
	unsigned char *r = readptr;
	for(int i = 0; i<blocks; i++)
	{
		if(peek > 1)
			peek--;
		else
		{
			memcpy(d, r, blocksize);
			d += blocksize;
		}
		r += blocksize;
		if(r == (buffer + blocksize * blockcount))
		{
			r = buffer;
			if(!peek)
				wrap--;
		}
	}
	if(!peek)
		readptr = r;

	return blocks;
}

int BlockFifo::Read16(void *dst, int blocks, int scale)
{
	unsigned char *d = (unsigned char*)dst;
	for(int i = 0; i<blocks; i++)
	{
		int len = blocksize/2;
		short *dx = (short *)d;
		short *sx = (short *)readptr;
		while(len--)
			*dx++ = ((*sx++) * scale) >> 16;
		//memcpy(d, readptr, blocksize);
		d += blocksize;
		readptr += blocksize;
		if(readptr == (buffer + blocksize * blockcount))
		{
			readptr = buffer;
			wrap--;
		}
	}
	return blocks;
}

int BlockFifo::LeftToRead()
{
	int bufsize = blocksize * blockcount;

	int left = writeptr - readptr;
	if(left < 0)
		left = bufsize + left;
	else
	if(left == 0)
		left = bufsize * wrap;
	return left / blocksize;
}

int BlockFifo::LeftToWrite()
{
	int bufsize = blocksize * blockcount;

	int left = writeptr - readptr;
	if(left < 0)
		left = bufsize + left;
	else
	if(left == 0)
		left = bufsize * wrap;
	return blockcount - (left / blocksize);
}

void BlockFifo::Reset()
{
	readptr = writeptr = buffer;
	wrap = 0;
}

void *BlockFifo::WriteTarget(int blocks)
{
	writeptr += (blocksize * blocks);
	return (void *)(writeptr - (blocksize * blocks));
}

void *BlockFifo::ReadTarget(int blocks)
{
	readptr += (blocksize * blocks);
	return (void *)(readptr - (blocksize * blocks));
}

