/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <math.h>
#include <string.h>
#include <SDL/SDL.h>

extern "C" {
#include "fft/CMP_fft.h"
}
#include "Equalizer.h"

#define FALLOFF 6
#define DIVIDE 16

// Analyze batch and queue for rendering
void Equalizer::WriteSamples(short *samples, int len)
{
    int i,x,y;

    fft(samples, table, len);

    max_hi = m_h+1;//-5;
    dirty = REDRAW;

    for(i=0; i<barcount; i++)
    {
        int start = startpos[i];
        int stop  = startpos[i+1];
        int v     = 0;

        for(x=start; x<stop; x+=2)
            v += (table[x]+table[x+1]);
        v /= DIVIDE;

        if(v > max_hi)
            v = max_hi;

        if(v > eq[i])
            eq[i] = v;
    }


}

void Equalizer::ClearBars()
{
    memset(table,0,sizeof(table));
    memset(eq,0,sizeof(eq));
	SDL_Rect rect;
	rect.x = m_x;
	rect.y = m_y;
	rect.w = m_w;
	rect.h = m_h;
	SDL_FillRect(*screen, &rect, 0);
    Update();
    Render();
}

//static int lastn = -1;

bool Equalizer::Update()
{
    uint32 *p;
    int x,y;

    unsigned int bg = UINT2RGB16(settings.colors[Screen::COL_EQ_BG]);
    bg = bg | (bg<<16);

    if(max_hi <= 0)
        return false;
    max_hi -= FALLOFF;

    dirty = REDRAW;

    int width = (*screen)->w;

    unsigned short *ptr = (unsigned short *)(*screen)->pixels + m_x + m_y * 
                                                                (*screen)->w;
    unsigned short *scr = ptr;

    p = (uint32*)ptr;
    for(y=0; y<m_h/2; y++)
    {
        for(x=0; x<m_w/2; x++)
            *p++ = bg;
        p += (width-(m_w/2));
    }

    int eh = m_h & 0xFFFE;

    //for(x=0; x<m_w; x+= barsize)
    for(x=0; x<barcount; x++)
    {
        y = eq[x];
        eq[x] -= FALLOFF;
        if(eq[x] < 0)
            eq[x] = 0;

        uint16 *p = &scr[x*barsize+(eh-2)*width];
        uint16 *e = &scr[x*barsize+(eh-y)*width];
        y = 0;
        for(; p > e; p-=width*2)
        {
            int bs = barsize-1;
            uint16 *pp = p;
            while(bs--)
                *pp++ = colors[y];
            //*pp = colors2[y];
            y+=2;
        }
    }

    //g_screen.Lock();
    //g_screen.Update();
    return true;
}

void spread(int col0, int col1, int width, unsigned short *dest)
{
    int r =  col0>>16;
    int g = (col0>>8)&0xff;
    int b =  col0    &0xff;
    int dr =  ((col1>>16)        - r) * 256 / width;
    int dg = (((col1>>8) & 0xff) - g) * 256 / width;
    int db =  ((col1     & 0xff) - b) * 256 / width;

    r <<= 8;
    g <<= 8;
    b <<= 8;

    while(width--)
    {
        *dest++ = TO_RGB16((r>>8), (g>>8), (b>>8));
        r += dr;
        g += dg;
        b += db;
    }
}

void Equalizer::Setup()
{
    unsigned short tmp[64];
    int eh = m_h/2;
    if(eh > 32) 
        eh = 32;

    spread(settings.colors[Screen::COL_EQ_LOW], 
           settings.colors[Screen::COL_EQ_MID], eh, tmp);

    spread(settings.colors[Screen::COL_EQ_MID], 
           settings.colors[Screen::COL_EQ_HIGH], eh, &tmp[eh]);

    for(int i=0; i<64; i++)
    {
        colors[i] = tmp[i] | (tmp[i]<<16);
        colors2[i] = tmp[i];
    }
}

void Equalizer::Resize(int w, int h)
{
    Pane::Resize(w, h);
    Setup();
}

Equalizer::Equalizer(int x, int y, int w, int h, Screen *scr) :
    Pane(x, y, w, h, scr)
{
    int r = 0;
    int b = 0;
    int g = 255;
    uint32 l;

    max_hi   = 0;
    barcount = 26;
    barsize  = w / barcount;

    settings.colors[Screen::COL_BG] = settings.colors[Screen::COL_EQ_BG];

    int numBuckets  = 26;
    int from        = 0;
    float startfreq = 500;
    float endfreq   = 2200;
    float hzperslot = 22050.0F / (2048.0F/2.0F);

    float curfreq  = startfreq;
    float mf       = exp(log(endfreq/startfreq)/(numBuckets+1));
    float nextfreq = curfreq*mf;

    memset(eq, 0, sizeof(eq));
    int i;
    for(i=0; i<=numBuckets; i++)
    {
        startpos[i] = from;
        while(curfreq<nextfreq)
        {
                curfreq+=hzperslot;
                from+=2;
        }
        nextfreq *= mf;
    }
    startpos[i] = from;

    Setup();
/*
    for(int i=0; i<31; i++)
    {
        l = TO_RGB16(r,g,b);
        colors[i] = l | (l<<16);
        colors2[i] = l;
        r+= 256/32;
    }
        
    for(int i=31; i<64; i++)
    {
        l = TO_RGB16(r,g,b);
        colors[i] = l | (l<<16);
        colors2[i] = l;
        g -= 256/32;
        if(g < 0) g = 0;
    }
*/
    dirty = REDRAW;
}
