/*  Copyright 2006 Jonas Minnberg

    This file is part of OldPlay - a portable, multiformat musicplayer.

    OldPlay is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OldPlay is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with OldPlay; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
    */
#include <SDL/SDL.h>
#include <vector>
#include "FileList.h"
#include "PlayList.h"
#include "M3uPlayList.h"
#include "ZipFileList.h"
#include "SDLScreen.h"
#include "MusicPlayer.h"
#include "PlayerApp.h"
#include "Input.h"
#include "State.h"
#include "Config.h"
#include "util.h"
#include "zip/zip.h"

#ifndef _WIN32
#define stricmp strcasecmp
#define strnicmp strncasecmp
#endif

extern PlayerApp *app;

BrowserState g_browserState;

static IFileList *parent_file_list = NULL;
static ZipFileList *zip_list;
static int parent_selected = 0;
static int add_index = -1;

// Depends on font width.. should meassure it's length.. but it's pain in the ass.
#define TITLEMAXLEN (36)
static char temp_title[TITLEMAXLEN];
inline char* maketitle()
{
    char bigtemp[1024]; // Assumed to be big enough (for dir and zip)
    int biglen;
    const char *curdir = app->file_list->GetCurDir();
    int dirlen = strlen(curdir);
    printf("zip_list = %p\n",zip_list);
    const char* zipdir = zip_list ? zip_list->GetCurDir() : NULL;
    int ziplen = zipdir ? strlen(zipdir) : 0;
    strcpy(bigtemp,curdir);
    if (zipdir)
    {
        bigtemp[dirlen] = '/';
        const char* zipfile = app->file_list->GetName(app->file_list->Marked());
        int zipfilelen = strlen(zipfile);
        strcpy(&bigtemp[dirlen+1],zipfile);
        bigtemp[dirlen+1+zipfilelen] = '/';
        strcpy(&bigtemp[dirlen+2+zipfilelen],zipdir);
        bigtemp[dirlen+ziplen+zipfilelen+2] = 0;
        biglen = dirlen+ziplen+zipfilelen+2;
    }
    else
    {
        bigtemp[dirlen] = 0;
        biglen = dirlen;
    }
    if (biglen > TITLEMAXLEN-1)
    {
        strcpy(temp_title,bigtemp+biglen-TITLEMAXLEN+1);
        strncpy(temp_title,"...",3);
    }
    else
        strncpy(&temp_title[0],bigtemp,TITLEMAXLEN);
}

BrowserState::BrowserState()
{
    zip_list = NULL;
}

void BrowserState::Enter()
{
    printf("Enter 1! zip_list = %p\n",zip_list);
    if(zip_list)
        app->pane->SetList(zip_list);
    else
        app->pane->SetList(app->file_list);
    //maketitle();
    printf("Enter 2! %p, %s\n",temp_title, temp_title);
    app->pane->SetTitle(temp_title);
    printf("Enter 3!\n");
    app->pane->SetTitleColor(app->settings.colors[Screen::COL_BROWSER_TITLE0], 
                             app->settings.colors[Screen::COL_BROWSER_TITLE1]);
    printf("Enter 4!\n");
}
bool BrowserState::HandleKey(int key)
{
    ListPane *pane = app->pane;
    int selected = -1;
    IFileList *current_list = pane->GetList();

    if(key)
    {

        switch(key)
        {
            int m,t;
            const char *curname;
            case Input::PGUP:
                pane->DoCommand(ListPane::PGUP);
                break;
            case Input::PGDOWN:
                pane->DoCommand(ListPane::PGDN);
                break;
            case Input::UP:
                pane->DoCommand(ListPane::UP);
                break;
            case Input::DOWN:
                pane->DoCommand(ListPane::DOWN);
                break;
            case Input::ENTER:
                selected = current_list->Marked();
                break;
            case Input::BACK:
                selected = -2;
                break;
            case Input::INC_SUBTUNE:
                m = current_list->Marked();
                curname = current_list->GetName(m);
                if (m < 0 ||
                    is_ext(curname,".m3u") || is_ext(curname,".pls") || 
                    is_ext(curname,".opl") || is_fex(curname))
                    break;
                current_list->SetTrack(current_list->GetTrack(m)+1);
                pane->DoCommand(ListPane::JUMPTO,m);
                break;
            case Input::DEC_SUBTUNE:
                m = current_list->Marked();
                curname = current_list->GetName(m);
                if (m < 0 ||
                    is_ext(curname,".m3u") || is_ext(curname,".pls") || 
                    is_ext(curname,".opl") || is_fex(curname))
                    break;
                t = current_list->GetTrack(m);
                if (t >= 0)
                {
                    current_list->SetTrack(t-1);
                    pane->DoCommand(ListPane::JUMPTO,m);
                }
                break;
            case Input::LENGTHUP:
                m = current_list->Marked();
                curname = current_list->GetName(m);
                if (m < 0 ||
                    is_ext(curname,".m3u") || is_ext(curname,".pls") || 
                    is_ext(curname,".opl") || is_fex(curname))
                    break;
                t = current_list->GetTime(m);
                if (t > 0)
                    current_list->SetTime(t+5000);
                else
                    current_list->SetTime(5000);
                pane->DoCommand(ListPane::JUMPTO,m);
                break;
            case Input::LENGTHDOWN:
                m = current_list->Marked();
                curname = current_list->GetName(m);
                if (m < 0 ||
                    is_ext(curname,".m3u") || is_ext(curname,".pls") || 
                    is_ext(curname,".opl") || is_fex(curname))
                    break;
                t = current_list->GetTime(m)/5000;
                if (t <= 1)
                    current_list->SetTime(-1);
                else
                    current_list->SetTime(t*5000 - 5000);
                pane->DoCommand(ListPane::JUMPTO,m);
                break;
            case Input::PLISTADD:
                //if(add_index != current_list->Marked()) // Don't want this anymore.
                {
                    add_index = current_list->Marked();
                    const char *name = current_list->GetName(add_index);
                    const char *path = current_list->GetPath(add_index);
                    if(app->music_player->CanHandle(name))
                        app->play_list->AddFile(name, current_list->GetSize(add_index), 
                                                path, current_list->GetIndex(add_index),
                                                current_list->GetTime(add_index),
                                                current_list->GetTrack(add_index));
                    else if (is_ext(name,".m3u") || is_ext(name,".pls") || 
                             is_ext(name,".opl"))
                    {
                        fprintf(stderr, "BrowserState::HandleKey - Appending %s\n",
                                        name);
                        this->AppendPlayList(path);
                        selected = -1;
                    }
                    pane->DoCommand(ListPane::DOWN);
                }
                break;
        }
    }

    if(selected == -1)
        return true;

    add_index = -1;

    if(selected < 0)
    {
        int rc = current_list->Enter(-1);
        if(rc < 0)
        {
            delete current_list;
            current_list = parent_file_list;
            rc = parent_selected;
            parent_selected = 0;
            parent_file_list = NULL;
            zip_list = NULL;
        }
        pane->SetList(current_list);
        maketitle();
        pane->SetTitle(temp_title);

        if(rc >= 0)
            current_list->SetMarked(rc);
        pane->DoCommand(ListPane::NOTHING);
    }
    else
    {
        int fsize = current_list->GetSize(selected);
        const char *name = current_list->GetName(selected);

        if(fsize == -1)
        {
            current_list->Enter(selected);
            pane->SetList(current_list);
            maketitle();
            pane->SetTitle(temp_title);
            current_list->SetMarked(0);
            current_list->SetStart(0);
            pane->DoCommand(ListPane::NOTHING);
            selected = -1;
        }
        else if(is_fex(name))
        {
            parent_file_list = current_list;
            parent_selected = selected;
            zip_list = new ZipFileList(current_list->GetPath(selected),
                                        &(plugin_filter_zip_cb) ,app);
            printf("new zip list!\n");
            current_list = zip_list;
            pane->SetList(current_list);
            maketitle();
            pane->SetTitle(temp_title);
            selected = -1;
        }
        else if(is_ext(name, ".m3u") || is_ext(name, ".pls") || is_ext(name, ".opl"))
        {
            fprintf(stderr, "BrowserState::HandleKey - Clearing playlist and"
                            " creating new\n");
            app->play_list->Clear();
            this->AppendPlayList(current_list->GetPath(selected));
            pane->DoCommand(ListPane::DOWN);
            selected = -1;
        }
        else
        {
            app->PlaySong(current_list, selected);
            pane->DoCommand(ListPane::DOWN);
        }
    }
    return true;
}

void BrowserState::AppendPlayList(const char* fname)
{
    M3uPlayList *m3u_list = new M3uPlayList(fname);
    for (; m3u_list->Size(); m3u_list->Next())
    {
        #ifdef APPENDPLAYLISTDEBUG
        fprintf(stderr, "BrowserState::AppendPlayList - Checking entry\n"); 
        #endif
        const char* m3u_path = m3u_list->GetPath();
        const char* m3u_name = m3u_list->GetName();
        int m3u_index        = m3u_list->GetIndex();
        int m3u_time         = m3u_list->GetTime();
        int m3u_track        = m3u_list->GetTrack();
        
        long fsize = 0;
#if 0
        if (is_ext(m3u_path, ".zip"))
        {
            // Open through zip, how bothersome
            struct zip* zipf;
            if (zipf = zip_open(m3u_path, 0,NULL))
            {
                struct zip_stat zs;
                char name[256];
                zs.name = &name[0];
                if (zip_stat_index(zipf, m3u_index, 1, &zs))
                    continue;
                fsize = zs.size;
                zip_close(zipf);
                // TODO WHY WONT IT FIND THE FILENAME?!
                /*printf("zs.name = %s\n",zs.name);
                if (!app->music_player->CanHandle(name))
                    continue;
                */
            }
            else
                continue;
        }
#endif
        if (is_fex(m3u_path))
        {
            // Open through fex, how bothersome
            fex_t *fex;
            char fname[128];
            error( fex_open( &fex, m3u_path ) );

            int i = 0;
            while( !fex_done( fex ) ) {
                if( i == m3u_index ) {
                    fex_stat( fex );
                    fsize = fex_size( fex );
                    strncpy(fname, fex_name( fex ), 128);
                    fex_close( fex );

                    break;
                }
                error( fex_next( fex ) );
                i++;
            }

            if( fsize == 0 ) continue;
            /* printf("zs.name = %s\n",zs.name); */
            if (!app->music_player->CanHandle(fname))
                continue;

        }
        else
        {
            if (!app->music_player->CanHandle(m3u_path))
                continue;
            FILE* fp = fopen(m3u_path,"rb");
            if (!fp)
                continue;
            fseek(fp,0,SEEK_END);
            fsize = ftell(fp);
            fclose(fp);
        }
#ifdef APPENDPLAYLISTDEBUG
        fprintf(stderr, "BrowserState::AppendPlayList - Entry looks alright, adding\n"); 
#endif
        app->play_list->AddFile(m3u_name, fsize, m3u_path, m3u_index, 
                                m3u_time, m3u_track);
    }
#ifdef APPENDPLAYLISTDEBUG
        fprintf(stderr, "BrowserState::AppendPlayList - Done\n"); 
#endif
    delete m3u_list;
}
