/*
 * libtta.c
 *
 * Description:	 TTA input plug-in for Audacious
 * Developed by: Alexander Djourik <ald@true-audio.com>
 * Audacious port: Yoshiki Yazawa <yaz@cc.rim.or.jp>
 *
 * Copyright (c) 2007 Alexander Djourik. All rights reserved.
 *
 */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Please see the file COPYING in this directory for full copyright
 * information.
 */

/* #define AUD_DEBUG 1 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>

#include "id3tag.h"
#include "ttalib.h"

#define OUTPUT_ERROR (MEMORY_ERROR+1)
#define MAX_BSIZE (MAX_BPS>>3)
#define BYTES(x) ((x) * sizeof(id3_ucs4_t))

/* return length in letters */
size_t tta_ucs4len(id3_ucs4_t *ucs)
{
	id3_ucs4_t *ptr = ucs;
	size_t len = 0;

	while(*ptr++ != 0)
    	    len++;

	return len;
}

/* duplicate id3_ucs4_t string. new string will be terminated with 0. */
id3_ucs4_t *tta_ucs4dup(id3_ucs4_t *org)
{
	id3_ucs4_t *new = NULL;
	size_t len = tta_ucs4len(org);

	new = malloc((len + 1) * sizeof(id3_ucs4_t));
	memcpy(new, org, len * sizeof(id3_ucs4_t));
	*(new + len) = 0; //terminate

	return new;
}

id3_ucs4_t *tta_parse_genre(const id3_ucs4_t *string)
{
    id3_ucs4_t *ret = NULL;
    id3_ucs4_t *tmp = NULL;
    id3_ucs4_t *genre = NULL;
    id3_ucs4_t *ptr, *end, *tail, *tp;
    size_t ret_len = 0; //num of ucs4 char!
    size_t tmp_len = 0;
    int is_num = TRUE;

    tail = (id3_ucs4_t *)string + tta_ucs4len((id3_ucs4_t *)string);

    ret = malloc(1024);
    memset(ret, 0, 1024);

    for(ptr = (id3_ucs4_t *)string; *ptr != 0 && ptr <= tail; ptr++) {
        if(*ptr == '(') {
            if(*(++ptr) == '(') { // escaped text like: ((something)
                for(end = ptr; *end != ')' && *end != 0;) { // copy "(something)"
                    end++;
                }
                end++; //include trailing ')'
                memcpy(ret, ptr, BYTES(end - ptr));
                ret_len += (end - ptr);
                *(ret + ret_len) = 0; //terminate
                ptr = end + 1;
            }
            else {
                // reference to an id3v1 genre code
                for(end = ptr; *end != ')' && *end != 0;) {
                    end++;
                }

                tmp = malloc(BYTES(end - ptr + 1));
                memcpy(tmp, ptr, BYTES(end - ptr));
                *(tmp + (end - ptr)) = 0; //terminate
                ptr += end - ptr;

                genre = (id3_ucs4_t *)id3_genre_name((const id3_ucs4_t *)tmp);

                free(tmp);
                tmp = NULL;

                tmp_len = tta_ucs4len(genre);

                memcpy(ret + BYTES(ret_len), genre, BYTES(tmp_len));

                ret_len += tmp_len;
                *(ret + ret_len) = 0; //terminate
            }
        }
        else {
            for(end = ptr; *end != '(' && *end != 0; ) {
                end++;
            }
            // scan string to determine whether a genre code number or not
            tp = ptr;
            is_num = TRUE;
            while(tp < end) {
                if(*tp < '0' || *tp > '9') { // anything else than number appears.
                    is_num = FALSE;
                    break;
                }
                tp++;
            }
            if(is_num) {
                tmp = malloc(BYTES(end - ptr + 1));
                memcpy(tmp, ptr, BYTES(end - ptr));
                *(tmp + (end - ptr)) = 0; //terminate
                ptr += end - ptr;

                genre = (id3_ucs4_t *)id3_genre_name((const id3_ucs4_t *)tmp);
                free(tmp);
                tmp = NULL;

                tmp_len = tta_ucs4len(genre);

                memcpy(ret + BYTES(ret_len), genre, BYTES(tmp_len));

                ret_len += tmp_len;
                *(ret + ret_len) = 0; //terminate
            }
            else { // plain text
                memcpy(ret + BYTES(ret_len), ptr, BYTES(end - ptr));
                ret_len = ret_len + (end - ptr);
                *(ret + ret_len) = 0; //terminate
                ptr += (end - ptr);
            }
        }
    }

    return ret;
}

char *tta_input_id3_get_string(struct id3_tag * tag, char *frame_name)
{
    char *rtn;
    char *rtn2;
    const id3_ucs4_t *string_const;
    id3_ucs4_t *string;
    id3_ucs4_t *ucsptr;
    struct id3_frame *frame;
    union id3_field *field;
    int flagutf = FALSE;

    frame = id3_tag_findframe(tag, frame_name, 0);
    if (!frame)
        return NULL;

    if (!strcmp(frame_name, ID3_FRAME_COMMENT))
        field = id3_frame_field(frame, 3);
    else
        field = id3_frame_field(frame, 1);

    if (!field)
        return NULL;

    if (!strcmp(frame_name, ID3_FRAME_COMMENT))
        string_const = id3_field_getfullstring(field);
    else
        string_const = id3_field_getstrings(field, 0);

    if (!string_const)
        return NULL;

    string = tta_ucs4dup((id3_ucs4_t *)string_const);

    if (!strcmp(frame_name, ID3_FRAME_GENRE)) {
        id3_ucs4_t *string2 = NULL;
        string2 = tta_parse_genre(string);
        free((void *)string);
        string = string2;
    }

    ucsptr = (id3_ucs4_t *)string;
    while (*ucsptr) {
        if (*ucsptr > 0x000000ffL) {
            flagutf = TRUE;
            break;
        }
        ucsptr++;
    }

    if (flagutf) {
        rtn = (char *)id3_ucs4_utf8duplicate(string);
    }
    else {
        rtn = (char *)id3_ucs4_latin1duplicate(string);
        /*
        rtn2 = aud_str_to_utf8(rtn);
        free(rtn);
        rtn = rtn2;
        */
    }
    free(string);
    string = NULL;

    return rtn;
}

int get_id3_tags (const char *filename, tta_info *ttainfo) {
	int id3v2_size = 0;
	char *str = NULL;

	struct id3_file *id3file = NULL;
	struct id3_tag  *tag = NULL;

	id3file = id3_file_open (filename, ID3_FILE_MODE_READONLY);

	if (id3file) {
		tag = id3_file_tag (id3file);

		if (tag) {
			ttainfo->ID3.id3has = 1;
			id3v2_size = tag->paddedsize;

			str = tta_input_id3_get_string (tag, ID3_FRAME_ARTIST);
			if(str) strncpy((char *)ttainfo->ID3.artist, str, MAX_LINE);
			free(str);
			str = NULL;

			str = tta_input_id3_get_string (tag, ID3_FRAME_ALBUM);
			if(str) strncpy((char *)ttainfo->ID3.album, str, MAX_LINE);
			free(str);
			str = NULL;

			str = tta_input_id3_get_string (tag, ID3_FRAME_TITLE);
			if(str) strncpy((char *)ttainfo->ID3.title, str, MAX_LINE);
			free(str);
			str = NULL;

			str = tta_input_id3_get_string (tag, ID3_FRAME_YEAR);
			if(!str) str = tta_input_id3_get_string (tag, "TYER");
			if(str) strncpy((char *)ttainfo->ID3.year, str, MAX_YEAR);
			free(str);
			str = NULL;

			str = tta_input_id3_get_string (tag, ID3_FRAME_TRACK);
			if(str) strncpy((char *)ttainfo->ID3.track, str, MAX_TRACK);
			free(str);
			str = NULL;

			str = tta_input_id3_get_string (tag, ID3_FRAME_GENRE);
			if(str) strncpy((char *)ttainfo->ID3.genre, str, MAX_GENRE);
			free(str);
			str = NULL;

			str = tta_input_id3_get_string (tag, ID3_FRAME_COMMENT);
			if(str) strncpy((char *)ttainfo->ID3.comment, str, MAX_LINE);
			free(str);
			str = NULL;
		}

		id3_file_close(id3file);
	}

	return id3v2_size; // not used
}

