/*
 * Adplug - Replayer for many OPL2/OPL3 audio file formats.
 * Copyright (C) 1999 - 2007 Simon Peter <dn.tlp@gmx.net>, et al.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * adplug.cpp - CAdPlug utility class, by Simon Peter <dn.tlp@gmx.net>
 */

#include <string>
#include <binfile.h>

#include "adplug.h"
#include "debug.h"

/***** Replayer includes *****/

#include "hsc.h"
#include "amd.h"
#include "a2m.h"
#include "imf.h"
#include "sng.h"
#include "adtrack.h"
#include "bam.h"
#include "d00.h"
#include "dfm.h"
#include "hsp.h"
#include "ksm.h"
#include "mad.h"
#include "mid.h"
#include "mkj.h"
#include "cff.h"
#include "dmo.h"
#include "s3m.h"
#include "dtm.h"
#include "fmc.h"
#include "mtk.h"
#include "rad.h"
#include "raw.h"
#include "sa2.h"
#include "bmf.h"
#include "flash.h"
#include "hybrid.h"
#include "hyp.h"
#include "psi.h"
#include "rat.h"
#include "lds.h"
#include "u6m.h"
#include "rol.h"
#include "xsm.h"
#include "dro.h"
#include "msc.h"
#include "rix.h"
#include "adl.h"
#include "jbm.h"

/***** CAdPlug *****/
typedef CPlayer *(*Factory)(Copl *);

struct AdPlayers {
	Factory factory;
	const char *type;
	const char *ext;
} Players[] = {
  { ChscPlayer::factory, "HSC-Tracker", ".hsc\0" },
  { CsngPlayer::factory, "SNGPlay", ".sng\0" },
  { CimfPlayer::factory, "Apogee IMF", ".imf\0.wlf\0.adlib\0" },
  { Ca2mLoader::factory, "Adlib Tracker 2", ".a2m\0" },
  { CadtrackLoader::factory, "Adlib Tracker", ".sng\0" },
  { CamdLoader::factory, "AMUSIC", ".amd\0" },
  { CbamPlayer::factory, "Bob's Adlib Music", ".bam\0" },
  { Cd00Player::factory, "Packed EdLib", ".d00\0" },
  { CdfmLoader::factory, "Digital-FM", ".dfm\0" },
  { ChspLoader::factory, "HSC Packed", ".hsp\0" },
  { CksmPlayer::factory, "Ken Silverman Music", ".ksm\0" },
  { CmadLoader::factory, "Mlat Adlib Tracker", ".mad\0" },
  { CmidPlayer::factory, "MIDI", ".mid\0.cmf\0.sci\0.laa\0" },
  { CmkjPlayer::factory, "MKJamz", ".mkj\0" },
  { CcffLoader::factory, "Boomtracker", ".cff\0" },
  { CdmoLoader::factory, "TwinTeam", ".dmo\0" },
  { Cs3mPlayer::factory, "Scream Tracker 3", ".s3m\0" },
  { CdtmLoader::factory, "DeFy Adlib Tracker", ".dtm\0" },
  { CfmcLoader::factory, "Faust Music Creator", ".sng\0" },
  { CmtkLoader::factory, "MPU-401 Trakker", ".mtk\0" },
  { CradLoader::factory, "Reality Adlib Tracker", ".rad\0"  },
  { CrawPlayer::factory, "RdosPlay RAW", ".raw\0"  },
  { Csa2Loader::factory, "Surprise! Adlib Tracker", ".sat\0.sa2\0" }, 
  { CxadbmfPlayer::factory, "BMF Adlib Tracker", ".xad\0" },
  { CxadflashPlayer::factory, "Flash", ".xad\0" },
  { CxadhybridPlayer::factory, "Hybrid", ".xad\0" },
  { CxadhypPlayer::factory, "Hypnosis", ".xad\0" },
  { CxadpsiPlayer::factory, "PSI", ".xad\0" },
  { CxadratPlayer::factory, "rat", ".xad\0" },
  { CldsPlayer::factory, "LOUDNESS Sound System", ".lds\0" },
  { Cu6mPlayer::factory, "Ultima 6 Music", ".m\0" },
  { CrolPlayer::factory, "Adlib Visual Composer", ".rol\0" },
  { CxsmPlayer::factory, "eXtra Simple Music", ".xsm\0" },
  { CdroPlayer::factory, "DOSBox Raw OPL", ".dro\0" },
  { CmscPlayer::factory, "Adlib MSC Player", ".msc\0" },
  { CrixPlayer::factory, "Softstar RIX OPL Music", ".rix\0" },
  { CadlPlayer::factory, "Westwood ADL", ".adl\0"},
  { CjbmPlayer::factory, "JBM Adlib Music", ".jbm\0" },
  { NULL, "\0"}
  };

CAdPlugDatabase *CAdPlug::database = 0;

const char *get_extension(const char *ext, unsigned int n)
{
  const char	*i = ext;
  unsigned int	j;

  for(j = 0; j < n && (*i); j++, i += strlen(i) + 1) ;
  return (*i != '\0' ? i : 0);
}

CPlayer *CAdPlug::factory(const std::string &fn, Copl *opl,
			  const CFileProvider &fp)
{
  CPlayer *p;
  int i, j;

  AdPlug_LogWrite("*** CAdPlug::factory(\"%s\",opl,fp) ***\n", fn.c_str());
  
  // Try all players one by one
  for(i = 0; Players[i].factory != NULL; i++) {
	for(j = 0; get_extension(Players[i].ext, j); j++) {
	  if(fp.extension(fn, get_extension(Players[i].ext, j))) {
	    if((p = Players[i].factory(opl)))
	      if(p->load(fn, fp)) {
	        return p;
	      } else
	      delete p;
	  }
    }
  }

  // Unknown file
  AdPlug_LogWrite("End of list!\n");
  AdPlug_LogWrite("--- CAdPlug::factory ---\n");
  return 0;
}

void CAdPlug::set_database(CAdPlugDatabase *db)
{
  database = db;
}

std::string CAdPlug::get_version()
{
  return std::string(VERSION);
}

void CAdPlug::debug_output(const std::string &filename)
{
  AdPlug_LogFile(filename.c_str());
  AdPlug_LogWrite("CAdPlug::debug_output(\"%s\"): Redirected.\n",filename.c_str());
}
