#include "game_logic.h"
#include "panmines.h"

#include <stack>
#include <cmath>
#include <iostream>
#include <sstream>
#include <fstream>
using namespace std;
#include <stdio.h>
#include <stdlib.h>
#include <time.h>


typedef struct{
	int x;
	int y;
} coord;


#define CELL(x,y) this->parent->getCell(x,y)
Cell::Cell(Grid *p, int posx, int posy, bool mine) :	
	GLObject((GLfloat)(GRID_LEFT+CELL_WIDTH*posx), (GLfloat)(GRID_TOP+CELL_HEIGHT*posy)), 
	x(posx), y(posy), parent(p), faceValue(0), haveMine(mine), shown(false), flaged(false)
{
	this->drawCell(-1);
}

bool Cell::mined() {
	return this->haveMine;
}

void Cell::setMined(bool m) {
	this->haveMine = m;
}

void Cell::setValue(int v) {
	this->faceValue = v;
}

void Cell::setValue() {
	int c = 0;
	if (this->y > 0) {
		if (this->x > 0 && CELL(this->x-1, this->y-1)->mined())			c++;
		if (CELL(this->x, this->y-1)->mined())					c++;
		if (this->x < (GRID_WIDTH-1) && CELL(this->x+1, this->y-1)->mined())	c++;
	}
	if (this->y < (GRID_HEIGHT-1)) {
		if (this->x > 0 && CELL(this->x-1, this->y+1)->mined())			c++;
		if (CELL(this->x, this->y+1)->mined())					c++;
		if (this->x < (GRID_WIDTH-1) && CELL(this->x+1, this->y+1)->mined())	c++;
	}
	if (this->x > 0 && CELL(this->x-1, this->y)->mined())				c++;
	if (this->x < (GRID_WIDTH-1) && CELL(this->x+1, this->y)->mined())		c++;

	this->faceValue = c;
}

int  Cell::getValue() {
	return this->faceValue;
}

bool Cell::isFlaged() {
	return this->flaged;
}

bool Cell::isShown() {
	return this->shown;
}

void Cell::reset() {
	this->drawCell(-1);
	this->haveMine = false;
	this->shown = false;
	this->flaged = false;
	this->faceValue=0;
}


void Cell::drawFlag(GLpoint a, int w, int h, GL_Color bg, GL_Color f, GL_Color b) {
	GLpoint i = {(GLfloat)(a.x+floor(w*0.25)), (GLfloat)(a.y+(GLfloat)floor(h*0.25))};
	GLpoint k = {(GLfloat)(a.x+floor(w*0.25))+3, (GLfloat)(a.y+floor(h*0.25))};
	this->addRect(a, w, h, bg);
	this->addRect(i, 2, floor(h*0.60), b);
	this->addRect(k, floor(w*0.45), floor(h*0.25), f);
}

void Cell::drawMine(GLpoint a, int r1, int r2, GL_Color ci, GL_Color co, int prec) {
	GLpoint pt1 = {a.x,a.y-r2};
	GLpoint pt2 = {a.x,a.y+r2};
	this->addLine(pt1, pt2, co, 4);
	pt1.x = a.x-r2; pt1.y = a.y;
	pt2.x = a.x+r2; pt2.y = a.y;
	this->addLine(pt1, pt2, co, 4);
	pt1.x = a.x+floor(r2*0.71);pt1.y=a.y-floor(r2*0.71);
	pt2.x = a.x-floor(r2*0.71);pt2.y=a.y+floor(r2*0.71);
	this->addLine(pt1, pt2, co, 6);
	pt1.x = a.x-floor(r2*0.71);pt1.y=a.y-floor(r2*0.71);
	pt2.x = a.x+floor(r2*0.71);pt2.y=a.y+floor(r2*0.71);
	this->addLine(pt1, pt2, co, 6);
	this->addDisc(a, r1, ci, co,0,360,20,-r1/2,-r1/2);
}


void Cell::drawCell(int v) {
	GLpoint a  	= {1, 1, 0};
	GLpoint a2 	= {CELL_WIDTH-1, CELL_HEIGHT-1, 0};
	GLpoint b  	= {CELL_WIDTH-1, 1, 0};
	GLpoint b2 	= {1, CELL_HEIGHT-1, 0};
	GLpoint z  	= {(GLfloat)floor(CELL_WIDTH/2), (GLfloat)floor(CELL_HEIGHT/2), 0};
	GL_Color red	= {1,   0.2, 0.3, 1};
	GL_Color dred	= {0.7, 0.1, 0.2, 1};
	GL_Color lgrey	= {0.7, 0.7, 0.7, 1};
	GL_Color dgrey	= {0.4, 0.4, 0.4, 1};
	GL_Color white	= {1,   1,   1,   1};
	GL_Color black	= {0,   0,   0,   1};
	GL_Color c1	= {0,   0.3, 1,   1};
	GL_Color c2	= {0,   1,   0.2, 1};
	GL_Color c3	= {1,   0.25, 0,   1};
	GL_Color c4	= {(float)1.0, (v%2==0)?(float)(8-v)/8:0, (v%2==0)?0:(float)(8-v)/8,1};
	GL_Color c;
	this->segs.clear();
	switch(v) {
	case -1:
		this->addButton(a, CELL_WIDTH-3, CELL_HEIGHT-3, lgrey, dgrey);
	break;
	case 10:
		this->drawFlag(a, CELL_WIDTH-3, CELL_HEIGHT-3, dgrey, red, white);
	break;
	case 11:
		this->drawFlag(a, CELL_WIDTH-3, CELL_HEIGHT-3, dgrey, red, white);
		this->addLine(a, a2, black, 6);
		this->addLine(b, b2, black, 6);
	break;
	case 0:
		this->addRect(a, CELL_WIDTH-3, CELL_HEIGHT-3, dgrey);
	break;
	case 9:
		this->addButton(a, CELL_WIDTH-3, CELL_HEIGHT-3, dred, red);
		this->drawMine(z, floor(min(CELL_HEIGHT, CELL_WIDTH)*0.34), floor(min(CELL_HEIGHT, CELL_WIDTH)*0.45), dgrey, black);
	break;
	default:
		this->addRect(a, CELL_WIDTH-3, CELL_HEIGHT-3, dgrey);
		stringstream s;
		s << "G" << v;
		Texture* t = this->parent->s->getTexture(s.str());
		switch(v) {
		case 1:
			c = c1;
		break;
		case 2:
			c = c2;
		break;
		case 3:
			c = c3;
		break;
		default:
			c = c4;
		break;
		}
		this->addTextureCenter(z, t, c);
	break;
	}
}

void Cell::show() {
	this->drawCell(this->mined()?9:this->faceValue);
	this->shown = true;
}

void Cell::clicked(bool f) {
	if (f) {
		if (this->isFlaged()) {
			this->drawCell(-1);
			this->flaged = false;
		} else if (!this->isShown()) {
			this->drawCell(10);
			this->flaged = true;
		}
	} else if (!this->isFlaged()) {
		this->show();
		if (this->mined()) {
			for(int i=0;i<GRID_HEIGHT;i++) {
				for(int j=0;j<GRID_WIDTH;j++) {
					if (CELL(j,i)->mined() && !CELL(j,i)->isFlaged())
						this->parent->getCell(j,i)->show();
					if (!CELL(j,i)->mined() && CELL(j,i)->isFlaged())
						this->parent->getCell(j,i)->drawCell(11); //mark as err in flag	
				}
			}
			this->parent->endGame();
		} else if (this->faceValue == 0) {
			coord p = {this->x,this->y};
			stack<coord> st;
			st.push(p);
			while (!st.empty()){
				coord c = st.top();
				st.pop();
				CELL(c.x, c.y)->show();
				if (c.x>0 ) {
					if (CELL(c.x-1, c.y)->getValue()==0 && ! CELL(c.x-1, c.y)->isShown() && !CELL(c.x-1, c.y)->isFlaged()){
						p.x=c.x-1;p.y=c.y;st.push(p);
					}else if (!CELL(c.x-1, c.y)->isFlaged())
						CELL(c.x-1, c.y)->show();

					if (c.y>0 && CELL(c.x-1, c.y-1)->getValue()==0 && ! CELL(c.x-1, c.y-1)->isShown() && !CELL(c.x-1, c.y-1)->isFlaged()) {
						p.x=c.x-1;p.y=c.y-1;st.push(p);
					}else if(c.y>0 && !CELL(c.x-1, c.y-1)->isFlaged())
						CELL(c.x-1, c.y-1)->show();

					if (c.y<GRID_HEIGHT-1 && CELL(c.x-1, c.y+1)->getValue()==0 && ! CELL(c.x-1, c.y+1)->isShown() && !CELL(c.x-1, c.y+1)->isFlaged()) {
						p.x=c.x-1;p.y=c.y+1;st.push(p);
					}else if(c.y<GRID_HEIGHT-1 && ! CELL(c.x-1, c.y+1)->isFlaged())
						CELL(c.x-1, c.y+1)->show();
				}
				if (c.x<GRID_WIDTH-1 ) {
					if (CELL(c.x+1, c.y)->getValue()==0 && ! CELL(c.x+1, c.y)->isShown() && !CELL(c.x+1, c.y)->isFlaged()) {
						p.x=c.x+1;p.y=c.y;st.push(p);
					}else if (!CELL(c.x+1, c.y)->isFlaged())
						CELL(c.x+1, c.y)->show();

					if (c.y>0 && CELL(c.x+1, c.y-1)->getValue()==0 && ! CELL(c.x+1, c.y-1)->isShown() && ! CELL(c.x+1, c.y-1)->isFlaged()) {
						p.x=c.x+1;p.y=c.y-1;st.push(p);
					}else if(c.y>0 && !CELL(c.x+1, c.y-1)->isFlaged())
						CELL(c.x+1, c.y-1)->show();

					if (c.y<GRID_HEIGHT-1 && CELL(c.x+1, c.y+1)->getValue()==0 && ! CELL(c.x+1, c.y+1)->isShown() && ! CELL(c.x+1, c.y+1)->isFlaged()) {
						p.x=c.x+1;p.y=c.y+1;st.push(p);
					}else if(c.y<GRID_HEIGHT-1 && ! CELL(c.x+1, c.y+1)->isFlaged())
						CELL(c.x+1, c.y+1)->show();
				}
				if (c.y>0 && CELL(c.x, c.y-1)->getValue()==0 && ! CELL(c.x, c.y-1)->isShown()&& ! CELL(c.x, c.y-1)->isFlaged()) {
					p.x=c.x;p.y=c.y-1;st.push(p);
				}else if(c.y>0 && ! CELL(c.x, c.y-1)->isFlaged())
					this->parent->getCell(c.x, c.y-1)->show();

				if (c.y<GRID_HEIGHT-1 && CELL(c.x, c.y+1)->getValue()==0 && ! CELL(c.x, c.y+1)->isShown() && ! CELL(c.x, c.y+1)->isFlaged()){
					p.x=c.x;p.y=c.y+1;st.push(p);
				}else if(c.y<GRID_HEIGHT-1 && ! CELL(c.x, c.y+1)->isFlaged())
					this->parent->getCell(c.x, c.y+1)->show();
			}
		}
	}
}







intDisplay::intDisplay(GLScreen *s, int v) : GLObject(), s(s), value(v), fontH(0) {
	this->fontH = this->s->getTexture("s0")->h;
	this->drawInt();
}
void intDisplay::setValue(int v) {
	this->value = v;
	this->drawInt();
}
int intDisplay::getValue() {
	return this->value;
}
GLScreen *intDisplay::getScreen() {
	return this->s;
}

void intDisplay::drawBox() {
	int x, r, r2;

	GL_Color bg = {0.3, 0.3, 0.3, 1};
	GL_Color black = {0, 0, 0, 1};
	GL_Color white = {1, 1, 1, 1};

	GLfloat  w  = (GLfloat)floor(23*GRID_LEFT/25);
	GLfloat  h  = SCREEN_HEIGHT/15;

	GLpoint  a  = {0,0,0};
	GLpoint  b  = {1,1,0};

	this->clear();
	this->addRect(a, w, h, bg);
	this->addBox(b, w-1, h-1, black);
}

void intDisplay::drawInt() {
	stringstream ss;
	int x, r, r2;

	GL_Color white = {1, 1, 1, 1};

	GLfloat  w2 = (GLfloat)floor(23*GRID_LEFT/125);
	GLfloat  h  = SCREEN_HEIGHT/15;

	GLpoint  n1 = {w2*3.6,h-this->fontH/3,0};
 	GLpoint  n2 = {w2*4.6,h-this->fontH/3,0};

	this->drawBox();
	
	r2 = this->value<0?0:this->value%100;
	r  = r2%10;	x = (r2 - r)/10;
	ss.str("");	ss << "s" << x;
	this->addTextureCenter(n1, this->s->getTexture(ss.str()), white);
	ss.str("");	ss << "s" << r;
	this->addTextureCenter(n2, this->s->getTexture(ss.str()), white);
}




timeDisplay::timeDisplay(GLScreen *s, int v) : intDisplay(s,v), startTicks(0), started(false) {
	this->drawTime();
}

void timeDisplay::drawTime() {
	stringstream ss;
	int x, r, r2;

	GL_Color white = {1, 1, 1, 1};

	GLfloat  w2 = (GLfloat)floor(23*GRID_LEFT/125);
	GLfloat  h  = SCREEN_HEIGHT/15;

	GLpoint  n1 = {w2*1,h-this->fontH/3,0};
	GLpoint  n2 = {w2*2,h-this->fontH/3,0};
	GLpoint  n3 = {w2*3.6,h-this->fontH/3,0};
 	GLpoint  n4 = {w2*4.6,h-this->fontH/3,0};
	GLpoint  d1 = {w2*2.5, h/3, 0};
	GLpoint  d2 = {w2*2.5, 2*h/3, 0};

	this->drawBox();
	this->addRect(d1, 2, 2, white);
	this->addRect(d2, 2, 2, white);
	
	r2 = this->value%6000;
	r  = r2%600;	x = (r2 - r)/600;
	ss.str("");	ss << "s" << x;
	this->addTextureCenter(n1, this->getScreen()->getTexture(ss.str()), white);
	r2 = r;		r  = r2%60;	x = (r2 - r)/60;
	ss.str("");	ss << "s" << x;
	this->addTextureCenter(n2, this->getScreen()->getTexture(ss.str()), white);
	r2 = r;		r  = r2%10;	x = (r2 - r)/10;
	ss.str("");	ss << "s" << x;
	this->addTextureCenter(n3, this->getScreen()->getTexture(ss.str()), white);
	ss.str("");	ss << "s" << r;
	this->addTextureCenter(n4, this->getScreen()->getTexture(ss.str()), white);
}

void timeDisplay::setValue(int v) {
	this->value = v;
	this->drawTime();
}
void timeDisplay::start() {
	this->started = true;
	this->startTicks = SDL_GetTicks();
}

void timeDisplay::stop() {
	this->started = false;
}

bool timeDisplay::running() {
	return this->started;
}

void timeDisplay::reset() {
	this->started = false;
	this->value   = 0;
	this->drawTime();
}

void timeDisplay::update() {
	int v;
	if (this->started) {
		v = floor((SDL_GetTicks()-this->startTicks)/1000);
		if (v>this->value) {
			this->value   = v;
			this->drawTime();
		}
	}
}







highScore::highScore(GLScreen *s, int v) : level(2), timeDisplay(s,v) {
	this->score[0] = v;
	this->score[1] = v;
	this->score[2] = v;
}

void highScore::loadFrom(string fname) {
	string line, tmp;
	int ln, scn;
	ifstream file (fname.c_str());
	if (!file.is_open())
		return;

	while(getline(file, line)) {
		istringstream iss(line);
		
		if (iss>>ln && ln>0 && ln<4) {
			if (iss >> scn)
				this->score[ln-1] = scn;
			else
			{
				cout << "Failed parsing the score file. ln= " << ln << " Line = " << line << endl;
				file.close();
				return;
			}
		}
		else
		{
			cout << "Failed parsing the score file. Line = " << line << endl;
			file.close();
			return;
		}
	}
	file.close();
	this->setValue(this->score[this->level-1]);
}

void highScore::saveTo(string fname) {
	this->score[this->level-1] = this->value;
	ofstream file (fname.c_str());
	if (!file.is_open())
	{
		cout << "Failed saving the score file" << endl;
		return;
	}

	for(int i=1;i<4;i++)
		file << i << " " << this->score[i-1] << endl;
	file.close();
}

void highScore::setLevel(int level) {
	this->score[this->level-1] = this->value;
	if (level != this->level && level >0 && level <4 ) {
		this->setValue(this->score[level-1]);
		this->level = level;
	}
}









cursor::cursor(GLfloat x, GLfloat y, GLfloat w, GLfloat h) : w(w), h(h), GLObject() {
	GLpoint a  = {0,	0,	0};
	GLpoint a1 = {w/4,	0,	0};
	GLpoint a2 = {0,	h/4,	0};
	GLpoint b  = {w,	0,	0};
	GLpoint b1 = {3*w/4,	0,	0};
	GLpoint b2 = {w,	h/4,	0};
	GLpoint c  = {0,	h,	0};
	GLpoint c1 = {w/4,	h,	0};
	GLpoint c2 = {0,	3*h/4,	0};
	GLpoint d  = {w,	h,	0};
	GLpoint d1 = {3*w/4,	h,	0};
	GLpoint d2 = {w,	3*h/4,	0};
	GL_Color col = {0.8,0.8,0,1};
	this->addLine(a, a1, col, 2);
	this->addLine(a, a2, col, 2);
	this->addLine(b, b1, col, 2);
	this->addLine(b, b2, col, 2);
	this->addLine(c, c1, col, 2);
	this->addLine(c, c2, col, 2);
	this->addLine(d, d1, col, 2);
	this->addLine(d, d2, col, 2);
	this->moveTo(x,y);
}





gridCursor::gridCursor(int posx, int posy) : posx(posx), posy(posy), cursor(GRID_LEFT+CELL_WIDTH*posx,GRID_TOP+CELL_HEIGHT*posy,CELL_WIDTH-1, CELL_HEIGHT-1) {	
}
void gridCursor::setPos(int posx, int posy) {
	this->posx = posx;
	this->posy = posy;
	this->moveTo(GRID_LEFT+CELL_WIDTH*posx,GRID_TOP+CELL_HEIGHT*posy);
}
int gridCursor::getPosX() {
	return this->posx;
}
int gridCursor::getPosY() {
	return this->posy;
}
void gridCursor::moveUp() {
	this->setPos(this->posx, this->posy>0?this->posy-1:this->posy);
}
void gridCursor::moveDown() {
	this->setPos(this->posx, this->posy<GRID_HEIGHT-1?this->posy+1:this->posy);
}
void gridCursor::moveLeft() {
	this->setPos(this->posx>0?this->posx-1:this->posx, this->posy);
}
void gridCursor::moveRight() {
	this->setPos(this->posx<GRID_WIDTH-1?this->posx+1:this->posx, this->posy);
}








faceButton::faceButton(GLfloat x, GLfloat y, GLfloat w, GLfloat h, bool hap) : w(w), h(h), happy(hap), GLObject() {
	this->moveTo(x,y);
	this->drawSegments();
}
void faceButton::setHappy(bool happy) {
	this->happy = happy;
	this->drawSegments();
}
void faceButton::drawFace(GLpoint a, bool happy, int radius) {
	GL_Color black = {0,   0,   0,   1};
	GL_Color orang = {1,0.6,0.1,1};
	GL_Color yellow = {0.9,0.9,0.1,1};
	GLpoint e1 = {a.x-radius/3.4,a.y-radius/2};
	GLpoint e2 = {a.x+radius/3.4,a.y-radius/2};
	GLpoint r = {a.x,a.y+radius/2};
	this->addDisc(a, radius, orang, yellow); //, 0, 360, radius/4, -radius/3);
	this->addArc(a, radius, black, 1);
	this->addRect(e1, 2, 3, black);
	this->addRect(e2, 2, 3, black);
	if (happy)
		this->addArc(a, radius/2, black, 2, -90, 90);
	else
		this->addArc(r, radius/2, black, 2, 90, 270);
}

void faceButton::drawSegments() {
	GLpoint a = {0, 0};
	GLpoint a2 = {1, 1};
	GLpoint b = {this->w/2,this->h/2};
	GL_Color green = {0.02,1,0.05,1};
	GL_Color red = {1,0.05,0.05,1};
	GL_Color black = {0,   0,   0,   1};
	if (this->happy)
		this->addRect(a,FACE_SIZE,FACE_SIZE,green);
	else
		this->addRect(a,FACE_SIZE,FACE_SIZE,red);
	this->addBox(a2,FACE_SIZE-1,FACE_SIZE-1,black);
	this->drawFace(b, this->happy, 0.4*FACE_SIZE);
}







DigSwitch::DigSwitch(GLScreen *s, GLfloat x, GLfloat y, GLfloat w, GLfloat h, bool dig):x(x),y(y),w(w),h(h),dig(dig) {
	Texture* t = s->getTexture("M?");
	GL_Color dgrey = {0.3, 0.3, 0.3, 1};
	GL_Color black = {0, 0, 0, 1};
	GL_Color white = {1, 1, 1, 1};
	GL_Color red = {1, 0, 0, 1};
	GLpoint a = {0,0};
	GLpoint a1 = {1,1};
	GLpoint a2 = {w/4+1,h/2};
	GLpoint b = {w/2,0};
	GLpoint b1 = {w/2+1,1};
	GLpoint b2 = {5*w/8,h/6};
	GLpoint b3 = {5*w/8+2,h/6};
	this->ob = new GLObject(x+1,y+1);
	// first cell : ?
	this->ob->addRect(a,w/2-3, h-2, dgrey);
	this->ob->addBox(a1,w/2-3, h-3, black);
	this->ob->addTextureCenter(a2,  t, red);
	

	// second Cell : Flag
	this->ob->addRect(b,w/2-2, h-2, dgrey);
	this->ob->addBox(b1,w/2-3, h-3, black);
	this->ob->addRect(b2,2, 4*h/6, white);
	this->ob->addRect(b3,w/4-2, 2*h/6, red);

	this->cur = new cursor(x,y,w/2,h);
	s->addObject("DigObj", this->ob);
	s->addObject("DigCur", this->cur);
}

void DigSwitch::setDig(bool d) {
	this->dig = d;
	this->cur->moveTo(this->x+(d?0:this->w/2), this->y);
}

void DigSwitch::flipSwitch() {
	this->setDig(!this->dig);
}

bool DigSwitch::shouldDig() {
	return this->dig;
}



DifficultySwitch::DifficultySwitch(GLScreen *s, GLfloat x, GLfloat y, GLfloat w, GLfloat h):x(x),y(y),w(w),h(h),diff(2) {
	Texture* t1 = s->getTexture("M1");
	Texture* t2 = s->getTexture("M2");
	Texture* t3 = s->getTexture("M3");
	GL_Color dgrey = {0.3, 0.3, 0.3, 1};
	GL_Color black = {0, 0, 0, 1};
	GL_Color col1 = {0, 1, 0, 1};
	GL_Color col2 = {1, 1, 0, 1};
	GL_Color col3 = {1, 0, 0, 1};
	GLpoint a = {0,0};
	GLpoint a1 = {1,1};
	GLpoint a2 = {w/2,1+h/6};
	GLpoint b = {0,h/3};
	GLpoint b1 = {1,1+h/3};
	GLpoint b2 = {w/2,1+h/2};
	GLpoint c = {0,2*h/3};
	GLpoint c1 = {1,1+2*h/3};
	GLpoint c2 = {w/2,1+5*h/6};
	this->ob = new GLObject(x+1,y+1);
	this->ob->addRect(a,w-3, h/3-2, dgrey);
	this->ob->addBox(a1,w-3, h/3-3, black);
	this->ob->addTextureCenter(a2,  t1, col1);
	this->ob->addRect(b,w-3, h/3-2, dgrey);
	this->ob->addBox(b1,w-3, h/3-3, black);
	this->ob->addTextureCenter(b2,  t2, col2);
	this->ob->addRect(c,w-3, h/3-2, dgrey);
	this->ob->addBox(c1,w-3, h/3-3, black);
	this->ob->addTextureCenter(c2,  t3, col3);
	this->cur = new cursor(x,y+h/3,w,h/3);
	s->addObject("DiffObj", this->ob);
	s->addObject("DiffCur", this->cur);
}
void DifficultySwitch::setDifficulty(int d) {
	this->diff=(d<1?1:(d>3?3:d));
	this->cur->moveTo(x,y+(this->diff-1)*h/3);
}
int DifficultySwitch::getDifficulty() {
	return this->diff;
}

void DifficultySwitch::clicked(GLfloat x, GLfloat y) {
	if (x>this->x&&x<this->x+this->w) {
		if (y>this->y&&y<this->y+this->h/3) {
			this->setDifficulty(1);
		} else if (y>this->y+this->h/3&&y<this->y+2*this->h/3) {
			this->setDifficulty(2);
		} else if (y>this->y+this->h/3&&y<this->y+this->h) {
			this->setDifficulty(3);
		} 
	}
}
int  DifficultySwitch::getMineCount() {
	switch(this->diff) {
	case 1:
		return DIFICULTY_1;
	break;
	case 2:
		return DIFICULTY_2;
	break;
	case 3:
		return DIFICULTY_3;
	break;
	}
}



dialog::dialog(GLScreen *s) :GLObject(DIALOG_X, DIALOG_Y), s(s), w(DIALOG_W), h(DIALOG_H) {
	
}
void dialog::hide() {
	this->clear();
}
void dialog::showIsWin(bool win) {
	GL_Color c1 = {0.3,0.3,0.3,1};
	GL_Color c2 = {0,0,0,1};
	GL_Color txt = {1,1,1,1};
	GLpoint  a  = {0,0,0}; 
	GLpoint  c  = {4*(this->w/5+1), this->h/2+2, 0};
	this->addButton(a, this->w, this->h, c1, c2);
	if (win)
		this->addTextureCenter(c, this->s->getTexture("WIN"), txt);
	else
		this->addTextureCenter(c, this->s->getTexture("LOSE"), txt);
}



Grid::Grid(GLScreen *s) : s(s), finished(false), keypress(0) {
	stringstream ss;
	GLObject *o;
	Texture  *t;
	GLpoint p = {0, 0, 0};
	GL_Color c = {1, 1, 1, 1};
	GL_Color b = {0.1, 0.5, 1, 1};
	GL_Color b2 = {0, 0.2, 0.8, 1};

	// Create the left bar
	o = new GLObject(0, 0);
	o->addButton(p,GRID_LEFT,SCREEN_HEIGHT, b2, b);
	this->s->addObject("BG", o);

	t = this->s->getTexture("time");
	o = new GLObject(GRID_LEFT/25, SCREEN_HEIGHT/15-t->h);
	o->addTexture(p,t,c);
	this->s->addObject("timerLabel", o);

	t = this->s->getTexture("highsc");
	o = new GLObject(GRID_LEFT/25, 3*SCREEN_HEIGHT/15-t->h);
	o->addTexture(p,t,c);
	this->s->addObject("highscLabel", o);

	t = this->s->getTexture("ml");
	o = new GLObject(GRID_LEFT/25, 5*SCREEN_HEIGHT/15-t->h);
	o->addTexture(p,t,c);
	this->s->addObject("minesLeft", o);

	t = this->s->getTexture("diff");
	o = new GLObject(GRID_LEFT/25, 11*SCREEN_HEIGHT/15-t->h);
	o->addTexture90(p,t,c);
	this->s->addObject("difficultyLabel", o);

	this->ds = new DigSwitch(this->s, DS_X, DS_Y, DS_W, DS_H);
 	this->fs = new DifficultySwitch(this->s, FS_X, FS_Y, FS_W, FS_H);
	this->minesCnt = new intDisplay(this->s, this->fs->getMineCount());
	this->timer = new timeDisplay(this->s);
	this->highsc = new highScore(this->s);
	this->highsc->loadFrom();
	this->face = new faceButton(FACE_X, FACE_Y, FACE_SIZE, FACE_SIZE);
	this->timer->moveTo(GRID_LEFT/25, SCREEN_HEIGHT/15);
	this->highsc->moveTo(GRID_LEFT/25, 3*SCREEN_HEIGHT/15);
	this->minesCnt->moveTo(GRID_LEFT/25, 5*SCREEN_HEIGHT/15);
	this->s->addObject("timer", this->timer);
	this->s->addObject("highscore", this->highsc);
	this->s->addObject("minesCnt", this->minesCnt);
	this->s->addObject("face", this->face);

	// Create the grid
	this->g = (Cell**)malloc(sizeof(Cell*)*GRID_HEIGHT*GRID_WIDTH);
	for(int i=0;i<GRID_HEIGHT;i++) {
		for(int j=0;j<GRID_WIDTH;j++) {
			ss.flush();
			ss << "c" << j << "x" << i;
			this->g[i*GRID_WIDTH+j] = new Cell(this, j, i);
			this->s->addObject(ss.str(), this->g[i*GRID_WIDTH+j]);
		}
	}
	this->cur = new gridCursor((int)floor(GRID_WIDTH/2), (int)floor(GRID_HEIGHT/2));
	this->s->addObject("gridCursor", this->cur);

	this->dia = new dialog(this->s);
	this->s->addObject("dialog", this->dia);
	
}

Grid::~Grid() {
	this->highsc->saveTo();
	for(int i=0;i<GRID_HEIGHT;i++) {
		for(int j=0;j<GRID_WIDTH;j++) {
			delete this->g[i*GRID_WIDTH+j];
		}
	}
	delete this->timer;
	delete this->dia;
	delete this->highsc;
	delete this->ds;
	delete this->fs;
	delete this->face;
	delete this->cur;
	/*free(this->g);*/
}

Cell *Grid::getCell(int x, int y) {
	if ( x<0 || y<0 || x>GRID_WIDTH || y>GRID_HEIGHT)
		return NULL;
	return this->g[y*GRID_WIDTH+x];
}

void Grid::reNew() {
	this->finished  = false;
	this->ds->setDig(true);
	this->gen(this->fs->getMineCount());
	//this->draw();
	this->timer->reset();
	this->face->setHappy(true);
 	this->minesCnt->setValue(this->fs->getMineCount());
	this->dia->hide();
}



// Active part of the EventManager
void Grid::clickCell(bool flag, int x, int y) {
	bool tmp = this->getCell(x,y)->isFlaged();
	bool res = true;

	if (this->finished)
		return;
	if (!flag && !this->timer->running()) {
		// force first click to be safe
		while (this->getCell(x,y)->mined())
			this->gen(this->fs->getMineCount());
	}
	if (!this->timer->running())
		this->timer->start();

	this->getCell(x,y)->clicked(flag);
	if (flag) {
		if (!this->getCell(x,y)->isFlaged())
			this->minesCnt->setValue(this->minesCnt->getValue()+1);
		else
			this->minesCnt->setValue(this->minesCnt->getValue()-1);
		if (this->minesCnt->getValue()==0) {
			// check if all the mines are found
			for(int i=0;i<GRID_HEIGHT && res;i++) {
				for(int j=0;j<GRID_WIDTH && res;j++) {
					if (this->getCell(j, i)->mined())
						res=this->getCell(j, i)->isFlaged();
				}
			}
			if (res)
				this->endGame(true);
		}
	} else if (!flag && !this->getCell(x,y)->isFlaged()) {
		// Check if all the discoverable cell are shown
		for(int i=0;i<GRID_HEIGHT && res;i++) {
			for(int j=0;j<GRID_WIDTH && res;j++) {
				if (!this->getCell(j, i)->isShown())
					res=this->getCell(j, i)->mined();
			}
		}
		if (res)
			this->endGame(true);
	}
}


void Grid::eventManager(SDL_Event *e) {
	Uint32 now = SDL_GetTicks();
	int x,y;
	switch(e->type) {
#ifdef CAANOO
	case SDL_JOYAXISMOTION:
		printf("SDL_JOYAXISMOTION axis(%d)=%d", e->jaxis.axis, e->jaxis.value);
		// up side
		if ((( e->jaxis.value < -11585 ) && (e->jaxis.axis == 1)) && ((this->keypress &1) != 1)) {
			this->cur->moveUp();
			this->keypress |= 1;
			this->keyStartTicks  = now;
			printf(" +UP");
		}
		if ((( e->jaxis.value >  11585 ) && (e->jaxis.axis == 1)) && ((this->keypress &2) != 2)) {
			this->cur->moveDown();
			this->keypress |= 2;
			this->keyStartTicks  = now;
			printf(" +DOWN");
		}
		if ((( e->jaxis.value < -11585 ) && (e->jaxis.axis == 0)) && ((this->keypress &4) != 4)) {
			this->cur->moveLeft();
			this->keypress |= 4;
			this->keyStartTicks  = now;
			printf(" +LEFT");
		}
		if ((( e->jaxis.value >  11585 ) && (e->jaxis.axis == 0)) && ((this->keypress &8) != 8)) {
			this->cur->moveRight();
			this->keypress |= 8;
			this->keyStartTicks  = now;
			printf(" +RIGHT");
		}
		// down side
		if ((( e->jaxis.value >= -11585 ) && (e->jaxis.axis == 1)) && ((this->keypress &1) == 1)) {
			this->keypress -= 1;
			printf(" -UP");
		}
		if ((( e->jaxis.value <=  11585 ) && (e->jaxis.axis == 1)) && ((this->keypress &2) == 2)) {
			this->keypress -= 2;
			printf(" -DOWN");
		}
		if ((( e->jaxis.value >= -11585 ) && (e->jaxis.axis == 0)) && ((this->keypress &4) == 4)) {
			this->keypress -= 4;
			printf(" -LEFT");
		}
		if ((( e->jaxis.value <=  11585 ) && (e->jaxis.axis == 0)) && ((this->keypress &8) == 8)) {
			this->keypress -= 8;
			printf(" -RIGHT");
		}
		printf("\tkey=%d\n", this->keypress);
	break;
#endif
	case SDL_KEYDOWN:
		if (this->finished)	return;
		switch( e->key.keysym.sym ){
		case SDLK_UP:
			this->cur->moveUp();
			this->keypress |= 1;
			this->keyStartTicks  = now;
		break;
		case SDLK_DOWN:
			this->cur->moveDown();
			this->keypress |= 2;
			this->keyStartTicks  = now;
		break;
		case SDLK_LEFT:
			this->cur->moveLeft();
			this->keypress |= 4;
			this->keyStartTicks  = now;
		break;
		case SDLK_RIGHT:
			this->cur->moveRight();
			this->keypress |= 8;
			this->keyStartTicks  = now;
		break;
		}
	break;
	case SDL_KEYUP:
		switch( e->key.keysym.sym ){
		case SDLK_UP:
			if ((this->keypress &1) == 1)
				this->keypress -= 1;
		break;
		case SDLK_DOWN:
			if ((this->keypress &2) == 2)
				this->keypress -= 2;
		break;
		case SDLK_LEFT:
			if ((this->keypress &4) == 4)
				this->keypress -= 4;
		break;
		case SDLK_RIGHT:
			if ((this->keypress &8) == 8)
				this->keypress -= 8;
		break;
		case SDLK_SPACE:
		case SDLK_RETURN:
#ifdef PANDORA
		case SDLK_END:      // (B)
		case SDLK_PAGEUP:   // (Y)
#endif
			if (!this->finished)
				this->clickCell(false, this->cur->getPosX(),this->cur->getPosY());
			else
				this->reNew();
		break;
		case SDLK_f:
#ifdef PANDORA
		case SDLK_HOME:     // (A)
		case SDLK_PAGEDOWN: // (X)
#endif
			if (!this->finished)
				this->clickCell(true, this->cur->getPosX(),this->cur->getPosY());
			else
				this->reNew();
		break;
		case SDLK_r:
#ifdef PANDORA
		case SDLK_LALT: // (Start)
#endif
			this->reNew();
		break;
		}
	break;
#ifdef CAANOO
	case SDL_JOYBUTTONUP:
		switch(e->jbutton.button) {
		case 2: // (B)
		case 3: // (Y)
			if (!this->finished)
				this->clickCell(false, this->cur->getPosX(),this->cur->getPosY());
			else
				this->reNew();
		break;
		case 0: // (A)
		case 1: // (X)
			if (!this->finished)
				this->clickCell(true, this->cur->getPosX(),this->cur->getPosY());
			else
				this->reNew();
		break;
		case 8: // (I)
			this->reNew();
		break;
		}
	break;
#endif
	case SDL_MOUSEBUTTONDOWN:
		this->mousePress	= true;
		this->keyStartTicks	= now;
	break;
	case SDL_MOUSEBUTTONUP:
		x = e->button.x;
		y = e->button.y;
		if ((this->mousePress && now-this->keyStartTicks>800 && x>GRID_LEFT && y>GRID_TOP && !this->finished) ||  // Grid clicked for more than 0.8 sec
		    (e->button.button==SDL_BUTTON_RIGHT && x>GRID_LEFT && y>GRID_TOP && !this->finished)) {
			this->clickCell(true, (int)floor((x-GRID_LEFT)/(CELL_WIDTH)), (int)floor((y-GRID_TOP)/(CELL_HEIGHT)));		
		}else if (x>GRID_LEFT && y>GRID_TOP && !this->finished) {
			this->clickCell(!this->ds->shouldDig(), (int)floor((x-GRID_LEFT)/(CELL_WIDTH)), (int)floor((y-GRID_TOP)/(CELL_HEIGHT)));
		} else if (x>DS_X && x<DS_X+DS_W && y>DS_Y && y<DS_Y+DS_H)
			this->ds->flipSwitch();
		  else if (x>FS_X && x<FS_X+FS_W && y>FS_Y && y<FS_Y+FS_H) {
			this->fs->clicked(x,y);
			this->highsc->setLevel(this->fs->getDifficulty());
			this->reNew();
		} else if (x>FACE_X && x<FACE_X+FACE_SIZE && y>FACE_Y && y<FACE_Y+FACE_SIZE)
			this->reNew();
	
		this->mousePress	= false;
 	break;
	}

}

void Grid::updateInLoop() {
	Uint32 now = SDL_GetTicks();
	if (this->keypress > 0 && now - this->keyStartTicks > 100) {
		if ((this->keypress &1) == 1)
			this->cur->moveUp();
		if ((this->keypress &2) == 2)
			this->cur->moveDown();
		if ((this->keypress &4) == 4)
			this->cur->moveLeft();
		if ((this->keypress &8) == 8) 
			this->cur->moveRight();
		this->keyStartTicks = now;
	}

	this->timer->update();
}


void Grid::reset() {
	for(int i=0;i<GRID_HEIGHT;i++) {
		for(int j=0;j<GRID_WIDTH;j++) {
			this->getCell(j, i)->reset();
		}
	}
}

void Grid::gen(int maxMines) {
	// set the mines
	this->reset();
	srand(time(NULL));
	for(int m = maxMines; m>0; m--) {
		bool s = false;
		do {
			int x = rand() % GRID_WIDTH;
			int y = rand() % GRID_HEIGHT;

			s = this->getCell(x, y)->mined();
			if ( ! s ) { // start mine is protected
				this->getCell(x, y)->setMined(true);
			}
		}
		while( s );
	}

	// build the face values
	for(int i=0;i<GRID_HEIGHT;i++) {
		for(int j=0;j<GRID_WIDTH;j++) {
			this->getCell(j,i)->setValue();
		}
	}
	this->minesCnt->setValue(maxMines);
}

void Grid::endGame(bool win) {
	this->finished = true;
	this->timer->stop();
	this->face->setHappy(win);
	this->dia->showIsWin(win);
	if (win && this->timer->getValue() < this->highsc->getValue())
		this->highsc->setValue(this->timer->getValue());
}
