/***********************************************************************
    filename:   CEGUIOpenGLESRenderTarget.cpp
    created:    Wed Jan 14 2009
    author:     Paul D Turner
*************************************************************************/
/***************************************************************************
 *   Copyright (C) 2004 - 2009 Paul D Turner & The CEGUI Development Team
 *
 *   Permission is hereby granted, free of charge, to any person obtaining
 *   a copy of this software and associated documentation files (the
 *   "Software"), to deal in the Software without restriction, including
 *   without limitation the rights to use, copy, modify, merge, publish,
 *   distribute, sublicense, and/or sell copies of the Software, and to
 *   permit persons to whom the Software is furnished to do so, subject to
 *   the following conditions:
 *
 *   The above copyright notice and this permission notice shall be
 *   included in all copies or substantial portions of the Software.
 *
 *   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 *   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 *   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 *   OTHER DEALINGS IN THE SOFTWARE.
 ***************************************************************************/
#include "CEGUIOpenGLESRenderTarget.h"
#include "CEGUIRenderQueue.h"
#include "CEGUIOpenGLESGeometryBuffer.h"
#include "CEGUIOpenGLES.h"
#include <cmath>

#include <GLES/glues.h>

// Start of CEGUI namespace section
namespace CEGUI
{
//----------------------------------------------------------------------------//
const float OpenGLESRenderTarget::d_yfov_tan = 0.267949192431123;

//----------------------------------------------------------------------------//
OpenGLESRenderTarget::OpenGLESRenderTarget(OpenGLESRenderer& owner) :
    d_owner(owner),
    d_area(0, 0, 0, 0),
    d_matrixValid(false)
{
}

//----------------------------------------------------------------------------//
void OpenGLESRenderTarget::draw(const GeometryBuffer& buffer)
{
    buffer.draw();
}

//----------------------------------------------------------------------------//
void OpenGLESRenderTarget::draw(const RenderQueue& queue)
{
    queue.draw();
}

//----------------------------------------------------------------------------//
void OpenGLESRenderTarget::setArea(const Rect& area)
{
    d_area = area;
    d_matrixValid = false;
}

//----------------------------------------------------------------------------//
const Rect& OpenGLESRenderTarget::getArea() const
{
    return d_area;
}

//----------------------------------------------------------------------------//
void OpenGLESRenderTarget::activate()
{
	glViewport(static_cast<GLsizei>(d_area.d_left),
               static_cast<GLsizei>(d_area.d_top),
               static_cast<GLsizei>(d_area.getWidth()),
               static_cast<GLsizei>(d_area.getHeight()));

    if (!d_matrixValid)
        updateMatrix();

    glMatrixMode(GL_PROJECTION);
    glLoadMatrixf(d_matrix);
}

//----------------------------------------------------------------------------//
void OpenGLESRenderTarget::deactivate()
{
}

//----------------------------------------------------------------------------//
void OpenGLESRenderTarget::unprojectPoint(const GeometryBuffer& buff,
    const Vector2& p_in, Vector2& p_out) const
{
    if (!d_matrixValid)
        updateMatrix();

    const OpenGLESGeometryBuffer& gb =
        static_cast<const OpenGLESGeometryBuffer&>(buff);

    const GLint vp[4] = {
        static_cast<GLint>(d_area.d_left),
        static_cast<GLint>(d_area.d_top),
        static_cast<GLint>(d_area.getWidth()),
        static_cast<GLint>(d_area.getHeight())
    };

    GLfloat in_x, in_y, in_z = 0.0;

    // unproject the ends of the ray
    GLfloat r1_x, r1_y, r1_z;
    GLfloat r2_x, r2_y, r2_z;
    in_x = vp[2] * 0.5;
    in_y = vp[3] * 0.5;
    in_z = -d_viewDistance;

	float gb_matrixd[16], d_matrixd[16];
	for (uint i = 0; i < 16; i++)
	{
		gb_matrixd[i] = (float)gb.getMatrix()[i];
		d_matrixd[i] = (float)d_matrixd[i];
	}

    gluUnProject(in_x, in_y, in_z, gb_matrixd, d_matrixd, vp,
                 &r1_x, &r1_y, &r1_z);
    in_x = p_in.d_x;
    in_y = vp[3] - p_in.d_y;
    in_z = 0.0;
    gluUnProject(in_x, in_y, in_z, gb_matrixd, d_matrixd, vp,
                 &r2_x, &r2_y, &r2_z);

    // project points to orientate them with GeometryBuffer plane
    GLfloat p1_x, p1_y, p1_z;
    GLfloat p2_x, p2_y, p2_z;
    GLfloat p3_x, p3_y, p3_z;
    in_x = 0.0;
    in_y = 0.0;
    gluProject(in_x, in_y, in_z, gb_matrixd, d_matrixd, vp,
               &p1_x, &p1_y, &p1_z);
    in_x = 1.0;
    in_y = 0.0;
    gluProject(in_x, in_y, in_z, gb_matrixd, d_matrixd, vp,
               &p2_x, &p2_y, &p2_z);
    in_x = 0.0;
    in_y = 1.0;
    gluProject(in_x, in_y, in_z, gb_matrixd, d_matrixd, vp,
               &p3_x, &p3_y, &p3_z);

    // calculate vectors for generating the plane
    const float pv1_x = p2_x - p1_x;
    const float pv1_y = p2_y - p1_y;
    const float pv1_z = p2_z - p1_z;
    const float pv2_x = p3_x - p1_x;
    const float pv2_y = p3_y - p1_y;
    const float pv2_z = p3_z - p1_z;
    // given the vectors, calculate the plane normal
    const float pn_x = pv1_y * pv2_z - pv1_z * pv2_y;
    const float pn_y = pv1_z * pv2_x - pv1_x * pv2_z;
    const float pn_z = pv1_x * pv2_y - pv1_y * pv2_x;
    // calculate plane
    const float pn_len = std::sqrt(pn_x * pn_x + pn_y * pn_y + pn_z * pn_z);
    const float pl_a = pn_x / pn_len;
    const float pl_b = pn_y / pn_len;
    const float pl_c = pn_z / pn_len;
    const float pl_d = -(p1_x * pl_a + p1_y * pl_b + p1_z * pl_c);
    // calculate vector of picking ray
    const float rv_x = r1_x - r2_x;
    const float rv_y = r1_y - r2_y;
    const float rv_z = r1_z - r2_z;
    // calculate intersection of ray and plane
    const float pn_dot_r1 = (r1_x * pn_x + r1_y * pn_y + r1_z * pn_z);
    const float pn_dot_rv = (rv_x * pn_x + rv_y * pn_y + rv_z * pn_z);
    const float tmp1 = pn_dot_rv != 0.0 ? (pn_dot_r1 + pl_d) / pn_dot_rv : 0.0;
    const float is_x = r1_x - rv_x * tmp1;
    const float is_y = r1_y - rv_y * tmp1;

    p_out.d_x = static_cast<float>(is_x);
    p_out.d_y = static_cast<float>(is_y);
}

//----------------------------------------------------------------------------//
void OpenGLESRenderTarget::updateMatrix() const
{
    const float w = d_area.getWidth();
    const float h = d_area.getHeight();
    const float aspect = w / h;
    const float midx = w * 0.5;
    const float midy = h * 0.5;
    d_viewDistance = midx / (aspect * d_yfov_tan);

    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    gluPerspective(30.0, aspect, d_viewDistance * 0.5, d_viewDistance * 2.0);
    // Projection matrix abuse!
    gluLookAt(midx, midy, -d_viewDistance, midx, midy, 1, 0, -1, 0);
	glGetFloatv(GL_PROJECTION_MATRIX, d_matrix);
    glTranslatef(0.375, 0.375, 0);
    glPopMatrix();

    d_matrixValid = true;
}

//----------------------------------------------------------------------------//

} // End of  CEGUI namespace section
